/********** L3Processor.C *******************************************************************************************\

 PURPOSE
   Main driver for L3 Daily processor.

 ASSUMPTIONS ON ENTRY
   FILES
     The PGE PCF has staged all required files.

 ASSUMPTIONS ON EXIT
   FILES
     MOP03 and MOP3Q files have been produced for this processing data.
   ERROR HANDLING
     Errors detected during PGE run have been written to the LogStatus file.
   MISCELLANEOUS
     PGE information has been written to the LogReport file.

 REVISION HISTORY
   07/12   Debbie Mao -- CF HDF-EOS5 for V6 & Augmentation
   07/10   Debbie Mao -- for V5 new release
   11/12   Debbie Mao -- V6 in HDF-EOS5 format
   07/13   Debbie Mao -- V6 in HDF-EOS5 format, filter out noisy pixel: TIR & JNT-keep Pix1&2; NIR-keep all pixels
   07/16   Test new filtering method for V7 -- DM

\*********************************************************************************************************************/
#include "Driver.h" 
#include "MOP02.h" 
#include "Grid.h"
#include "MOP03File.h" 
#include "DiagnosticReporter.h"


// make the diagnostic reporter globally accessible
diagnostic_reporter diagnosticreporter;

using namespace std;

int main ()
{

  MOP02_Geo  M2geo;
  MOP02_Dat1 M2dat1;
  MOP02_Dat2 M2dat2;
  MOP02_Num  M2num;
  MOP03_Num  M3num;
  Product    Prod;  

  double starttime, stoptime;
  float  psurfmaj[DAY][XDim][YDim]; 
  bool   isread = true;
  int    rmPix;

  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 900,"Welcome to MOPITT V6 daily Level 3 Processor.");

  // get to be removed pixel number
  isread = Get_rmPixel(rmPix);
  if ( isread ) {
    cout << "removing pixel info:  " << rmPix << endl;
    diagnosticreporter.Write(DIAGNOSTICS_WARNING, 901,"After read rePix info.");
  }
  else {
    diagnosticreporter.Write(DIAGNOSTICS_ERROR, 902, "Failed at reading rmPix switch. STOP the processor!");
  }

  // get the Level 2 information from MOP02.hdf file
  isread = MOP02_Read_Data(starttime, stoptime, M2geo, M2dat1, M2dat2);
  if ( isread ) {
    Prod.Fstarttime = starttime;
    Prod.Fstoptime = stoptime;
    MOP02_Get_Num( M2geo, M2num );
    diagnosticreporter.Write(DIAGNOSTICS_WARNING, 910,"After read MOP02 info.");
  }
  else {
    diagnosticreporter.Write(DIAGNOSTICS_ERROR, 912, "Failed at rading M2_All. STOP the processor!");
  }

  // use the geo info to set the majority surface pressure and majority surface index for each grid
  GetMaj_PsurfInd(M2num, psurfmaj, Prod.isurf);
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 920,"After GetMaj_PsurfInd.");

  // average valid pixels into the grid 
  PixeltoGrid (rmPix, psurfmaj, M2geo, M2dat1, M2dat2, M3num, Prod);
  cout << " Pix to Grid -- no: " <<  M3num.nco_totmp[0][0][30][30] <<"  "<< M3num.nco_totmp[0][0][40][40] << endl;
  cout << " Pix to Grid: " << Prod.co_colm[0][0][30][30] <<"  "<< Prod.co_colm[0][0][40][40] << endl;
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 730, "After PixeltoGrid");

  Avg_Grid ( M3num, Prod );
  cout << " Avg to Grid: " << Prod.co_colm[0][0][30][30] <<"  "<< Prod.co_colm[0][0][40][40] << endl;
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 740, "After Avg_Grid");

  // get info for std dev
  Stddev_Grid (rmPix, M2geo, M2dat1, M2dat2, psurfmaj, Prod);
  cout << " stddev to Grid: " << Prod.co_colm[0][2][30][30] <<"  "<< Prod.co_colm[0][2][40][40] << endl;
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 744, "After Stddev_Grid");

  Var_Grid (M3num, Prod);
  cout << " var to Grid: " << Prod.co_colm[0][2][30][30] <<"  "<< Prod.co_colm[0][2][40][40] << endl;
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 746, "After Var_Grid");

  //initialize MOP03 file
  mop03_file mop03file;
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 950, "After mop03file initialize");

  //write info in to MOP03 file 
  mop03file.Write( Prod );
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 960, "After mop03_write");

  // write mop03 metadata file and close mop03 file
  mop03file.Close (Prod.Fstarttime,Prod.Fstoptime);
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 970, "After mop03_close");

  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 980, "Normal Termination");
  return 0;
}
