/********** MOPITTTime.C **********************************************************************************************\

 $Header$

 REVISION HISTORY
   12/97   Charles Cavanaugh
   09/99   Charles Cavanaugh
   12/99   Charles Cavanaugh
   10/02   Debbie Mao

 $Log$

\**********************************************************************************************************************/

#include "PGS_SMF.h"
#include "PGS_TD.h"
#include "MOPITTTime.h"
#include "DiagnosticReporter.h"

extern diagnostic_reporter diagnosticreporter;

int const mopitt_time::NCAR_UTC_ERROR   = 900;
int const mopitt_time::NCAR_UTC_WARNING = 800;
int const mopitt_time::TAI_UTC_ERROR    = 901;
int const mopitt_time::TAI_UTC_WARNING  = 801;
int const mopitt_time::UTC_NCAR_ERROR   = 902;
int const mopitt_time::UTC_TAI_ERROR    = 903;
int const mopitt_time::UTC_TAI_WARNING  = 803;

int const UTC_SIZE = 28;

mopitt_time :: mopitt_time ()
             : utc ()
{
  day   = 0;
  msecs = 0;
  tai   = 0.0;
}


mopitt_time :: mopitt_time (int dayvalue, int msecsvalue)
{
  Set (dayvalue, msecsvalue);
}


mopitt_time :: mopitt_time (double taitime)
{
  Set (taitime);
}


mopitt_time :: mopitt_time (string const& utctime)
{
  Set (utctime);
}


mopitt_time :: ~mopitt_time ()
{
  ;
}


bool mopitt_time :: MakeNCARFromUTC ()
{
  bool ismade = true;

  // convert utc to spacecraft time
  PGSt_scTime sctime [8];
  PGSt_SMF_status smfstatus = PGS_TD_UTC_to_SCtime (PGSd_EOS_AM, (char*) utc.c_str (), sctime);

  // warn if leap seconds error; all other non-successes warrant an error
  if (smfstatus == PGSTD_E_NO_LEAP_SECS) {
    string message = "No leap seconds correction available for ";
    message += utc.c_str ();
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_TIME_MODULE, NCAR_UTC_WARNING, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }
  else if (smfstatus != PGS_S_SUCCESS) {
    ismade = false;
    string message = "Could not convert UTC time ";
    message += utc.c_str ();
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_TIME_MODULE, NCAR_UTC_ERROR, 0, 0, 0, 0, 0, 0, 0, 0, 
                                 message);
  }

  // convert spacecraft time to day/time
  day = (sctime [0] * 256) + (int) sctime [1];
  msecs = (sctime [2] * 16777216) + (sctime [3] * 65536) + (sctime [4] * 256) + (int) sctime [5];
  sctime [6] = sctime [7] = 0;

  return ismade;
}


bool mopitt_time :: MakeTAIFromUTC ()
{
  bool ismade = true;

  // convert utc to tai
  PGSt_SMF_status smfstatus = PGS_TD_UTCtoTAI ((char*) utc.c_str (), &tai);

  // warn if leap seconds error; all other non-successes warrant an error
  if (smfstatus == PGSTD_E_NO_LEAP_SECS) {
    string message = "No leap seconds correction available for ";
    message += utc.c_str ();
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_TIME_MODULE, TAI_UTC_WARNING, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }
  else if (smfstatus != PGS_S_SUCCESS) {
    ismade = false;
    string message = "Could not convert UTC time ";
    message += utc.c_str ();
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_TIME_MODULE, TAI_UTC_ERROR, 0, 0, 0, 0, 0, 0, 0, 0, 
                                 message);
  }

  return ismade;
}


bool mopitt_time :: MakeUTCFromNCAR ()
{
  bool ismade = true;

  // make spacecraft time from the day/msecs
  PGSt_scTime sctime [8];
  sctime [0] = (PGSt_scTime) (day / 256);
  sctime [1] = (PGSt_scTime) (day % 256);
  sctime [2] = (PGSt_scTime) (msecs / 16777216);
  sctime [3] = (PGSt_scTime) ((msecs % 16777216) / 65536);
  sctime [4] = (PGSt_scTime) ((msecs % 65536) / 256);
  sctime [5] = (PGSt_scTime) (msecs % 256);
  sctime [6] = sctime [7] = 0;

  // convert spacecraft time to utc; any non-success warrants an error
  char utctime [UTC_SIZE];
  PGSt_double offsets;
  if (PGS_TD_SCtime_to_UTC (PGSd_EOS_AM, &sctime, 1, utctime, &offsets) != PGS_S_SUCCESS) {
    ismade = false;
    char timevalues [50];
    (void) sprintf (timevalues, "%hx %hx %hx %hx %hx %hx %hx %hx", sctime [0], sctime [1], sctime [2], sctime [3],
                    sctime [4], sctime [5], sctime [6], sctime [7]);
    string message = "Could not convert SC time values ";
    message += timevalues;
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_TIME_MODULE, UTC_NCAR_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }

  // make sure the utc field has a value (even a wrong one)
  utc = utctime;
  
  return ismade;
}


bool mopitt_time :: MakeUTCFromTAI ()
{
  bool ismade = true;

  // convert tai to utc
  char utctime [UTC_SIZE];
  PGSt_SMF_status smfstatus = PGS_TD_TAItoUTC (tai, utctime);

  // warn if leap seconds error; all other non-successes warrant an error
  if (smfstatus == PGSTD_E_NO_LEAP_SECS) {
    char taitime [25];
    (void) sprintf (taitime, "%20.10lf", tai);
    string message = "No leap seconds correction available for ";
    message += taitime;
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_TIME_MODULE, UTC_TAI_WARNING, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }
  else if (smfstatus != PGS_S_SUCCESS) {
    ismade = false;
    char taitime [25];
    (void) sprintf (taitime, "%20.10lf", tai);
    string message = "Could not convert TAI time ";
    message += taitime;
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_TIME_MODULE, UTC_TAI_ERROR, 0, 0, 0, 0, 0, 0, 0, 0, 
                                 message);
  }

  // make sure the utc field has a value (even a wrong one)
  utc = utctime;

  return ismade;
}


mopitt_time& mopitt_time :: operator = (mopitt_time const& intime)
{
  // check for assignment to self; if not self, then grab day/msecs and build time
  if (this != &intime)
    Set (intime.GetDay (), intime.GetMsecs ());

  // return reference to self
  return *this;
}


bool mopitt_time :: operator == (mopitt_time const& rhs) const
{
  return (day == rhs.day && msecs == rhs.msecs);
}


bool mopitt_time :: operator != (mopitt_time const& rhs) const
{
  return (day != rhs.day || msecs != rhs.msecs);
}


bool mopitt_time :: operator > (mopitt_time const& rhs) const
{
  return ((day > rhs.day) || (day == rhs.day && msecs > rhs.msecs));
}


bool mopitt_time :: operator >= (mopitt_time const& rhs) const
{
  return ((day > rhs.day) || (day == rhs.day && msecs >= rhs.msecs));
}


bool mopitt_time :: operator < (mopitt_time const& rhs) const
{
  return ((day < rhs.day) || (day == rhs.day && msecs < rhs.msecs));
}


bool mopitt_time :: operator <= (mopitt_time const& rhs) const
{
  return ((day < rhs.day) || (day == rhs.day && msecs <= rhs.msecs));
}


bool mopitt_time :: Set (int dayvalue, int msecsvalue)
{
  bool isset = false;

  // store the day and milliseconds and compute the tai and utc times
  day = dayvalue;
  msecs = msecsvalue;
  if (MakeUTCFromNCAR ())
    if (MakeTAIFromUTC ())
      isset = true;

  return isset;
}


bool mopitt_time :: Set (double taitime)
{
  bool isset = false;

  // store the tai time and compute the ncar and utc times
  tai = taitime;
  if (MakeUTCFromTAI ())
    if (MakeNCARFromUTC ())
      isset = true;

  return isset;
}


bool mopitt_time :: Set (string const& utctime)
{
  bool isset = false;

  // store the utc time and compute the ncar and tai times
  utc = utctime;
  if (MakeTAIFromUTC ())
    if (MakeNCARFromUTC ())
      isset = true;

  return isset;
}
