/********** SwathRadiances.C ******************************************************************************************\

 $Header$

 REVISION HISTORY
   05/99   Charles Cavanaugh
   11/99   Charles Cavanaugh
   06/00   Charles Cavanaugh
   10/01   Debbie Mao
   01/02   Debbie Mao
   02/02   Debbie Mao
   03/02   Debbie Mao  
   04/02   Debbie Mao
   10/09   Debbie Mao

 $Log$

\**********************************************************************************************************************/

#include <memory.h>
#include <math.h>
#include "SwathRadiances.h"
#include "DiagnosticReporter.h"

extern diagnostic_reporter diagnosticreporter;

int const swath_radiances::RADIANCE_DATASET_DIMENSIONS    = 5;
int const swath_radiances::CALIBRATION_DATASET_DIMENSIONS = 5;
int const swath_radiances::CALIBRATION_SECTOR_DIMENSIONS  = 5;
int const swath_radiances::NOTPRESENT_VERBOSE             = 300;
int const swath_radiances::INVALID_VERBOSE                = 301;
int const swath_radiances::SOLARNIGHT_VERBOSE             = 302;
int const swath_radiances::SIGNALINFO_VERBOSE             = 303;
int const swath_radiances::MISSING_WARNING                = 803;
int const swath_radiances::NEGATIVE_WARNING               = 804;

int const MINTRIPLETNUMBER                                = 11;

hdf_swath_data_dataset swath_radiances::radiancedataset;
hdf_swath_data_dataset swath_radiances::calibrationdataset;
hdf_swath_data_dataset swath_radiances::calibrationsectordataset;
hdf_swath_data_dataset swath_radiances::level0sddataset;

swath_radiances :: swath_radiances ()
{
  for (int channel = 0; channel < SWATH_CHANNELS; channel++)
    for (int pixel = 0; pixel < SWATH_PIXELS; pixel++) {
      for (int stare = 0; stare < SWATH_STARES; stare++) {
	lastcell[stare][channel] = SWATH_MISSING_INT;  
        for (int sigterm = 0; sigterm < SWATH_SAWTOOTH_SET; sigterm++)
          signals [stare] [channel] [pixel] [sigterm] = SWATH_MISSING_VALUE;
        for (int radstate = 0; radstate < SWATH_RADIANCE_STATES; radstate++) {
          radiances [stare] [pixel] [channel] [radstate] = SWATH_MISSING_VALUE;
	  level0sd  [stare] [pixel] [channel] [radstate] = SWATH_MISSING_VALUE;
	}
      }
      for (int caliterm = 0; caliterm < SWATH_CALIBRATION_TERMS; caliterm++) {
	for (int calistate = 0; calistate < SWATH_CALIBRATION_STATES; calistate++)
          calibration [pixel] [channel] [calistate] [caliterm] = SWATH_MISSING_VALUE;
	for (int radstate = 0; radstate < SWATH_RADIANCE_STATES; radstate++)
          calibrationAD [pixel] [channel] [radstate] [caliterm] = SWATH_MISSING_VALUE;
	for (int sectstate = 0; sectstate < SWATH_LMC_SECTORS; sectstate++)
          calibrationsector [pixel] [channel] [sectstate] [caliterm] = SWATH_MISSING_VALUE;
      }
    }
}


swath_radiances :: ~swath_radiances ()
{
  ;
}


void swath_radiances :: AddLastcellState (int stareindex, int lastcellstate [SWATH_CHANNELS]) const
{
  (void) memcpy ((void*) lastcell [stareindex], (const void*) lastcellstate, sizeof (int [SWATH_CHANNELS]));
}


void swath_radiances :: AddSignals (int stareindex, 
                                    double const staresignals [SWATH_CHANNELS] [SWATH_PIXELS] [SWATH_SAWTOOTH_SET]) const
{
  (void) memcpy ((void*) signals [stareindex], (const void*) staresignals, 
                 sizeof (double [SWATH_CHANNELS] [SWATH_PIXELS] [SWATH_SAWTOOTH_SET]));
}


void swath_radiances :: CalculateSawtooth (int st, int px, int ch, int indcali [SWATH_SAWTOOTH_SET],  
                                           double radian [SWATH_SAWTOOTH_SET], int ind [SWATH_SAWTOOTH_SET] )
{ 
  for ( int sawtooth = 0; sawtooth < SWATH_SAWTOOTH_SET; sawtooth++ ) {
    radian[ind[sawtooth]] = calibration[px][ch][indcali[sawtooth]][SWATH_CALIBRATION_OFFSET_MEAN] +
      (calibration[px][ch][indcali[sawtooth]][SWATH_CALIBRATION_GAIN_MEAN] * 
       signals[st][ch][px][ind[sawtooth]] ); 
  }
}


void swath_radiances :: CalculateTriplets (int ind [SWATH_SAWTOOTH_SET],  double radian [SWATH_SAWTOOTH_SET], 
					   double gdtripavg[SWATH_TRIPLET_SET], double gdtripdif[SWATH_TRIPLET_SET], 
					   double gdtriprat[SWATH_TRIPLET_SET], int& tripcount)
{ 
  double tripavg[SWATH_TRIPLET_SET], tripdif[SWATH_TRIPLET_SET];
  double meanavg=0.0, meandif=0.0, stdavg=0.0, stddif=0.0;
  int trip;

  for ( trip = 0; trip < SWATH_TRIPLET_SET; trip++ ) {
    tripavg[trip] = ( ( radian [ind[trip]] + radian [ind[trip+2]] ) / 2.0 + radian [ind[trip+1]] ) / 2.0;
    if ( ind[trip] % 2 == 0 )       // long path
      tripdif[trip] =   radian [ind[trip+1]] - ( radian [ind[trip]] + radian [ind[trip+2]] ) / 2.0 ;
    else                            // short path
      tripdif[trip] = ( radian [ind[trip]] + radian [ind[trip+2]] ) / 2.0 - radian [ind[trip+1]] ;
    
    // get mean of avg and diff triplet radiances
    meanavg += tripavg[trip];
    meandif += tripdif[trip];
  }
  meanavg = meanavg / SWATH_TRIPLET_SET;
  meandif = meandif / SWATH_TRIPLET_SET;

  // get stddev of avg and diff triplet radiances 
  for ( trip = 0; trip < SWATH_TRIPLET_SET; trip++ ) {
    stdavg += pow ( (meanavg - tripavg[trip]), 2.0 );
    stddif += pow ( (meandif - tripdif[trip]), 2.0 );
  }
  stdavg = sqrt ( stdavg / SWATH_TRIPLET_SET ); 
  stddif = sqrt ( stddif / SWATH_TRIPLET_SET ); 

  tripcount = 0;
  // check valid triplet radiances
  for ( trip = 0; trip < SWATH_TRIPLET_SET; trip++ ) 
    if ( (tripavg[trip] < (meanavg + 2*stdavg)) && (tripavg[trip] > (meanavg - 2*stdavg)) &&
	 (tripdif[trip] < (meandif + 2*stddif)) && (tripdif[trip] > (meandif - 2*stddif)) ) {
      gdtripavg[tripcount] = tripavg[trip];
      gdtripdif[tripcount] = tripdif[trip];
      if ( fabs (tripavg[trip]) >  pow ( (double) 10.00, (double) -5 ) )
	gdtriprat[tripcount] = tripdif[trip] / tripavg[trip]; 
      else 
	gdtriprat[tripcount] = 0.0;
      tripcount ++;
    }
}


void  swath_radiances :: GetADSectorCalibration ( )
{
  for (int pixel = 0; pixel < SWATH_PIXELS; pixel++) 
    for (int channel = 0; channel < SWATH_CHANNELS; channel++)
      for (int caliterm = 0; caliterm < SWATH_CALIBRATION_TERMS; caliterm++) {

	calibrationAD [pixel][channel][SWATH_RADIANCE_AVERAGE][caliterm]    = 
	                     calibration [pixel][channel][SWATH_CALIBRATION_STATE_AVERAGE][caliterm];
	calibrationAD [pixel][channel][SWATH_RADIANCE_DIFFERENCE][caliterm] = 
                             calibration [pixel][channel][SWATH_CALIBRATION_STATE_DIFFERENCE][caliterm];

	calibrationsector [pixel][channel][SWATH_LMC_A_SECTOR][caliterm] = 
	                     calibration [pixel][channel][SWATH_CALIBRATION_A_SECTOR][caliterm];
	calibrationsector [pixel][channel][SWATH_LMC_B_SECTOR][caliterm] = 
	                     calibration [pixel][channel][SWATH_CALIBRATION_B_SECTOR][caliterm];
	calibrationsector [pixel][channel][SWATH_LMC_C_SECTOR][caliterm] = 
	                     calibration [pixel][channel][SWATH_CALIBRATION_C_SECTOR][caliterm];
	calibrationsector [pixel][channel][SWATH_LMC_D_SECTOR][caliterm] = 
	                     calibration [pixel][channel][SWATH_CALIBRATION_D_SECTOR][caliterm];
      }
}


void swath_radiances :: Calibrate (swath_flags const& flags, swath_times const& times, swath_packetpositions positions, 
				   calibration_histories& calhistories, int tracknumber,
                                   processor_parameters const& processorparameters )
{
  int channel, pixel, stare, packetposition, goodtripnum;
  bool invalidate;
  double rad [SWATH_SAWTOOTH_SET], tripletavg [SWATH_TRIPLET_SET], tripletdif [SWATH_TRIPLET_SET], 
         tripletrat [SWATH_TRIPLET_SET];

  int ind0 [SWATH_SAWTOOTH_SET] = { 1, 2, 3, 0, 5, 6, 7, 4, 9, 10, 11, 8, 13, 14, 15, 12 };
  int ind1 [SWATH_SAWTOOTH_SET] = { 2, 3, 0, 1, 6, 7, 4, 5, 10, 11, 8, 9, 14, 15, 12, 13 };
  int ind2 [SWATH_SAWTOOTH_SET] = { 3, 0, 1, 2, 7, 4, 5, 6, 11, 8, 9, 10, 15, 12, 13, 14 };
  int ind3 [SWATH_SAWTOOTH_SET] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 };

  int calistate0 [SWATH_SAWTOOTH_SET] = { 3, 4, 5, 2, 3, 4, 5, 2, 3, 4, 5, 2, 3, 4, 5, 2 };
  int calistate1 [SWATH_SAWTOOTH_SET] = { 4, 5, 2, 3, 4, 5, 2, 3, 4, 5, 2, 3, 4, 5, 2, 3 };
  int calistate2 [SWATH_SAWTOOTH_SET] = { 5, 2, 3, 4, 5, 2, 3, 4, 5, 2, 3, 4, 5, 2, 3, 4 };
  int calistate3 [SWATH_SAWTOOTH_SET] = { 2, 3, 4, 5, 2, 3, 4, 5, 2, 3, 4, 5, 2, 3, 4, 5 };

  // compute the track time (the midpoint of the track)
  mopitt_time starttime ((times.GetStartTime ()).GetTAI ());
  mopitt_time tracktime ((((times.GetStopTime ()).GetTAI () + starttime.GetTAI ()) / 2.0));

  // retrieve the track's calibration terms and store them in the calibration container
  float calibdata [SWATH_CHANNELS] [SWATH_PIXELS] [SWATH_CALIBRATION_STATES] [SWATH_CALIBRATION_TERMS];
  double coldcaldata [SWATH_CHANNELS] [SWATH_PACKETPOSITION] [SWATH_PIXELS] [SWATH_PMC_STATES];
  calhistories.GetTrackData (tracktime, calibdata, coldcaldata);
  for (channel = 0; channel < SWATH_CHANNELS; channel++)
    for (pixel = 0; pixel < SWATH_PIXELS; pixel++)
      (void) memcpy ((void*) calibration [pixel] [channel], (const void*) calibdata [channel] [pixel],
                     sizeof (float [SWATH_CALIBRATION_STATES] [SWATH_CALIBRATION_TERMS]));
  GetADSectorCalibration ( );

  // for each stare in the track
  for ( stare = 0; stare < SWATH_STARES; stare++) {

    // if the stare is present
    if (! flags.IsPresent (stare)) 
      diagnosticreporter.AddEntry (DIAGNOSTICS_VERBOSE, DIAGNOSTICS_SWATH_MODULE, NOTPRESENT_VERBOSE, 0, tracknumber, 
                                   0, (stare + 1), 0, 0, 0, 0, "The stare is not present");
    else {
              
      //get stare packet position
      packetposition = positions.GetPosition(stare) - 1;

      // for each pixel
      for (pixel = 0; pixel < SWATH_PIXELS; pixel++) {

        // for each channel
        for (channel = 0; channel < SWATH_CHANNELS; channel++) {

          // reset the invalidation flag
          invalidate = false;

          // add an entry of the time/stare/pixel/channel of the signal and it's error
          OutputSignalInfo (starttime, tracknumber, stare, channel, pixel, signals [stare] [channel] [pixel]);

          // if stare/pixel/channel is invalid, set radiance to invalid
          if (flags.IsInvalid (stare, channel, pixel)) {
            radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE]    = SWATH_INVALID_VALUE;
            radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] = SWATH_INVALID_VALUE;
            diagnosticreporter.AddEntry (DIAGNOSTICS_VERBOSE, DIAGNOSTICS_SWATH_MODULE, INVALID_VERBOSE, 0, 
                                         tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                         "The data is marked invalid");
          }

          // output message if solar night, else process if the stare/pixel/channel is thermal or night flag is not set
          else if ((channel % 2) == 1 && flags.IsNight (stare, pixel))
            diagnosticreporter.AddEntry (DIAGNOSTICS_VERBOSE, DIAGNOSTICS_SWATH_MODULE, SOLARNIGHT_VERBOSE, 0,
                                         tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                         "Solar channel at night");
          else {

            // compute radiances for PMC channel 
	    if ( channel == 2 || channel == 6 ) {
              // compute PMC average radiances 
	      if ( (int) (signals [stare] [channel] [pixel] [0] + 0.5) == SWATH_MISSING_INT) {
                if ( (processorparameters.GetCh7Info() ) == 0 || 
                     (processorparameters.GetCh7Info() ) == 1 && channel != 2 )
                  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, MISSING_WARNING, 0,
                                             tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                             "Average radiance not computed due to missing signal");
	      }
              else if ((int)(calibration[pixel][channel][SWATH_CALIBRATION_STATE_AVERAGE][SWATH_CALIBRATION_GAIN_MEAN]+0.5)
                           == SWATH_MISSING_INT ||
                       (int)(coldcaldata[channel][packetposition][pixel][SWATH_PMC_AVERAGE]+0.5) == SWATH_MISSING_INT) {
                if ( (processorparameters.GetCh7Info() ) == 0 || 
                     (processorparameters.GetCh7Info() ) == 1 && channel != 2 )
		  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, MISSING_WARNING, 0,
                                             tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                             "Average radiance not computed due to missing calibration data");
	      }
              else {
		radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] = (float)
                  calibration [pixel] [channel] [SWATH_CALIBRATION_STATE_AVERAGE] [SWATH_CALIBRATION_GAIN_MEAN] *
		  (signals [stare] [channel] [pixel] [0] - 
		  coldcaldata [channel] [packetposition] [pixel] [SWATH_PMC_AVERAGE]);

                if (radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] < 0.0) {
                  invalidate = true;
                  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, NEGATIVE_WARNING, 0,
                                               tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                               "Radiance PMC computation yielded negative average radiance");
                }
                if ( isnan(radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE]) != 0) {
                  invalidate = true;
                  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, NEGATIVE_WARNING, 0,
                                               tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                               "Radiance PMC computation yielded NaN average radiance");
                }
              }

              // compute PMC difference radiances
              if ( (int) (signals [stare] [channel] [pixel] [2] + 0.5) == SWATH_MISSING_INT) {
                if ( (processorparameters.GetCh7Info() ) == 0 || 
                     (processorparameters.GetCh7Info() ) == 1 && channel != 2 )
		  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, MISSING_WARNING, 0,
                                             tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                             "Difference radiance not computed due to missing signal");
	      }
              else if ((int)(calibration[pixel][channel][SWATH_CALIBRATION_STATE_DIFFERENCE][SWATH_CALIBRATION_GAIN_MEAN] 
                           + 0.5) == SWATH_MISSING_INT ||
                       (int)(coldcaldata[channel][packetposition][pixel][SWATH_PMC_DIFFERENCE]+0.5) == SWATH_MISSING_INT){
                if ( (processorparameters.GetCh7Info() ) == 0 || 
                     (processorparameters.GetCh7Info() ) == 1 && channel != 2 )
		  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, MISSING_WARNING, 0,
                                             tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                             "Difference radiance not computed due to missing calibration data");
	      }
              else {
                radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] = (float)
                  calibration [pixel] [channel] [SWATH_CALIBRATION_STATE_DIFFERENCE] [SWATH_CALIBRATION_GAIN_MEAN] *
                  ( signals [stare] [channel] [pixel] [2] - 
		  coldcaldata [channel] [packetposition] [pixel] [SWATH_PMC_DIFFERENCE] );

                if (radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] < 0.0) {
                  invalidate = true;
                  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, NEGATIVE_WARNING, 0,
                                               tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                               "Radiance PMC computation yielded negative difference radiance");
                }
                if (isnan(radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE]) != 0) {
                  invalidate = true;
                  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, NEGATIVE_WARNING, 0,
                                               tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                               "Radiance PMC computation yielded NaN difference radiance");
                }
              }
	    }
	    else { // compute radiance for LMC channels 
              if ( (int) (signals [stare] [channel] [pixel] [0] + 0.5) == SWATH_MISSING_INT) {
		if ( (processorparameters.GetCh7Info() ) == 0 || 
		     (processorparameters.GetCh7Info() ) == 1 && channel != 0 && channel != 1 && channel != 3 )
		  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, MISSING_WARNING, 0,
                                             tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
                                             "Average radiance not computed due to missing signal");
	      }
              else if ((int)(calibration[pixel][channel][SWATH_CALIBRATION_STATE_AVERAGE][SWATH_CALIBRATION_GAIN_MEAN]+0.5)
                           == SWATH_MISSING_INT ||
                       (int)(calibration[pixel][channel][SWATH_CALIBRATION_STATE_AVERAGE][SWATH_CALIBRATION_OFFSET_MEAN] 
			      + 0.5) == SWATH_MISSING_INT) {
		if ( (processorparameters.GetCh7Info() ) == 0 || 
		     (processorparameters.GetCh7Info() ) == 1 && channel != 0 && channel != 1 && channel != 3 )
		  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, MISSING_WARNING, 0,
					       tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
					       "Average radiance not computed due to missing calibration data");
	      }
              else {
		switch ( lastcell [stare] [channel] ) {
		case 0: CalculateSawtooth ( stare, pixel, channel, calistate0, rad, ind0 );
		        CalculateTriplets ( ind0, rad, tripletavg, tripletdif, tripletrat, goodtripnum );	
	                break;
		case 1: CalculateSawtooth ( stare, pixel, channel, calistate1, rad, ind1 );
		        CalculateTriplets ( ind1, rad, tripletavg, tripletdif, tripletrat, goodtripnum );
		        break;
		case 2: CalculateSawtooth ( stare, pixel, channel, calistate2, rad, ind2 );
		        CalculateTriplets ( ind2, rad, tripletavg, tripletdif, tripletrat, goodtripnum );
	                break;
		case 3: CalculateSawtooth ( stare, pixel, channel, calistate3, rad, ind3 );
                        CalculateTriplets ( ind3, rad, tripletavg, tripletdif, tripletrat, goodtripnum );
	                break;
		default:diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, NEGATIVE_WARNING, 0,
                                 tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0, "wrong last cell state");
	        	break;
		}

		if ( goodtripnum <= MINTRIPLETNUMBER  ) 
		  invalidate = true;
		else {
		  radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] = (float) tripletavg[0];
		  if ( (channel % 2) == 1 )
		    radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] = (float) tripletrat[0];
		  else
		    radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] = (float) tripletdif[0];
		  
		  for ( int trip = 1; trip < goodtripnum ; trip++ ) {
		    radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] += (float) tripletavg[trip];
		    if ( (channel % 2) == 1 )
		      radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] += (float) tripletrat[trip];
		    else
		      radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] += (float) tripletdif[trip];
		  }
		  radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] =
		    radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] / goodtripnum;
		  radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] = 
		    radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] / goodtripnum;		
		  if ( (channel % 2) == 1 )
		    radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] *= 
		      radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE]; 

		  // add level0 stddev
		  level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] =  
		    pow((radiances[stare][pixel][channel][SWATH_RADIANCE_AVERAGE] - tripletavg[0]), 2.0);
		  if ( (channel % 2) == 1 )
		    level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] =
		      pow(( radiances[stare][pixel][channel][SWATH_RADIANCE_DIFFERENCE] / 
			    radiances[stare][pixel][channel][SWATH_RADIANCE_AVERAGE] - tripletrat[0]), 2.0);	      
		  else
		    level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] = 
		      pow((radiances[stare][pixel][channel][SWATH_RADIANCE_DIFFERENCE] - tripletdif[0]), 2.0);	      
		  
		  for ( int trip = 1; trip < goodtripnum ; trip++ ) {
		    level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] += 
		      pow((radiances[stare][pixel][channel][SWATH_RADIANCE_AVERAGE] - tripletavg[trip]), 2.0);
		    if ( (channel % 2) == 1 )
		      level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] += 
			pow(( radiances[stare][pixel][channel][SWATH_RADIANCE_DIFFERENCE] / 
			      radiances[stare][pixel][channel][SWATH_RADIANCE_AVERAGE] - tripletrat[trip]), 2.0);
		    else
		      level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] += 
			pow((radiances[stare][pixel][channel][SWATH_RADIANCE_DIFFERENCE] - tripletdif[trip]), 2.0);
		  }
		  level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] =
		    sqrt(level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] / goodtripnum);
		  level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] = 
		    sqrt(level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] / goodtripnum);		
		  if ( (channel % 2) == 1 )
		    level0sd [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] *= 
		      radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE]; 

		  if (radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE] < 0.0 ) {
		    invalidate = true;
		    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, NEGATIVE_WARNING, 0,
						 tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
						 "Radiance LMC computation yielded negative average radiance");
		  }
		  if (isnan(radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE]) != 0 ) {
		    invalidate = true;
		    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, NEGATIVE_WARNING, 0,
						 tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
						 "Radiance LMC computation yielded NaN average radiance");
		  }
		  if (radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] < 0.0) {
		    invalidate = true;
		    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, NEGATIVE_WARNING, 0,
						 tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
						 "Radiance LMC computation yielded negative difference radiance");
		  }
		  if (isnan(radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE]) != 0) {
		    invalidate = true;
		    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, NEGATIVE_WARNING, 0,
						 tracknumber, 0, (stare + 1), 0, (channel + 1), (pixel + 1), 0,
						 "Radiance LMC computation yielded NaN difference radiance");
		  }
		} 
	      }
	    }

	    // make sure there are no negative computed radiances
	    if (invalidate ) {
	      radiances [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE]    = SWATH_INVALID_VALUE;
	      radiances [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] = SWATH_INVALID_VALUE;
	      level0sd  [stare] [pixel] [channel] [SWATH_RADIANCE_AVERAGE]    = SWATH_INVALID_VALUE;
	      level0sd  [stare] [pixel] [channel] [SWATH_RADIANCE_DIFFERENCE] = SWATH_INVALID_VALUE;
	    }
	  }
	}
      }
    }
  }
}



bool swath_radiances :: Define (hid_t dataid)
{
  bool isdefined = false;

  hsize_t const radiancesizes [RADIANCE_DATASET_DIMENSIONS] = { DIM_TMP_TRACK, DIM_STARES, DIM_PIXELS, 
								DIM_CHANNELS, DIM_RADIANCE_STATES };
  string const radiancenames [RADIANCE_DATASET_DIMENSIONS] = { "ntrack", "nstare", "npixels", "nchan", "nstate" };
  string const unitsrad = "W/m^2str";
  
  hsize_t const calibrationsizes [CALIBRATION_DATASET_DIMENSIONS] = { DIM_TMP_TRACK, DIM_PIXELS, DIM_CHANNELS, 
                                                                    DIM_RADIANCE_STATES, DIM_CALIBRATION_TERMS };
  string const calibrationnames [CALIBRATION_DATASET_DIMENSIONS] = { "ntrack", "npixels", "nchan", "nstate", "ncalib" };

  hsize_t const calibrationsectorsizes [CALIBRATION_SECTOR_DIMENSIONS] = { DIM_TMP_TRACK, DIM_PIXELS, DIM_CHANNELS, 
                                                                          DIM_LMC_SECTORS, DIM_CALIBRATION_TERMS };
  string const calibrationsectornames [CALIBRATION_SECTOR_DIMENSIONS] = {"ntrack","npixels","nchan","nsector","ncalib" };

  string const unitslev0 = "NA";

  
  if (radiancedataset.Define (dataid, "MOPITTRadiances", H5T_NATIVE_FLOAT, RADIANCE_DATASET_DIMENSIONS, radiancesizes,
                              radiancenames, unitsrad))
    if (calibrationdataset.Define (dataid, "CalibrationData", H5T_NATIVE_FLOAT, CALIBRATION_DATASET_DIMENSIONS,
                                   calibrationsizes, calibrationnames, unitsrad))
      if (calibrationsectordataset.Define (dataid, "SectorCalibrationData", H5T_NATIVE_FLOAT, 
					   CALIBRATION_SECTOR_DIMENSIONS,calibrationsectorsizes, calibrationsectornames, unitsrad))
	if (level0sddataset.Define (dataid, "Level0StdDev", H5T_NATIVE_FLOAT, RADIANCE_DATASET_DIMENSIONS, 
				    radiancesizes, radiancenames, unitslev0))
	  isdefined = true;

  return isdefined;
}


bool swath_radiances :: Write (hid_t dataid, int trackindex)
{
  bool iswritten = true;

  if (radiancedataset.Write (dataid, trackindex, (VOIDP) radiances))
    if (calibrationdataset.Write (dataid, trackindex, (VOIDP) calibrationAD))
      if (calibrationsectordataset.Write (dataid, trackindex, (VOIDP) calibrationsector))
	if (calibrationsectordataset.Write (dataid, trackindex, (VOIDP) calibrationsector))
	  if (level0sddataset.Write (dataid, trackindex, (VOIDP) level0sd))
	    iswritten = true;

  return iswritten;
}


void swath_radiances :: OutputSignalInfo (mopitt_time const& starttime, int tracknumber, int stare, int channel, 
                                          int pixel, double const signals [SWATH_SAWTOOTH_SET])
{
  char signalchars [320];
  (void) sprintf (signalchars, "%5d  %8d  %e  %e  %e  %e  %e  %e  %e  %e  %e  %e  %e  %e  %e  %e  %e  %e", 
                  starttime.GetDay (), starttime.GetMsecs (), signals [0], signals [1],signals [2], 
                  signals [3], signals [4], signals [5], signals [6], signals [7],signals [8], signals [9],
                  signals [10],signals [11], signals [12],signals [13], signals [14], signals [15]);
  string message = signalchars;
  diagnosticreporter.AddEntry (DIAGNOSTICS_VERBOSE, DIAGNOSTICS_SWATH_MODULE, SIGNALINFO_VERBOSE, 0, tracknumber,
                               0, (stare + 1), 0, (channel + 1), (pixel + 1), 0, message);
}
