/********** SwathLocation.C ******************************************************************************************\

 $Header$

 REVISION HISTORY
   05/99   Charles Cavanaugh
   11/99   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include "PGS_CSC.h"
#include "SwathLocation.h"
#include "DiagnosticReporter.h"

extern diagnostic_reporter diagnosticreporter;

int const    swath_location::DATASET_DIMENSIONS    = 3;
int const    swath_location::LATLON_WARNING        = 800;
int const    swath_location::SUNECR_WARNING        = 801;
int const    swath_location::ZENITHAZIMUTH_WARNING = 802;
double const swath_location::DEGREES_IN_RADIAN  = 57.29578;

hdf_swath_geolocation_dataset swath_location::latitudedataset;
hdf_swath_geolocation_dataset swath_location::longitudedataset;
hdf_swath_data_dataset swath_location::solarzenithdataset;
hdf_swath_data_dataset swath_location::solarazimuthdataset;
hdf_swath_data_dataset swath_location::satellitezenithdataset;
hdf_swath_data_dataset swath_location::satelliteazimuthdataset;

swath_location :: swath_location ()
{
  for (int stare = 0; stare < SWATH_STARES; stare++)
    for (int pixel = 0; pixel < SWATH_PIXELS; pixel++) {
      latitude         [stare] [pixel] = SWATH_MISSING_VALUE;
      longitude        [stare] [pixel] = SWATH_MISSING_VALUE;
      solarzenith      [stare] [pixel] = SWATH_MISSING_VALUE;
      solarazimuth     [stare] [pixel] = SWATH_MISSING_VALUE;
      satellitezenith  [stare] [pixel] = SWATH_MISSING_VALUE;
      satelliteazimuth [stare] [pixel] = SWATH_MISSING_VALUE;
    }
}


swath_location :: ~swath_location ()
{
  ;
}


bool swath_location :: GetLatLon (string const& utctime, PGSt_double timeoffsets [SWATH_PIXELS], 
                                  PGSt_double pixeluvs [SWATH_PIXELS][VECTOR_LENGTH], PGSt_double lats [SWATH_PIXELS],
                                  PGSt_double lons [SWATH_PIXELS], PGSt_double ecrscuvs [SWATH_PIXELS][VECTOR_LENGTH])
   const
{
  bool isretrieved = true;

  // get the latitude and longitude of the pixels
  PGSt_double xyzoffsets [SWATH_PIXELS][VECTOR_LENGTH], slantrange [SWATH_PIXELS], dopvel [SWATH_PIXELS];
  PGSt_SMF_status smfcode;
  smfcode = PGS_CSC_GetFOV_Pixel (PGSd_EOS_AM, SWATH_PIXELS, (char*) utctime.c_str (), timeoffsets, "WGS84", PGS_FALSE,
                                  pixeluvs, xyzoffsets, lats, lons, ecrscuvs, slantrange, dopvel);
  if (smfcode != PGS_S_SUCCESS && smfcode != PGSTD_W_PRED_LEAPS && smfcode != PGSCSC_W_PREDICTED_UT1)
    isretrieved = false;

  return isretrieved;
}


//int const CB_ID_SUN = 11;

bool swath_location :: GetSunECR (string const& utctime, PGSt_double timeoffsets [SWATH_PIXELS], 
                                  PGSt_double ecrsunuvs [SWATH_PIXELS][VECTOR_LENGTH]) const
{
  bool isretrieved = true;
  PGSt_SMF_status smfcode;
  int CB_ID_SUN = 11;

  // get the Earth-to-Sun ECI values
  PGSt_double ecisun [SWATH_PIXELS] [VECTOR_LENGTH];
  smfcode = PGS_CBP_Earth_CB_Vector (SWATH_PIXELS, (char*) utctime.c_str (), timeoffsets, CB_ID_SUN, ecisun);
  if (smfcode != PGS_S_SUCCESS && smfcode != PGSTD_W_PRED_LEAPS) {
      diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE,803,0,0,0,0,0,0,0,0,"failed at PGS_CBP_Earth_CB_Vector");
      isretrieved = false;
  }
  else {

    // convert the Sun ECI values to ECR values (first unitizing the ECI values)
    PGSt_double mag, ecisunuvs [SWATH_PIXELS] [VECTOR_LENGTH * 2];
    for (int pixel = 0; pixel < SWATH_PIXELS; pixel++) {
       mag = (PGSt_double) sqrt ((ecisun [pixel] [0] * ecisun [pixel] [0]) + (ecisun [pixel] [1] * ecisun [pixel] [1]) +
                                 (ecisun [pixel] [2] * ecisun [pixel] [2]));
       ecisunuvs [pixel] [0] = ecisun [pixel] [0] / mag;
       ecisunuvs [pixel] [1] = ecisun [pixel] [1] / mag;
       ecisunuvs [pixel] [2] = ecisun [pixel] [2] / mag;
       ecisunuvs [pixel] [3] = ecisunuvs [pixel] [4] = ecisunuvs [pixel] [5] = 0.0;
    }
    PGSt_double ecruvs [SWATH_PIXELS] [VECTOR_LENGTH*2];
    smfcode = PGS_CSC_ECItoECR (SWATH_PIXELS, (char*) utctime.c_str (), timeoffsets, ecisunuvs, ecruvs);
    if (smfcode != PGS_S_SUCCESS && smfcode != PGSTD_W_PRED_LEAPS && smfcode != PGSCSC_W_PREDICTED_UT1) {
        diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE,804,0,0,0,0,0,0,0,0,"failed at PGS_CSC_ECItoECR");      
        isretrieved = false;
    }
    else 
      for (int pixel = 0; pixel < SWATH_PIXELS; pixel++)
        for (int i = 0; i < VECTOR_LENGTH; i++)
          ecrsunuvs [pixel] [i] = ecruvs [pixel] [i];
  }

  return isretrieved;
}


bool swath_location :: GetZenithAzimuth (PGSt_double lats [SWATH_PIXELS], PGSt_double lons [SWATH_PIXELS], 
                                         PGSt_double ecruvs [SWATH_PIXELS][VECTOR_LENGTH], PGSt_tag tagval, 
                                         float zeniths [SWATH_PIXELS], float azimuths [SWATH_PIXELS], 
                                         bool isnight [SWATH_PIXELS]) const
{
  bool isretrieved = true;

  PGSt_double zenith, azimuth, refraction;
  PGSt_SMF_status smfcode;

  for (int pixel = 0; pixel < SWATH_PIXELS; pixel++) {
  
    /*** get the zenith and azimuth for this pixel ***/
    isnight [pixel] = false;
    smfcode = PGS_CSC_ZenithAzimuth (ecruvs [pixel], lats [pixel], lons [pixel], 0.0, tagval, PGS_FALSE, PGS_FALSE,
                                     &zenith, &azimuth, &refraction);
    if (smfcode == PGSCSC_W_BELOW_HORIZON) {
      isnight [pixel] = true;
      zeniths [pixel] = (float) (zenith  * DEGREES_IN_RADIAN);
      azimuths[pixel] = (float) (azimuth * DEGREES_IN_RADIAN);
    }
    else if (smfcode == PGSCSC_W_UNDEFINED_AZIMUTH) {
      zeniths[pixel] = 0.0;
      azimuths[pixel] = 0.0;
    }
    else if (smfcode == PGS_S_SUCCESS) {
      zeniths [pixel] = (float) (zenith  * DEGREES_IN_RADIAN);
      azimuths[pixel] = (float) (azimuth * DEGREES_IN_RADIAN);
    }
    else
      isretrieved = false;
  }

  return isretrieved;
}


void swath_location :: GeoLocate (swath_flags& flags, swath_times const& times, mopip_file const& mopipfile, 
                                  int tracknumber)
{
  int pixel;

  // for each stare in the track
  for (int stare = 0; stare < SWATH_STARES; stare++)

    // if this stare is present
    if (flags.IsPresent (stare)) {

      // retrieve the utc time for this stare
      string utctime = ((times.GetTime (stare)).GetUTC ()).c_str ();
  
      // retrieve the unit vectors for all pixels in this stare
      double mopipuvs [SWATH_PIXELS][VECTOR_LENGTH];
      mopipfile.GetUVs (stare, mopipuvs);
      PGSt_double uvs [SWATH_PIXELS][VECTOR_LENGTH];
      for (pixel = 0; pixel < PIXELS; pixel++)
        for (int i = 0; i < VECTOR_LENGTH; i++)
          uvs[pixel][i] = (PGSt_double) mopipuvs[pixel][i];

      // retrieve the latitude and longitude 
      PGSt_double lats [SWATH_PIXELS], lons [SWATH_PIXELS], ecrscuvs [SWATH_PIXELS] [VECTOR_LENGTH],
                  timeoffsets [SWATH_PIXELS] = {0.0, 0.0, 0.0, 0.0};
      if (! GetLatLon (utctime, timeoffsets, uvs, lats, lons, ecrscuvs))
        diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, LATLON_WARNING, 0, tracknumber,
                                     0, (stare + 1), 0, 0, 0, 0, "Could not retrieve latitude and longitude");
      else {

        // store the latitude and longitude in degrees
        for (pixel = 0; pixel < SWATH_PIXELS; pixel++) {
          latitude[stare][pixel] = lats[pixel] * DEGREES_IN_RADIAN;
          longitude[stare][pixel] = lons[pixel] * DEGREES_IN_RADIAN;
        }

        // retrieve the sun ECR values
        PGSt_double ecrsunuvs [SWATH_PIXELS][VECTOR_LENGTH];
        if (! GetSunECR (utctime, timeoffsets, ecrsunuvs)) 
          diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, SUNECR_WARNING, 0, tracknumber, 
                                       0, (stare + 1), 0, 0, 0, 0, "Could not retrieve the Sun ECR coordinates");
        else {

          // retrieve the spacecraft zenith and azimuth
          bool isnight [SWATH_PIXELS];
          if (! GetZenithAzimuth (lats, lons, ecrscuvs, PGSd_LOOK, satellitezenith[stare], satelliteazimuth[stare], 
                                isnight))
            diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, ZENITHAZIMUTH_WARNING, 0,
                                         tracknumber, 0, (stare + 1), 0, 0, 0, 0,
                                         "Could not retrieve satellite zenith and azimuth");

          // retrieve the solar zenith and azimuth
          if (! GetZenithAzimuth (lats, lons, ecrsunuvs, PGSd_SUN, solarzenith[stare], solarazimuth[stare], isnight))
            diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SWATH_MODULE, ZENITHAZIMUTH_WARNING, 0,
                                         tracknumber, 0, (stare + 1), 0, 0, 0, 0,
                                         "Could not retrieve solar zenith and azimuth");

          // set the night flags (as output from solar zenith/azimuth retrieval)
          for (pixel = 0; pixel < SWATH_PIXELS; pixel++)
            flags.SetNight (stare, pixel, isnight [pixel]);
        }
      }
    }
}


bool swath_location :: Define (hid_t dataid)
{
  bool isdefined = false;

  hsize_t const sizes [DATASET_DIMENSIONS] = { DIM_TMP_TRACK, DIM_STARES, DIM_PIXELS };
  string const names [DATASET_DIMENSIONS] = { "ntrack", "nstare", "npixels"};
  string const unitslat = "degrees_north";
  string const unitslon = "degrees_east";
  string const unitsdeg = "deg";
  
  if (latitudedataset.Define (dataid, "Latitude", H5T_NATIVE_FLOAT, DATASET_DIMENSIONS, sizes, names, unitslat))
    if (longitudedataset.Define (dataid, "Longitude", H5T_NATIVE_FLOAT, DATASET_DIMENSIONS, sizes, names, unitslon))
      if (solarzenithdataset.Define (dataid, "SolarZenith", H5T_NATIVE_FLOAT, DATASET_DIMENSIONS, sizes, names, unitsdeg))
        if (solarazimuthdataset.Define (dataid, "SolarAzimuth", H5T_NATIVE_FLOAT, DATASET_DIMENSIONS, sizes, names, unitsdeg))
          if (satellitezenithdataset.Define (dataid, "SatelliteZenith", H5T_NATIVE_FLOAT, DATASET_DIMENSIONS, sizes, 
                                             names, unitsdeg))
            if (satelliteazimuthdataset.Define (dataid, "SatelliteAzimuth", H5T_NATIVE_FLOAT, DATASET_DIMENSIONS, 
						sizes, names, unitsdeg))
              isdefined = true;

  return isdefined;
}


bool swath_location :: Write (hid_t dataid, int trackindex)
{
  bool iswritten = false;

  if (latitudedataset.Write (dataid, trackindex, (VOIDP) latitude))
    if (longitudedataset.Write (dataid, trackindex, (VOIDP) longitude))
      if (solarzenithdataset.Write (dataid, trackindex, (VOIDP) solarzenith))
        if (solarazimuthdataset.Write (dataid, trackindex, (VOIDP) solarazimuth))
          if (satellitezenithdataset.Write (dataid, trackindex, (VOIDP) satellitezenith))
            if (satelliteazimuthdataset.Write (dataid, trackindex, (VOIDP) satelliteazimuth))
              iswritten = true;

  return iswritten;
}
