/********** ScienceTrainData.C ****************************************************************************************\

 $Header$

 REVISION HISTORY
   11/99   Charles Cavanaugh
   11/00   Charles Cavanaugh
   02/02   Debbie Mao
   04/02   Debbie Mao
   05/02   Debbie Mao

 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include <memory.h>
#include <values.h>
#include "PGS_CBP.h"
#include "PGS_CSC.h"
#include "ScienceTrainData.h"

string const science_train_data::FIELDNAMES [FIELDCOUNT] = { "Stare Count", "Day", "Start Milliseconds", 
                                                             "Stop Milliseconds", "Shadow Flag", 
                                                             "Internal Temperature", "Channel I Signals", 
                                                             "Channel II Signals", "Channel I Sector Signals", 
                                                             "Channel II Sector Signals" ,"PMC ColdCal",
                                                             "PMC Warm Stddev"};
int32 const science_train_data::DATATYPES [FIELDCOUNT] = { DFNT_UINT16, DFNT_INT32, DFNT_INT32, DFNT_INT32, 
                                                           DFNT_UINT16, DFNT_FLOAT32, DFNT_FLOAT32, DFNT_FLOAT32, 
                                                           DFNT_FLOAT32, DFNT_FLOAT32, DFNT_FLOAT32, DFNT_FLOAT32};

science_train_data :: science_train_data ()
                    : vgroup ()
                    , vdata ()
{
  ResetNext ();
  haveshadowedinfo = false;
  shadowedcount = 0;
  shadowedindeces = NULL;
}


science_train_data :: ~science_train_data ()
{
  ResetShadowedInfo ();
}


bool science_train_data :: PreDefine (int train, string const& grouptype, int32 maingroupid, int32 fileid)
{
  bool isdefined = false;

  char groupname [40];
  (void) sprintf (groupname, "Science Train %1d %s Group", train, grouptype.c_str ());
  if (vgroup.Create (groupname, "Science Train Group", fileid)) {
    char dataname [40];
    (void) sprintf (dataname, "Science Train %1d %s Data", train, grouptype.c_str ());
    if (vdata.Create (dataname, "Science Train Data", fileid))
      if (vdata.Insert (vgroup.GetID ()))
        if (vgroup.Insert (maingroupid))
          isdefined = true;
  }

  return isdefined;
}


void science_train_data :: AddShadowedIndex (int index)
{
  // if there are no indeces, make this the first one
  if (shadowedcount == 0) {
    shadowedindeces = new int [1];
    shadowedindeces [0] = index;
  }

  // else add this index to the list
  else {
    int* temp = new int [shadowedcount];
    (void) memcpy ((void*) temp, (const void*) shadowedindeces, (sizeof (int) * shadowedcount));
    delete [] shadowedindeces;
    shadowedindeces = new int [shadowedcount + 1];
    (void) memcpy ((void*) shadowedindeces, (const void*) temp, (sizeof (int) * shadowedcount));
    delete [] temp;
    shadowedindeces [shadowedcount] = index;
  }
  shadowedcount++;
}
  

bool science_train_data :: Attach (int train, string const& grouptype, int32 fileid)
{
  bool isattached = false;
 
  char groupname [40];
  (void) sprintf (groupname, "Science Train %1d %s Group", train, grouptype.c_str ());
  if (vgroup.Attach (fileid, groupname)) {
    char dataname [40];
    (void) sprintf (dataname, "Science Train %1d %s Data", train, grouptype.c_str ());
    if (vdata.Attach (fileid, dataname, FIELDNAMES))
      isattached = true;
  }

  ResetShadowedInfo ();

  return isattached;
}


bool science_train_data :: Close ()
{
  bool isclosed = false;

  if (vgroup.Close ())
    if (vdata.Close ())
      isclosed = true;

  ResetShadowedInfo ();

  return isclosed;
}


int science_train_data :: GetShadowedCount ()
{
  // set info if necessary
  if (! haveshadowedinfo)
    BuildShadowedInfo ();

  return shadowedcount;
}


bool science_train_data :: ReadClosest (mopitt_time const& searchtime, science_calibration_record* record)
{
  bool isread = false;

  // save the index of the closest record
  int closestindex = 0;

  // make room for time comparison info
  double recorddiff;
  double timediff = MAXDOUBLE;

  // loop over all the records
  for (int index = 0; index < GetCount (); index++)

    // read the current record
    if (Read (index, record)) {

      // get the absolute time difference between this record's time and the searchtime
      recorddiff = AbsoluteTimeDiff (searchtime, record->GetStartTime ());

      // if the time difference is less than the previous lowest time difference, save index and difference
      if (recorddiff < timediff) {
        closestindex = index;
        timediff = recorddiff;
        isread = true;
      }
    }

  // read the record corresponding to the index of the closest record
  if (isread)
    if (! Read (closestindex, record))
      isread = false;

  return isread;
}


bool science_train_data :: ReadClosestShadowed (mopitt_time const& searchtime, science_calibration_record* record)
{
  bool isread = false;

  // set the shadowed info
  if (! haveshadowedinfo)
    BuildShadowedInfo ();

  // if there are shadowed records
  if (shadowedcount > 0) {

    // save the index of the closest shadowed record
    int closestindex = 0;

    // make room for time comparison info
    double recorddiff;
    double timediff = MAXDOUBLE;

    // loop over all the shadowed records
    for (int index = 0; index < GetShadowedCount (); index++)

      // read the current shadowed record
      if (Read (shadowedindeces [index], record)) {

        // get the absolute time difference between this record's time and the searchtime
        recorddiff = AbsoluteTimeDiff (searchtime, record->GetStartTime ());

        // if the time difference is less than the previous lowest time difference, save index and difference
        if (recorddiff < timediff) {
          closestindex = index;
          timediff = recorddiff;
          isread = true;
        }
      }

    // read the record corresponding to the index of the closest shadowed record
    if (isread)
      if (! Read (shadowedindeces [closestindex], record))
        isread = false;
  }

  return isread;
}


bool science_train_data :: ReadShadowed (int index, science_calibration_record* record)
{
  // default return is false
  bool isread = false;

  // set the shadowed info
  if (! haveshadowedinfo)
    BuildShadowedInfo ();

  // read only if valid index
  if (index < shadowedcount)
    isread = Read (shadowedindeces [index], record);

  return isread;
}


void science_train_data :: ResetNext ()
{
  next = 0;
}


void science_train_data :: ResetShadowedInfo ()
{
  if (haveshadowedinfo) {
    shadowedindeces = NULL;
    shadowedcount = 0;
    haveshadowedinfo = false;
  }
}
