/********** ScienceLMCChannel.C ***************************************************************************************\

 $Header$

 REVISION HISTORY
   02/99   Charles Cavanaugh
   05/00   Charles Cavanaugh
   06/00   Charles Cavanaugh
   09/01   Debbie Mao
   01/02   Debbie Mao
   02/02   Debbie Mao
   06/02   Debbie Mao
   08/02   Debbie Mao

 $Log$

\**********************************************************************************************************************/

#include <iomanip>
#include <math.h>
#include "ScienceLMCChannel.h"
#include "DiagnosticReporter.h"
#include "ScienceGlobalSectorMeans.h"

int const science_lmc_channel::LMC_SECTOR_SIZE = 48;
int const science_lmc_channel::MAX_DELTA_COUNTS = pow(2.0, 16.0);

extern diagnostic_reporter diagnosticreporter;
extern science_global_sector_means globalsectormeans;

science_lmc_channel :: science_lmc_channel ()
                     : science_channel ()
                     , science_lmc_signal ()
{
  ;
}


science_lmc_channel :: science_lmc_channel (int channelnumber)
                     : science_channel (channelnumber)
                     , science_lmc_signal ()
{
  ;
}


science_lmc_channel :: science_lmc_channel (int channelnumber, int mincount, int maxcount)
                     : science_channel (channelnumber, mincount, maxcount)
                     , science_lmc_signal ()
{
  ;
}


science_lmc_channel :: ~science_lmc_channel ()
{
  ;
}


void science_lmc_channel :: GetSignal (double signals [SCIENCE_PIXELS] [LMC_SAWTOOTH_SET],
                                       bool isvalid [SCIENCE_PIXELS]) const
{
  int sector, rotation, sawtooth;

  // for each pixel
  for (int pixel = 0; pixel < SCIENCE_PIXELS; pixel++) {

    // if any detectors are out of range, set the isvalid flag to false
    if (! PixelWithinRange (pixel))
      isvalid [pixel] = false;
    else {

      // set flag to valid
      isvalid [pixel] = true;

      // for each sector and rotation 
      sawtooth = 0;
      for ( rotation = 0; rotation < LMC_ROTATIONS; rotation++ )
	for (sector = 0; sector < LMC_SECTORS; sector++) 
	  signals [pixel][sawtooth++] = opencounts[pixel][sector][rotation]-closedcounts[pixel][sector];
    }
  }
}


void science_lmc_channel :: GetMeans (double openmeans [SCIENCE_PIXELS] [LMC_SECTORS], 
                                      double closedmeans [SCIENCE_PIXELS] [LMC_SECTORS],
                                      bool isvalid [SCIENCE_PIXELS]) const
{
  for (int pixel = 0; pixel < SCIENCE_PIXELS; pixel++) {

    // if any detectors are out of range, set the isvalid flag to false
    if (! PixelWithinRange (pixel))
      isvalid [pixel] = false;
    else {

      // set flag to valid
      isvalid [pixel] = true;

      // for each sector
      for (int sector = 0; sector < LMC_SECTORS; sector++) {

        // generate the open count at the midpoint of the 4 sector rotation values
        ComputeMidpointSignal (opencounts [pixel] [sector], openmeans [pixel] [sector]);

        // copy the closed counts
        closedmeans [pixel] [sector] = closedcounts [pixel] [sector];
      }
    }
  }
}


void science_lmc_channel :: GetOffset ( int delta, int sig0, int cn[13], int sig[13] )
{
  int offset, delcn, i, j;

  sig[cn[0]] = sig0;

  for ( i = 1; i < 13; i ++ ) {
    offset = 0;
    delcn = sig[cn[i]] - sig[cn[i-1]];
    if ( abs(abs(delcn) + MAX_DELTA_COUNTS - delta ) < abs(abs(delcn) - delta ) ){
      if ( delcn < 0 )
	offset = -MAX_DELTA_COUNTS;
      else
	offset = MAX_DELTA_COUNTS;
    }
    if ( abs(abs(delcn) - MAX_DELTA_COUNTS - delta ) < abs(abs(delcn) - delta ) ) {
      if ( delcn < 0 )
	offset = MAX_DELTA_COUNTS;
      else 
	offset = -MAX_DELTA_COUNTS;
    }
    if ( offset != 0 )
      for ( j = i; j < 13; j++ )
	sig[cn[j]] += offset; 
  }
}

void science_lmc_channel :: Read (bool active, int const pixelmaps [SCIENCE_PIXELS], unsigned char* traindata,
                                  short chopperopensamples, short chopperclosedsamples, short scalefactor,
                                  short choppercycles, processor_parameters const& processorparameters)
{
  int sumA, sumB, sumC, sumD;
  unsigned char* pixelptr;

  int laststate, rot0delcn, sig[13];
  int ind0[13] = { 0, 2, 3, 4, 1, 6, 7, 8, 5, 10, 11, 12, 9 };
  int ind1[13] = { 0, 3, 4, 1, 2, 7, 8, 5, 6, 11, 12, 9, 10 };
  int ind2[13] = { 0, 4, 1, 2, 3, 8, 5, 6, 7, 12, 9, 10, 11 };
  int ind3[13] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12 };

  // read the header info
  ReadHeader (active, traindata);
  GetLastcellState (laststate);

  // if the channel is not active, output message and set range flags to false
  if (! active) {
    InvalidateRange ();
    diagnosticreporter.AddEntry (DIAGNOSTICS_STATIC, DIAGNOSTICS_SCIENCE_MODULE, ACTIVE_STATIC, 0, 0, 0, 0, 0, channel,
                                 0, 0, "Inactive channel");
  }
  else {

    // for each pixel
    for (int pixel = 0; pixel < SCIENCE_PIXELS; pixel++) {

      // point to the beginning of the pixel's data
      pixelptr = traindata + CHANNEL_HEADER_SIZE + (pixel * LMC_SECTOR_SIZE);

      // unload into the correct spot
      int l1pixel = GetL1Pixel (pixel, pixelmaps);

      if ( (channel == 1 || channel == 2 || channel == 4) && (processorparameters.GetCh7Info() ) == 1 )
        for ( int sector = 0; sector < LMC_SECTORS; sector++)
	  for (int rotation = 0; rotation < LMC_ROTATIONS; rotation++) {
	    opencounts   [pixel] [sector] [rotation] = 0;
	    closedcounts [pixel] [sector]            = 0;
	  }
      else {

	// unpack the data.  remember to byte-swap
	sumA = (pixelptr [ 1] * 65536) + (pixelptr [ 0] * 256) + pixelptr [ 3];
	sumB = (pixelptr [ 8] * 65536) + (pixelptr [11] * 256) + pixelptr [10];
	sumC = (pixelptr [19] * 65536) + (pixelptr [18] * 256) + pixelptr [21];
	sumD = (pixelptr [26] * 65536) + (pixelptr [29] * 256) + pixelptr [28];

	opencounts [l1pixel] [LMC_A_SECTOR] [0] = CheckAndNormalize (l1pixel, chopperopensamples, scalefactor, 
								     choppercycles, ConvertOpenCount (sumA));
	opencounts [l1pixel] [LMC_B_SECTOR] [0] = CheckAndNormalize (l1pixel, chopperopensamples, scalefactor, 
								     choppercycles, ConvertOpenCount (sumB));
	opencounts [l1pixel] [LMC_C_SECTOR] [0] = CheckAndNormalize (l1pixel, chopperopensamples, scalefactor, 
								     choppercycles, ConvertOpenCount (sumC));
	opencounts [l1pixel] [LMC_D_SECTOR] [0] = CheckAndNormalize (l1pixel, chopperopensamples, scalefactor, 
								     choppercycles, ConvertOpenCount (sumD));
	sig[ 1] = (int) ((short)(pixelptr [ 2] * 256 + pixelptr [ 5])) + sumA; 
	sig[ 2] = (int) ((short)(pixelptr [13] * 256 + pixelptr [12])) + sumB; 
	sig[ 3] = (int) ((short)(pixelptr [20] * 256 + pixelptr [23])) + sumC; 
	sig[ 4] = (int) ((short)(pixelptr [31] * 256 + pixelptr [30])) + sumD; 
	sig[ 5] = (int) ((short)(pixelptr [ 4] * 256 + pixelptr [ 7])) + sumA; 
	sig[ 6] = (int) ((short)(pixelptr [15] * 256 + pixelptr [14])) + sumB; 
	sig[ 7] = (int) ((short)(pixelptr [22] * 256 + pixelptr [25])) + sumC; 
	sig[ 8] = (int) ((short)(pixelptr [33] * 256 + pixelptr [32])) + sumD; 
	sig[ 9] = (int) ((short)(pixelptr [ 6] * 256 + pixelptr [ 9])) + sumA; 
	sig[10] = (int) ((short)(pixelptr [17] * 256 + pixelptr [16])) + sumB; 
	sig[11] = (int) ((short)(pixelptr [24] * 256 + pixelptr [27])) + sumC; 
	sig[12] = (int) ((short)(pixelptr [35] * 256 + pixelptr [34])) + sumD;

	switch ( laststate) {
        case 0:  rot0delcn = ( abs(sumC - sumB) + abs( sumD-sumC) + abs(sumA-sumD) )/3.0;
	         GetOffset ( rot0delcn, sumA, ind0, sig );             
	         break;
        case 1:  rot0delcn = ( abs(sumD - sumC) + abs( sumA-sumD) + abs(sumB-sumA) )/3.0;
	         GetOffset ( rot0delcn, sumB, ind1, sig );
		 break;
        case 2:  rot0delcn = ( abs(sumA - sumD) + abs( sumB-sumA) + abs(sumC-sumB) )/3.0;
	         GetOffset ( rot0delcn, sumC, ind2, sig );
		 break;
        case 3:  rot0delcn = ( abs(sumB - sumA) + abs( sumC-sumB) + abs(sumD-sumC) )/3.0;
	         GetOffset ( rot0delcn, sumD, ind3, sig );
		 break;
        default: diagnosticreporter.AddEntry (DIAGNOSTICS_STATIC, DIAGNOSTICS_SCIENCE_MODULE, ACTIVE_STATIC, 
                            0, 0, 0, 0, 0, channel,0, 0, "wrong last cell state.");    
		 break;
	}

	opencounts [l1pixel] [LMC_A_SECTOR] [1] = CheckAndNormalize 
	                              (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[ 1]));
	opencounts [l1pixel] [LMC_B_SECTOR] [1] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[ 2]));
	opencounts [l1pixel] [LMC_C_SECTOR] [1] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[ 3]));
	opencounts [l1pixel] [LMC_D_SECTOR] [1] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[ 4]));
	opencounts [l1pixel] [LMC_A_SECTOR] [2] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[ 5]));
	opencounts [l1pixel] [LMC_B_SECTOR] [2] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[ 6]));
	opencounts [l1pixel] [LMC_C_SECTOR] [2] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[ 7]));
	opencounts [l1pixel] [LMC_D_SECTOR] [2] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[ 8]));
	opencounts [l1pixel] [LMC_A_SECTOR] [3] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[ 9]));
	opencounts [l1pixel] [LMC_B_SECTOR] [3] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[10]));
	opencounts [l1pixel] [LMC_C_SECTOR] [3] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[11]));
	opencounts [l1pixel] [LMC_D_SECTOR] [3] = CheckAndNormalize 
                                      (l1pixel, chopperopensamples, scalefactor, choppercycles, ConvertOpenCount(sig[12]));
	
	closedcounts [l1pixel] [LMC_A_SECTOR] = 
                                          Normalize (chopperclosedsamples, scalefactor, choppercycles,
                                                     ((pixelptr [37] * 65536) + (pixelptr [36] * 256) + pixelptr [39]));
	closedcounts [l1pixel] [LMC_B_SECTOR] = 
                                          Normalize (chopperclosedsamples, scalefactor, choppercycles,
                                                     ((pixelptr [38] * 65536) + (pixelptr [41] * 256) + pixelptr [40]));
	closedcounts [l1pixel] [LMC_C_SECTOR] = 
                                          Normalize (chopperclosedsamples, scalefactor, choppercycles,
                                                     ((pixelptr [43] * 65536) + (pixelptr [42] * 256) + pixelptr [45]));
	closedcounts [l1pixel] [LMC_D_SECTOR] = 
                                          Normalize (chopperclosedsamples, scalefactor, choppercycles,
                                                     ((pixelptr [44] * 65536) + (pixelptr [47] * 256) + pixelptr [46]));
 
	// Set chopper closecounts = 0 becase of the chopper pbm
	if ( (channel == 5 || channel == 6) && (processorparameters.GetChopper() ) == 0 ) {
	  closedcounts [l1pixel] [LMC_A_SECTOR] = 0.0;
	  closedcounts [l1pixel] [LMC_B_SECTOR] = 0.0;
	  closedcounts [l1pixel] [LMC_C_SECTOR] = 0.0;
	  closedcounts [l1pixel] [LMC_D_SECTOR] = 0.0;
	}
      }
    }
  }
}
