/********** ScienceCollection.C ***************************************************************************************\

 $Header$

 REVISION HISTORY
   12/98   Charles Cavanaugh
   09/00   Daniel Ziskin
   07/02   Debbie Mao

 $Log$

\**********************************************************************************************************************/

#include "ScienceCollection.h"
#include "DiagnosticReporter.h"

int const science_collection::APID                 = 192;
int const science_collection::PACKETSIZE           = 6616;
int const science_collection::PACKETLENGTH         = PACKETSIZE - CCSDS_HEADER_LENGTH;
int const science_collection::APID_ERROR           = 902;
int const science_collection::LENGTH_ERROR         = 903;
int const science_collection::MONOTONICITY_WARNING = 804;
int const science_collection::ID_WARNING           = 805;

extern diagnostic_reporter diagnosticreporter;

science_collection :: science_collection (table_collection& table, processor_parameters const& parameters)
                    : l0_collection (SCIENCE_COLLECTION_01_LOGICAL, PACKETSIZE)
                    , tablecollection (table)
                    , processorparameters (parameters)
                    , maxstaretime ()
{
  ;
}


science_collection :: ~science_collection ()
{
  ;
}


void science_collection :: CheckMonotonicity (mopitt_time const& staretime, int packetnumber, int starenumber)
{
  if (staretime > maxstaretime) 
    maxstaretime = staretime;
  else
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SCIENCE_MODULE, MONOTONICITY_WARNING, 
                                 GetCurrentLogical (), 0, packetnumber + 1, starenumber, 0, 0, 0, 0, 
                                 "Stare time not greater than maximum encountered stare time");
}


science_packet* science_collection :: NextSciencePacket ()
{
  science_packet* sciencepacket = NULL;
  PGSt_IO_L0_Packet* data;
  static int oldsequence = -1;

  // while there are packets and we haven't yet successfully retrieved one
  bool findanotherpacket = true;
  while (findanotherpacket) {

    // make room for the packet data and then read the packet; if failure, delete the packet data container
    data = new PGSt_IO_L0_Packet [PACKETSIZE];
    if (GetPacket (data) == NULL) {
      delete [] data;
      findanotherpacket = false;
    }

    // packet read was successful, so now make a new science packet and then check the apid and length, 
    // and reject the packet if either is wrong
    else {
      sciencepacket = new science_packet (data);
      int sequence = sciencepacket->GetSequence ();

      if (sciencepacket->GetAPID () != APID) {
        string message = sciencepacket->MakeWrongAPIDMessage ();
        delete sciencepacket;
        sciencepacket = NULL;
        diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_SCIENCE_MODULE, APID_ERROR, GetCurrentLogical (), 
                                     0, sequence, 0, 0, 0, 0, 0, message);
      }
      else if (sciencepacket->GetLength () != PACKETLENGTH) {
        string message = sciencepacket->MakeWrongLengthMessage ();
        delete sciencepacket;
        sciencepacket = NULL;
        diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SCIENCE_MODULE, LENGTH_ERROR, GetCurrentLogical (),
                                     0, sequence, 0, 0, 0, 0, 0, message);
      }
      else {
        findanotherpacket = false;
        if ((sciencepacket->SequenceGap (oldsequence, sequence)) && (oldsequence >= 0)) {
          string message = sciencepacket->MakeSequenceGapMessage(oldsequence, sequence);
          diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SCIENCE_MODULE,
                                       sciencepacket->GetSequenceErrCode (), GetCurrentLogical (), 0, sequence, 0, 0, 
                                       0, 0, 0, message);
        }
        oldsequence = sequence;
      }
    }
  }

  return sciencepacket;
}


science_stare* science_collection :: NextScienceStare ()
{
  science_stare* sciencestare = NULL;
  static science_packet* sciencepacket = NULL;
  static bool endofpacket = true;
  
  // while there are stares and we haven't yet successfully retrieved one
  bool findanotherstare = true;
  while (findanotherstare) {

    // if there are no more stares in the current packet
    if (endofpacket) { 

      // delete the old packet and retrieve a new packet;  if successful, set the "at end of packet" flag to false;
      // else we're through getting science stares
      delete sciencepacket;
      if ((sciencepacket = NextSciencePacket ()) != NULL)
        endofpacket = false;
      else
        findanotherstare = false;
    }

    // else there are still stares in the current packet
    else {
 
      // retrieve the next stare in the packet
      if ((sciencestare = sciencepacket->GetNextStare ()) != NULL) {

        // if the stare does not have the correct id, reject the stare
        if (sciencestare->HasCorrectID ()) {
          findanotherstare = false;
          CheckMonotonicity (sciencestare->GetTime (), sciencepacket->GetSequence(), sciencestare->GetPacketPosition ());
        }
        else {
          string message = "A science stare has an incorrect stare id : ";
          message += sciencestare->GetID ();
          diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SCIENCE_MODULE, ID_WARNING, 
                                       GetCurrentLogical (), 0, sciencepacket->GetSequence (), 
                                       sciencestare->GetPacketPosition (), 0, 0, 0, 0, message);
          delete sciencestare;
          sciencestare = NULL;
        }
      }

      // query the packet to see if it has more stares; if it does not, set the "at end of packet" flag to true 
      if (! sciencepacket->HasMoreStares ())
        endofpacket = true;
    }
  }

  return sciencestare;
}


science_train* science_collection :: GetTrain ()
{
  science_train* sciencetrain = NULL;
  static science_stare* sciencestare = NULL;
  static bool endofstare = true;

  // while there are trains and we haven't yet successfully retrieved one
  bool findanothertrain = true;
  while (findanothertrain) {

    // if there are no more trains in the current stare
    if (endofstare) {

      // delete the old stare and retrieve a new stare; if a stare is successfully retrieved, set the "at end of stare"
      // flag to false; else we're through getting science trains
      delete sciencestare;
      if ((sciencestare = NextScienceStare ()) != NULL)
        endofstare = false;
      else
        findanothertrain = false;
    }

    // else there are still trains in the current stare
    else {

      // retrieve the next train in the stare; if successful, set code to exit this routine
      if ((sciencetrain = sciencestare->GetNextTrain (tablecollection,processorparameters)) != NULL)
        findanothertrain = false;
 
      // query the stare to see if it has more trains; if it does not, set the "at end of stare" flag to true 
      if (! sciencestare->HasMoreTrains ())
        endofstare = true;
    }
  }

  return sciencetrain;
}
