/********** MOPIRTable.C **********************************************************************************************\

 $Header$

 REVISION HISTORY
   12/98   Charles Cavanaugh
   04/99   Tom Lauren
   09/99   Charles Cavanaugh
   04/00   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include <memory.h>
#include "MOPIRTable.h"
#include "DiagnosticReporter.h"

int const mopir_table::PARSE_ERROR          = 902;
int const mopir_table::OUT_OF_RANGE_WARNING = 800;
int const mopir_table::NOT_IN_RANGE_WARNING = 801;
int const mopir_table::RETRIEVE_WARNING     = 803;

extern diagnostic_reporter diagnosticreporter;

mopir_table :: mopir_table ()
{
  ;
}


mopir_table :: ~mopir_table ()
{
  ;
}


bool mopir_table :: BuildChannelCode (int channel, enum mopir_radiance_type radtype, 
                                      enum mopir_channel_code& channelcode) const
{
  bool isbuilt = true;

  switch (channel) {
    case 1 : channelcode = (radtype == RADIANCE_AVERAGE ? CHANNEL1_AVERAGE : CHANNEL1_DIFFERENCE); 
             break;
    case 2 : channelcode = (radtype == RADIANCE_AVERAGE ? CHANNEL2_AVERAGE : CHANNEL2_DIFFERENCE); 
             break;
    case 3 : channelcode = (radtype == RADIANCE_AVERAGE ? CHANNEL3_AVERAGE : CHANNEL3_DIFFERENCE); 
             break;
    case 4 : channelcode = (radtype == RADIANCE_AVERAGE ? CHANNEL4_AVERAGE : CHANNEL4_DIFFERENCE); 
             break;
    case 5 : channelcode = (radtype == RADIANCE_AVERAGE ? CHANNEL5_AVERAGE : CHANNEL5_DIFFERENCE); 
             break;
    case 6 : channelcode = (radtype == RADIANCE_AVERAGE ? CHANNEL6_AVERAGE : CHANNEL6_DIFFERENCE); 
             break;
    case 7 : channelcode = (radtype == RADIANCE_AVERAGE ? CHANNEL7_AVERAGE : CHANNEL7_DIFFERENCE); 
             break;
    case 8 : channelcode = (radtype == RADIANCE_AVERAGE ? CHANNEL8_AVERAGE : CHANNEL8_DIFFERENCE); 
             break;
    default : isbuilt = false;
  }

  return isbuilt;
}


bool mopir_table :: GetTemperatureIndex (float temp, int& index) const
{
  bool isexact = false;

  // Convert the temperature to its nearest integer.
  int converted_temp = (int)(temp + 0.5);

  // If the temperature is below all of the temps in the list, give the lowest index.
  if (converted_temp < temps[0]) {
    char tempchars [10];
    (void) sprintf (tempchars, "%d", converted_temp);
    string message = "Temperature (";
    message += tempchars;
    message += ") is out of table range";
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_MOPIR_MODULE, OUT_OF_RANGE_WARNING, 0, 0, 0, 0, 0, 0,
                                 0, 0, message);
    index = 0;
  }

  // If the temperature is above all of the temps in the list, give the highest index.
  else if (converted_temp > temps[MOPIR_RADS-1]) {
    char tempchars [10];
    (void) sprintf (tempchars, "%d", converted_temp);
    string message = "Temperature (";
    message += tempchars;
    message += ") is out of table range";
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_MOPIR_MODULE, OUT_OF_RANGE_WARNING, 0, 0, 0, 0, 0, 0,
                                 0, 0, message);
    index = MOPIR_RADS - 1;
  }

  // Otherwise return the index of the corresponding temperature in the list.
  else  {
    int tempindex = 0;
    bool keeplooking = true;
    while (keeplooking && tempindex < MOPIR_RADS)
      if (converted_temp == temps[tempindex])
        keeplooking = false;
      else
        tempindex++;

    if (! keeplooking) {
      index = tempindex;
      isexact = true;
    }
    else {
      char tempchars [10];
      (void) sprintf (tempchars, "%d", converted_temp);
      string message = "Temperature (";
      message += tempchars;
      message += ") was not found in table";
      diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_MOPIR_MODULE, NOT_IN_RANGE_WARNING, 0, 0, 0, 0, 0,
                                   0, 0, 0, message);
      index = MOPIR_RADS / 2;
    }
  }

  return isexact;
}
  

bool mopir_table :: Parse (string const& mopirline, enum mopir_table_part part, enum mopir_channel_code channel)
{
  bool isparsed = true;

  // Parses a line of text (taken from an input file) and stores the data value.
  char *token;
  int i, tokenskips;

  // make a copy of the line
  char* linecopy = new char [mopirline.length () + 1];
  (void) strcpy (linecopy, mopirline.c_str ());

  // get the radiance index (first entry on the line
  int radindex = atoi (strtok (linecopy, " ")) - 1;

  if (part == TEMPERATURE_PART)
    tokenskips = 0;
  else
    tokenskips = 1;

  // skip over some tokens
  for (i = tokenskips; i > 0; i--)
    token = strtok (NULL, " \t");

  // read in the radiance value
  token = strtok (NULL, " \n");

  if (token == NULL) {
    isparsed = false;
    char channelchars [10];
    (void) sprintf (channelchars, "%d", (int) channel);
    string message = "Could not successfully parse line for channel code ";
    message += channelchars;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_MOPIR_MODULE, PARSE_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }
  else {
    if (part == TEMPERATURE_PART)
      temps[radindex] = (float) atof (token);
    else
      rads[channel][radindex] = (float) atof (token);
  }

  delete [] linecopy;

  return isparsed;
}


bool mopir_table :: GetRadiance (float temp, int channel, enum mopir_radiance_type radtype, float& radiance) const
{
  bool isretrieved = false;
  enum mopir_channel_code channelcode;
  int temperatureindex;

  if (! BuildChannelCode (channel, radtype, channelcode))
    diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_MOPIR_MODULE, RETRIEVE_WARNING, 0, 0, 0, 0, 0, 
                                 channel, 0, 0, "Could not retrieve internal radiance");
  else {
    float floortemp = floor (temp);
    GetTemperatureIndex ((float) floor (temp), temperatureindex);
    float floorradiance = rads [channelcode][temperatureindex];
    GetTemperatureIndex ((float) ceil (temp), temperatureindex);
    float ceilradiance = rads [channelcode][temperatureindex];
    radiance = floorradiance + ((ceilradiance - floorradiance) * (temp - floortemp));
    isretrieved = true;
  }

  return isretrieved;
}
