/********** ThermistorStandardParameter.C *****************************************************************************\

 $Header$

 REVISION HISTORY
   03/99   Charles Cavanaugh
   10/99   Charles Cavanaugh
   02/00   Tom Lauren
   05/00   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "ThermistorStandardParameter.h"

int const thermistor_standard_parameter::PARAMETER_COUNT = 7;

thermistor_standard_parameter :: thermistor_standard_parameter ()
                               : mopip_parameter ()
{
  bank = 0;
  rB = 0.0;
  a = 0.0;
  b = 0.0;
  c = 0.0;
  low = 0;
  high = 0;
}


thermistor_standard_parameter :: ~thermistor_standard_parameter ()
{
  ;
}


void thermistor_standard_parameter :: Compute (float sensor [COMPUTE_TERMS], float hiref [COMPUTE_TERMS], 
                                               float loref [COMPUTE_TERMS], float average [COMPUTE_TERMS]) const
{
  // convert the readings to resistance and generate the variance of the resistance
  double rT = rB * ((float) (sensor [COMPUTE_TERM_AVERAGE] - loref [COMPUTE_TERM_AVERAGE]) / 
                    (float) (hiref [COMPUTE_TERM_AVERAGE] - sensor [COMPUTE_TERM_AVERAGE]));
  double lorefderiv = -rB / (hiref [COMPUTE_TERM_AVERAGE] - sensor [COMPUTE_TERM_AVERAGE]);
  double rDvariance = (pow (lorefderiv, 2.0) * loref [COMPUTE_TERM_VARIANCE]) +
                      (pow ((lorefderiv * (sensor [COMPUTE_TERM_AVERAGE] - loref [COMPUTE_TERM_AVERAGE]) / 
                                           (hiref [COMPUTE_TERM_AVERAGE] - sensor [COMPUTE_TERM_AVERAGE])), 2.0) * 
                      hiref [COMPUTE_TERM_VARIANCE]) +
                      (pow ((lorefderiv * (-1.0 - ((sensor [COMPUTE_TERM_AVERAGE] - loref [COMPUTE_TERM_AVERAGE]) / 
                                              (hiref [COMPUTE_TERM_AVERAGE] - sensor [COMPUTE_TERM_AVERAGE])))), 2.0) *
                      sensor [COMPUTE_TERM_VARIANCE]);

  // convert to degrees Kelvin and generate the standard deviation
  double denom = a + (log (rT) * (b + (c * log (rT) * log (rT))));
  average [COMPUTE_TERM_AVERAGE] = 1.0 / denom;
  double variance = ((3 * c * log (rT) * log (rT)) + b) / (rT * pow (denom, 2.0));
  average [COMPUTE_TERM_VARIANCE] = variance * variance * rDvariance;
}


void thermistor_standard_parameter :: GetRange (float& highrange, float& lowrange) const
{
  highrange = high;
  lowrange = low;
}


bool thermistor_standard_parameter :: Parse (string const& mopipline)
{
  bool isparsed = false;

  // make a copy of the line
  char* linecopy = new char [mopipline.length () + 1];
  (void) strcpy (linecopy, mopipline.c_str ());

  // skip over the value type
  char* token = strtok (linecopy, " ");

  // set the enumerator, based on the sensor name
  string name = strtok (NULL, " ");
  if      (name == "LMC1")          enumerator = LMCTemperature1;
  else if (name == "LMC2")          enumerator = LMCTemperature2;
  else if (name == "LMC3")          enumerator = LMCTemperature3;
  else if (name == "LMC4")          enumerator = LMCTemperature4;
  else if (name == "PMC1")          enumerator = PMCTemperature1;
  else if (name == "PMC2")          enumerator = PMCTemperature2;
  else if (name == "Chopper1")      enumerator = ChopperTemperature1;
  else if (name == "Chopper2")      enumerator = ChopperTemperature2;
  else if (name == "Chopper3")      enumerator = ChopperTemperature3;
  else if (name == "Chopper4")      enumerator = ChopperTemperature4;
  else if (name == "MirrorMotor1")  enumerator = MirrorMotorTemperature1;
  else if (name == "MirrorMotor2")  enumerator = MirrorMotorTemperature2;
  else if (name == "MirrorMotor3")  enumerator = MirrorMotorTemperature3;
  else if (name == "MirrorMotor4")  enumerator = MirrorMotorTemperature4;
  else if (name == "Table1-1")      enumerator = TableTemperature1_1;
  else if (name == "Table1-2")      enumerator = TableTemperature1_2;
  else if (name == "Table1-3")      enumerator = TableTemperature1_3;
  else if (name == "Table1-4")      enumerator = TableTemperature1_4;
  else if (name == "Table2-1")      enumerator = TableTemperature2_1;
  else if (name == "Table2-2")      enumerator = TableTemperature2_2;
  else if (name == "Table2-3")      enumerator = TableTemperature2_3;
  else if (name == "Table2-4")      enumerator = TableTemperature2_4;
  else if (name == "SPM1")          enumerator = SPMTemperature1;
  else if (name == "SPM2")          enumerator = SPMTemperature2;
  else if (name == "Baseplate1")    enumerator = BaseplateTemperature1;
  else if (name == "Baseplate2")    enumerator = BaseplateTemperature2;
  else if (name == "Baseplate3")    enumerator = BaseplateTemperature3;
  else if (name == "PSM")           enumerator = PSMTemperature;
  else if (name == "TCMMA-1")       enumerator = TCMMTemperatureA1;
  else if (name == "TCMMA-2")       enumerator = TCMMTemperatureA2;
  else if (name == "TCMMB-1")       enumerator = TCMMTemperatureB1;
  else if (name == "TCMMB-2")       enumerator = TCMMTemperatureB2;
  else if (name == "Compressor1")   enumerator = CompressorTemperature1;
  else if (name == "Compressor2")   enumerator = CompressorTemperature2;
  else ReportNameError ("ThermistorStandard", name);

  // read in the values
  int valuecount = 0;
  while ((token = strtok (NULL, " =,[]\n")) != NULL && valuecount < PARAMETER_COUNT) {
    switch (valuecount) {
      case 0 : bank = (short) atoi (token);
               break;
      case 1 : rB = atof (token);
               break;
      case 2 : a = atof (token);
               break;
      case 3 : b = atof (token);
               break;
      case 4 : c = atof (token);
               break;
      case 5 : low = (float) atof (token);
               break;
      case 6 : high = (float) atof (token);
               break;
      default : break;
    }

    valuecount++;
  }

  delete [] linecopy;

  if (valuecount == PARAMETER_COUNT)
    isparsed = true;
  else
    ReportValuesError ("ThermistorStandard", name);
  
  return isparsed;
}
