/********** ThermistorPCDMParameter.C *********************************************************************************\

 $Header$

 REVISION HISTORY
   03/99   Charles Cavanaugh
   10/99   Charles Cavanaugh
   05/00   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "ThermistorPCDMParameter.h"

int const thermistor_pcdm_parameter::PARAMETER_COUNT= 7;

thermistor_pcdm_parameter :: thermistor_pcdm_parameter ()
                           : mopip_parameter ()
{
  gain = 0.0;
  offset = 0.0;
  a = 0.0;
  b = 0.0;
  c = 0.0;
  low = 0;
  high = 0;
}


thermistor_pcdm_parameter :: ~thermistor_pcdm_parameter ()
{
  ;
}


void thermistor_pcdm_parameter :: Compute (float sensor [COMPUTE_TERMS], float hiref [COMPUTE_TERMS],
                                           float loref [COMPUTE_TERMS], float average [COMPUTE_TERMS]) const
{
  // convert the readings to resistance and generate their variances
  double vT = (sensor [COMPUTE_TERM_AVERAGE] * gain) + offset;
  double vTvariance = (gain * gain * sensor [COMPUTE_TERM_VARIANCE]);
  double vH = (hiref  [COMPUTE_TERM_AVERAGE] * gain) + offset;
  double vHvariance = (gain * gain * hiref [COMPUTE_TERM_VARIANCE]);
  double vL = (loref  [COMPUTE_TERM_AVERAGE] * gain) + offset;
  double vLvariance = (gain * gain * loref [COMPUTE_TERM_VARIANCE]);

  double rT1 = ((((vH * 2.5988) - vT) + (2 * vL)) - 1.0299) / 20000.0;
  double rT1variance = (pow ((2.5988 / 20000.0), 2.0) * vHvariance) +
                       (pow ((   1.0 / 20000.0), 2.0) * vTvariance) +
                       (pow ((   2.0 / 20000.0), 2.0) * vLvariance);

  double rT2 = ((1.72 - vT) + vL) / 16700.0;
  double rT2variance = (pow ((1.0 / 16700.0), 2.0) * vTvariance) + (pow ((1.0 / 16700.0), 2.0) * vLvariance);

  double rT = (vT - vL) / (rT1 + rT2);
  double vderiv = 1.0 / (rT1 + rT2);
  double rTderiv = (vL - vT) / pow ((rT1 + rT2), 2.0);
  double rTvariance = (pow (vderiv, 2.0) * vTvariance) + (pow (vderiv, 2.0) * vLvariance) +
                      (pow (rTderiv, 2.0) * rT1variance) + (pow (rTderiv, 2.0) * rT2variance);

  // convert to degrees Kelvin
  double denom = a + (log (rT) * (b + (c * log (rT) * log (rT))));
  average [COMPUTE_TERM_AVERAGE] = 1.0 / denom;
  double variance = ((3 * c * log (rT) * log (rT)) - b) / (rT * pow (denom, 2.0));
  average [COMPUTE_TERM_VARIANCE] = variance * variance * rTvariance;
}


void thermistor_pcdm_parameter :: GetRange (float& highrange, float& lowrange) const
{
  highrange = high;
  lowrange = low;
}


bool thermistor_pcdm_parameter :: Parse (string const& mopipline)
{
  bool isparsed = false;

  // make a copy of the line
  char* linecopy = new char [mopipline.length () + 1];
  (void) strcpy (linecopy, mopipline.c_str ());

  // skip over the value type
  char* token = strtok (linecopy, " ");

  // set the enumerator, based on the sensor name
  string name = strtok (NULL, " ");
  if      (name == "EarthBaffel") enumerator = EarthPortBaffelTemperature;
  else if (name == "SpaceBaffel") enumerator = SpacePortBaffelTemperature;
  else ReportNameError ("ThermistorPCDM", name);

  // read in the values
  int valuecount = 0;
  while ((token = strtok (NULL, " =,[]\n")) != NULL && valuecount < PARAMETER_COUNT) {
    switch (valuecount) {
      case 0 : gain = atof (token);
               break;
      case 1 : offset = atof (token);
               break;
      case 2 : a = atof (token);
               break;
      case 3 : b = atof (token);
               break;
      case 4 : c = atof (token);
               break;
      case 5 : low = (float) atof (token);
               break;
      case 6 : high = (float) atof (token);
               break;
      default : break;
    }

    valuecount++;
  }

  delete [] linecopy;

  if (valuecount == PARAMETER_COUNT)
    isparsed = true;
  else
    ReportValuesError ("ThermistorPCDM", name);

  return isparsed;
}
