/********** PRTSieveParameter.C ***************************************************************************************\

 $Header$

 REVISION HISTORY
   03/99   Charles Cavanaugh
   10/99   Charles Cavanaugh
   05/00   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "PRTSieveParameter.h"

int const prt_sieve_parameter::PARAMETER_COUNT = 8;

prt_sieve_parameter :: prt_sieve_parameter ()
                     : mopip_parameter ()
{
  bits = 0;
  r2 = 0.0;
  r3 = 0.0;
  r0 = 0.0;
  alpha = 0.0;
  delta = 0.0;
  low = 0;
  high = 0;
}


prt_sieve_parameter :: ~prt_sieve_parameter ()
{
  ;
}


void prt_sieve_parameter :: Compute (float sensor [COMPUTE_TERMS], float hiref [COMPUTE_TERMS], 
                                     float loref [COMPUTE_TERMS], float average [COMPUTE_TERMS]) const
{
  // do a bit conversion on the sensor, hi ref and lo ref
  float newsensor, newhiref, newloref;
  BitConvert (bits, sensor [COMPUTE_TERM_AVERAGE], hiref [COMPUTE_TERM_AVERAGE], loref [COMPUTE_TERM_AVERAGE], 
              newsensor, newhiref, newloref);

  // convert the readings to resistance and generate the variance of the resistance
  double rT = r3 + ((r2 * (newsensor - newloref)) / (newhiref - newloref));
  double sensorderiv = r2 / (newhiref - newloref);
  double rDvariance = (pow (sensorderiv, 2.0) * sensor [COMPUTE_TERM_VARIANCE]) +
                      (pow ((sensorderiv * (newloref - newsensor) / (newhiref - newloref)), 2.0) *
                       hiref [COMPUTE_TERM_VARIANCE]) +
                      (pow ((sensorderiv * (-1.0 + ((newsensor - newloref) / (newhiref - newloref)))), 2.0) *
                       loref [COMPUTE_TERM_VARIANCE]);

  // generate some intermediate values
  double intermediate1 = .5 * ((10000.0 / delta) + 100.0);
  double intermediate2 = pow (((10000.0 / delta) + 100.0), 2.0);
  double intermediate3 = 4.0 * (10000.0 / (alpha * delta)) * (1.0 - (rT / r0));

  // convert to degrees Kelvin and generate the standard deviation
  average [COMPUTE_TERM_AVERAGE] = 273.15 + (intermediate1 - (.5 * sqrt ((intermediate2 + intermediate3))));
  double variance = pow ((10000.0 / (4.0 * alpha * delta * r0)), 2.0) / (intermediate2 + intermediate3);
  average [COMPUTE_TERM_VARIANCE] = variance * rDvariance;
}


void prt_sieve_parameter :: GetRange (float& highrange, float& lowrange) const
{
  highrange = high;
  lowrange = low;
}


bool prt_sieve_parameter :: Parse (string const& mopipline)
{
  bool isparsed = false;

  // make a copy of the line
  char* linecopy = new char [mopipline.length () + 1];
  (void) strcpy (linecopy, mopipline.c_str ());

  // skip over the value type
  char* token = strtok (linecopy, " ");

  // set the enumerator, based on the sensor name
  string name = strtok (NULL, " ");
  if      (name == "Sieve1") enumerator = SieveTemperature1;
  else if (name == "Sieve2") enumerator = SieveTemperature2;
  else if (name == "Sieve3") enumerator = SieveTemperature3;
  else if (name == "Sieve4") enumerator = SieveTemperature4;
  else ReportNameError ("PRTSieve", name);

  // read in the values
  int valuecount = 0;
  while ((token = strtok (NULL, " =,[]\n")) != NULL && valuecount < PARAMETER_COUNT) {
    switch (valuecount) {
      case 0 : bits = (short) atoi (token);
               break;
      case 1 : r2 = atof (token);
               break;
      case 2 : r3 = atof (token);
               break;
      case 3 : r0 = atof (token);
               break;
      case 4 : alpha = atof (token);
               break;
      case 5 : delta = atof (token);
               break;
      case 6 : low = (float) atof (token);
               break;
      case 7 : high = (float) atof (token);
               break;
      default : break;
    }

    valuecount++;
  }

  delete [] linecopy;

  if (valuecount == PARAMETER_COUNT)
    isparsed = true;
  else
    ReportValuesError ("PRTSieve", name);

  return isparsed;
}
