/********** PRTBlackbodyParameter.C ***********************************************************************************\

 $Header$

 REVISION HISTORY
   03/99   Charles Cavanaugh
   10/99   Charles Cavanaugh
   05/00   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "PRTBlackbodyParameter.h"

int const prt_blackbody_parameter::PARAMETER_COUNT = 10;

prt_blackbody_parameter :: prt_blackbody_parameter ()
                         : mopip_parameter ()
{
  bits = 0;
  r2 = 0.0;
  r3 = 0.0;
  k0 = 0.0;
  k1 = 0.0;
  k2 = 0.0;
  k3 = 0.0;
  k4 = 0.0;
  low = 0;
  high = 0;
}


prt_blackbody_parameter :: ~prt_blackbody_parameter ()
{
  ;
}


void prt_blackbody_parameter :: Compute (float sensor [COMPUTE_TERMS], float hiref [COMPUTE_TERMS], 
                                         float loref [COMPUTE_TERMS], float average [COMPUTE_TERMS]) const
{
  // do a bit conversion on the sensor, hi ref and lo ref
  float newsensor, newhiref, newloref;
  BitConvert (bits, sensor [COMPUTE_TERM_AVERAGE], hiref [COMPUTE_TERM_AVERAGE], loref [COMPUTE_TERM_AVERAGE], 
              newsensor, newhiref, newloref);

  // convert the readings to resistance and generate the variance of the resistance
  double rT = r3 + ((r2 * (newsensor - newloref)) / (newhiref - newloref));
  double sensorderiv = r2 / (newhiref - newloref);
  double rDvariance = (pow (sensorderiv, 2.0) * sensor [COMPUTE_TERM_VARIANCE]) +
                      (pow ((sensorderiv * (newloref - newsensor) / (newhiref - newloref)), 2.0) *
                       hiref [COMPUTE_TERM_VARIANCE]) +
                      (pow ((sensorderiv * (-1.0 + ((newsensor - newloref) / (newhiref - newloref)))), 2.0) *
                       loref [COMPUTE_TERM_VARIANCE]);

  // convert to degrees Kelvin and generate the variance
  double variance = k1 + (2 * k2 * rT) + (3 * k3 * rT * rT) + (4 * k4 * rT * rT * rT);
  average [COMPUTE_TERM_AVERAGE] = k0 + (rT * (k1 + (rT * (k2 + (rT * (k3 + (rT * k4)))))));
  average [COMPUTE_TERM_VARIANCE] = variance * variance * rDvariance;
}


void prt_blackbody_parameter :: GetRange (float& highrange, float& lowrange) const
{
  highrange = high;
  lowrange = low;
}


bool prt_blackbody_parameter :: Parse (string const& mopipline)
{  
  bool isparsed = false;

  // make a copy of the line
  char* linecopy = new char [mopipline.length () + 1];
  (void) strcpy (linecopy, mopipline.c_str ());

  // skip over the value type
  char* token = strtok (linecopy, " ");

  // set the enumerator, based on the sensor name
  string name = strtok (NULL, " ");
  if      (name == "BB1-1") enumerator = BlackbodyTemperature1_1;
  else if (name == "BB1-2") enumerator = BlackbodyTemperature1_2;
  else if (name == "BB2-1") enumerator = BlackbodyTemperature2_1;
  else if (name == "BB2-2") enumerator = BlackbodyTemperature2_2;
  else if (name == "BB3-1") enumerator = BlackbodyTemperature3_1;
  else if (name == "BB3-2") enumerator = BlackbodyTemperature3_2;
  else if (name == "BB4-1") enumerator = BlackbodyTemperature4_1;
  else if (name == "BB4-2") enumerator = BlackbodyTemperature4_2;
  else ReportNameError ("PRTBlackbody", name);

  // read in the values
  int valuecount = 0;
  while ((token = strtok (NULL, " =,[]\n")) != NULL && valuecount < PARAMETER_COUNT) {
    switch (valuecount) {
      case 0 : bits = (short) atoi (token);
               break;
      case 1 : r2 = atof (token);
               break;
      case 2 : r3 = atof (token);
               break;
      case 3 : k0 = atof (token);
               break;
      case 4 : k1 = atof (token);
               break;
      case 5 : k2 = atof (token);
               break;
      case 6 : k3 = atof (token);
               break;
      case 7 : k4 = atof (token);
               break;
      case 8 : low = (float) atof (token);
               break;
      case 9 : high = (float) atof (token);
               break;
      default : break;
    }

    valuecount++;
  }

  delete [] linecopy;

  if (valuecount == PARAMETER_COUNT)
    isparsed = true;
  else
    ReportValuesError ("PRTBlackbody", name);

  return isparsed;
}
