/********** MOPIPValues.C *********************************************************************************************\

 $Header$

 REVISION HISTORY
   12/98   Charles Cavanaugh
   10/99   Charles Cavanaugh
   05/00   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include <memory.h>
#include "MOPIPValues.h"
#include "PRTSieveParameter.h"
#include "PRTBlackbodyParameter.h"
#include "ThermistorStandardParameter.h"
#include "ThermistorPCDMParameter.h"
#include "ThermistorPressureTempParameter.h"
#include "ThermistorPressureParameter.h"
#include "DiodeDetectorParameter.h"
#include "DiodeColdFilterParameter.h"
#include "DiodeDetectorNestParameter.h"
#include "HeaterVoltageParameter.h"
#include "PMCPPOParameter.h"
#include "PMCPhaseErrorParameter.h"
#include "MotorCurrentParameter.h"
#include "CDEReadoutParameter.h"
#include "DiagnosticReporter.h"

int const mopip_values::PIXEL_ERROR    = 903;
int const mopip_values::TYPE_ERROR     = 904;
int const mopip_values::RETRIEVE_ERROR = 900;

extern diagnostic_reporter diagnosticreporter;

mopip_values :: mopip_values ()
{
  for (int i = 0; i < MOPIP_PARAMETER_COUNT; i++)
    parameters [i] = NULL;
}


mopip_values :: ~mopip_values ()
{
  // free all the parameters
  for (int i = 0; i < MOPIP_PARAMETER_COUNT; i++)
    delete parameters [i];
}


bool mopip_values :: AddPixelUV (string const& mopipline)
{
  bool isadded = false;

  // make a copy of the string
  char* linecopy = new char [mopipline.length () + 1];
  (void) strcpy (linecopy, mopipline.c_str ());

  // skip over the "PixelUV" and "Stare" tokens
  // get the stare (the third token)
  char* token = strtok (linecopy, " ");
  token = strtok (NULL, " ");

  // get the stare number
  int stare = atoi (strtok (NULL, " "));

  // skip over the "Pixel" token
  token = strtok (NULL, " ");
  
  // get the pixel number
  int pixel = atoi (strtok (NULL, " "));

  // initialize the vector
  pixeluvs [stare - 1] [pixel] [0] = pixeluvs [stare - 1] [pixel] [1] = pixeluvs [stare - 1] [pixel] [2] = 0.0;

  // read the three values and add them to the array
  int numbers = 0;
  while ((token = strtok (NULL, " =,[]\n")) != NULL && numbers < 3) {
    pixeluvs [stare - 1] [pixel] [numbers] = atof (token);
    numbers++;
  }

  delete [] linecopy;

  if (numbers == 3) 
    isadded = true;
  else
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_MOPIP_MODULE, PIXEL_ERROR, 0, 0, 0, stare, 0, 0, 
                                 (pixel + 1), 0, "Incorrect number of PixelUV values");

  return isadded;
}


void mopip_values :: ReportRetrieveError (enum mopip_parameter_enumerator mopipenum) const
{
  char enumvalues [10];
  (void) sprintf (enumvalues, "%d", (int) mopipenum);
  string message = "Parameter values have not been assigned for enumerator ";
  message += enumvalues;
  diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_MOPIP_MODULE, RETRIEVE_ERROR, 0, 0, 0, 0, 0, 0, 0, 0, 
                               message);
}


bool mopip_values :: AddEntry (string const& mopipline)
{
  bool isadded = false;

  // make a copy of the line (because strtok mangles the line) and get the value type
  char* linecopy = new char [mopipline.length () + 1];
  (void) strcpy (linecopy, mopipline.c_str ());
  string valuetype = strtok (linecopy, " ");

  // if type is pixel uvs, add the data values to the pixel uvs array
  if (valuetype == "PixelUV") {
    if (AddPixelUV (mopipline))
      isadded = true;
  }
  else {

    // build the parameter object according to the value type
    mopip_parameter* mopipparameter = NULL;
    if      (valuetype == "PRTSieve")               mopipparameter = new prt_sieve_parameter               ();
    else if (valuetype == "PRTBlackbody")           mopipparameter = new prt_blackbody_parameter           ();
    else if (valuetype == "ThermistorStandard")     mopipparameter = new thermistor_standard_parameter     ();
    else if (valuetype == "ThermistorPCDM")         mopipparameter = new thermistor_pcdm_parameter         ();
    else if (valuetype == "ThermistorPressureTemp") mopipparameter = new thermistor_pressuretemp_parameter ();
    else if (valuetype == "ThermistorPressure")     mopipparameter = new thermistor_pressure_parameter     ();
    else if (valuetype == "DiodeDetector")          mopipparameter = new diode_detector_parameter          ();
    else if (valuetype == "DiodeColdFilter")        mopipparameter = new diode_coldfilter_parameter        ();
    else if (valuetype == "DiodeDetectorNest")      mopipparameter = new diode_detectornest_parameter      ();
    else if (valuetype == "HeaterVoltage")          mopipparameter = new heater_voltage_parameter          ();
    else if (valuetype == "PMCPPO")                 mopipparameter = new pmc_ppo_parameter                 ();
    else if (valuetype == "PMCPhaseError")          mopipparameter = new pmc_phaseerror_parameter          ();
    else if (valuetype == "MotorCurrent")           mopipparameter = new motor_current_parameter           ();
    else if (valuetype == "CDEReadout")             mopipparameter = new cde_readout_parameter             ();
    else {
      string message = "Do not recognize sensor type ";
      message += valuetype.c_str ();
      diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_MOPIP_MODULE, TYPE_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                   message);
    }

    // add the parameter to the list
    if (mopipparameter != NULL) {
      if (! mopipparameter->Parse (mopipline))
        delete mopipparameter;
      else {
        isadded = true;
        parameters [mopipparameter->GetID ()] = mopipparameter;
      }
    }
  }

  delete [] linecopy;

  return isadded;
}


bool mopip_values :: Compute (enum mopip_parameter_enumerator mopipenum, float sensor [COMPUTE_TERMS], 
                              float highref [COMPUTE_TERMS], float lowref [COMPUTE_TERMS], 
                              float average [COMPUTE_TERMS]) const
{
  bool iscomputed = true;

  // if the parameter is valid, compute the unit value and standard deviation
  if (parameters [mopipenum] != NULL)
    parameters [mopipenum] ->Compute (sensor, highref, lowref, average);
  else {
    iscomputed = false;
    ReportRetrieveError (mopipenum);
  }

  return iscomputed;
}


bool mopip_values :: Compute (enum mopip_parameter_enumerator mopipenum, float sensor [COMPUTE_TERMS], 
                              float average [COMPUTE_TERMS]) const
{
  bool iscomputed = true;

  // if the parameter is valid, compute the unit value and standard deviation
  if (parameters [mopipenum] != NULL)
    parameters [mopipenum] ->Compute (sensor, average);
  else {
    iscomputed = false;
    ReportRetrieveError (mopipenum);
  }

  return iscomputed;
}


short mopip_values :: GetBank (enum mopip_parameter_enumerator mopipenum) const
{
  // if the parameter is valid, retrieve the bank number
  if (parameters [mopipenum] != NULL)
    return parameters [mopipenum] ->GetBank ();
  else {
    ReportRetrieveError (mopipenum);
    return 0;
  }
}


bool mopip_values :: GetRange (enum mopip_parameter_enumerator mopipenum, float& high, float& low) const
{
  bool isretrieved = true;

  // if the parameter is valid, retrieve the range
  if (parameters [mopipenum] != NULL)
    parameters [mopipenum] ->GetRange (high, low);
  else {
    isretrieved = false;
    ReportRetrieveError (mopipenum);
  }

  return isretrieved;
}


void mopip_values :: GetUVs (int stare, double uvs [PIXELS][VECTOR_LENGTH]) const
{
  (void) memcpy ((void*) uvs, (const void*) pixeluvs [stare], sizeof (double [PIXELS] [VECTOR_LENGTH]));
}


void mopip_values :: GetUVs (int stare, int pixel, double uvs [VECTOR_LENGTH]) const
{
  (void) memcpy ((void*) uvs, (const void*) pixeluvs [stare] [pixel], sizeof (double [VECTOR_LENGTH]));
}
