/********** DiodeDetectorParameter.C **********************************************************************************\

 $Header$

 REVISION HISTORY
   03/99   Charles Cavanaugh
   10/99   Charles Cavanaugh
   05/00   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "DiodeDetectorParameter.h"

int const diode_detector_parameter::PARAMETER_COUNT = 8;

diode_detector_parameter :: diode_detector_parameter ()
                          : mopip_parameter ()
{
  bank = 0;
  bits = 0;
  vH = 0.0;
  vL = 0.0;
  slope = 0.0;
  intercept = 0.0;
  low = 0;
  high = 0;
}


diode_detector_parameter :: ~diode_detector_parameter ()
{
  ;
}


void diode_detector_parameter :: Compute (float sensor [COMPUTE_TERMS], float hiref [COMPUTE_TERMS], 
                                          float loref [COMPUTE_TERMS], float average [COMPUTE_TERMS]) const
{
  // do a bit conversion on the sensor, hi ref and lo ref
  float newsensor, newhiref, newloref;
  BitConvert (bits, sensor [COMPUTE_TERM_AVERAGE], hiref [COMPUTE_TERM_AVERAGE], loref [COMPUTE_TERM_AVERAGE], 
              newsensor, newhiref, newloref);

  // convert readings to voltage and generate the variance of the voltage
  double vD  = vL + ((vH - vL) * ((newsensor - newloref) / (newhiref - newloref)));
  double sensorderiv = (vH - vL) / (newhiref - newloref);
  double vDvariance = (pow (sensorderiv, 2.0) * sensor [COMPUTE_TERM_VARIANCE]) +
                      (pow ((sensorderiv * (newloref - newsensor) / (newhiref - newloref)), 2.0) *
                       hiref [COMPUTE_TERM_VARIANCE]) +
                      (pow ((sensorderiv * (-1.0 + ((newsensor - newloref) / (newhiref - newloref)))), 2.0) *
                       loref [COMPUTE_TERM_VARIANCE]);

  // convert to degrees Kelvin and generate the variance
  average [COMPUTE_TERM_AVERAGE] = intercept + (slope * vD);
  average [COMPUTE_TERM_VARIANCE] = slope * slope * vDvariance;
}


void diode_detector_parameter :: GetRange (float& highrange, float& lowrange) const
{
  highrange = high;
  lowrange = low;
}


bool diode_detector_parameter :: Parse (string const& mopipline)
{
  bool isparsed = false;

  // make a copy of the line
  char* linecopy = new char [mopipline.length () + 1];
  (void) strcpy (linecopy, mopipline.c_str ());

  // skip over the value type
  char* token = strtok (linecopy, " ");

  // set the enumerator, based on the sensor name
  string name = strtok (NULL, " ");
  if      (name == "Detector1") enumerator = DetectorTemperature1;
  else if (name == "Detector2") enumerator = DetectorTemperature2;
  else if (name == "Detector3") enumerator = DetectorTemperature3;
  else if (name == "Detector4") enumerator = DetectorTemperature4;
  else if (name == "Detector5") enumerator = DetectorTemperature5;
  else if (name == "Detector6") enumerator = DetectorTemperature6;
  else if (name == "Detector7") enumerator = DetectorTemperature7;
  else if (name == "Detector8") enumerator = DetectorTemperature8;
  else ReportNameError ("DiodeDetector", name);

  // read in the values
  int valuecount = 0;
  while ((token = strtok (NULL, " =,[]\n")) != NULL && valuecount < PARAMETER_COUNT) {
    switch (valuecount) {
      case 0 : bank = (short) atoi (token);
               break;
      case 1 : bits = (short) atoi (token);
               break;
      case 2 : vH = atof (token);
               break;
      case 3 : vL = atof (token);
               break;
      case 4 : slope = atof (token);
               break;
      case 5 : intercept = atof (token);
               break;
      case 6 : low = (float) atof (token);
               break;
      case 7 : high = (float) atof (token);
               break;
      default : break;
    }

    valuecount++;
  }

  delete [] linecopy;

  if (valuecount == PARAMETER_COUNT)
    isparsed = true;
  else
    ReportValuesError ("DiodeDetector", name);

  return isparsed;
}
