/********** DiodeColdFilterParameter.C ********************************************************************************\

 $Header$

 REVISION HISTORY
   03/99   Charles Cavanaugh
   10/99   Charles Cavanaugh
   05/00   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "DiodeColdFilterParameter.h"

int const diode_coldfilter_parameter::PARAMETER_COUNT = 8 + COEFFS;

diode_coldfilter_parameter :: diode_coldfilter_parameter ()
                            : mopip_parameter ()
{
  bank = 0;
  bits = 0;
  vH = 0.0;
  vL = 0.0;
  zU = 0.0;
  zL = 0.0;
  low = 0;
  high = 0;
  for (int i = 0; i < COEFFS; i++)
    c[i] = 0.0;
}


diode_coldfilter_parameter :: ~diode_coldfilter_parameter ()
{
  ;
}


void diode_coldfilter_parameter :: Compute (float sensor [COMPUTE_TERMS], float hiref [COMPUTE_TERMS],
                                            float loref [COMPUTE_TERMS], float average [COMPUTE_TERMS]) const
{
  // do a bit conversion on the sensor, hi ref and lo ref
  float newsensor, newhiref, newloref;
  BitConvert (bits, sensor [COMPUTE_TERM_AVERAGE], hiref [COMPUTE_TERM_AVERAGE], loref [COMPUTE_TERM_AVERAGE], 
              newsensor, newhiref, newloref);

  // convert readings to voltage and generate the variance in the voltage
  double vD = vL + ((vH - vL) * ((newsensor - newloref) / (newhiref - newloref)));
  double sensorderiv = (vH - vL) / (newhiref - newloref);
  double vDvariance = (pow (sensorderiv, 2.0) * sensor [COMPUTE_TERM_VARIANCE]) + 
                      (pow ((sensorderiv * (newloref - newsensor) / (newhiref - newloref)), 2.0) * 
                       hiref [COMPUTE_TERM_VARIANCE]) +
                      (pow ((sensorderiv * (-1.0 + ((newsensor - newloref) / (newhiref - newloref)))), 2.0) *
                       loref [COMPUTE_TERM_VARIANCE]);

  // get an intermediate value for use in the sum equation and generate the variance in this value
  double vu = ((2 * vD) - zL - zU) / (zU - zL);
  double vuvariance = 2.0 / (zU - zL);
  double v1 = acos (vu);
  double v1variance = pow (vuvariance, 2.0) / (1.0 - (vu * vu));

  // convert to degrees Kelvin and generate the variance in the temperature
  double sum = 0.0;
  double variance = 0.0;
  for (int i = 0; i < COEFFS; i++) {
    sum += (c[i] * cos ((i * v1)));
    variance += (-1.0 * i * c[i] * sin ((i * v1)));
  }

  average [COMPUTE_TERM_AVERAGE] = (float) sum;
  average [COMPUTE_TERM_VARIANCE] = pow (variance, 2.0) * v1variance * vDvariance;
}


void diode_coldfilter_parameter :: GetRange (float& highrange, float& lowrange) const
{
  highrange = high;
  lowrange = low;
}


bool diode_coldfilter_parameter :: Parse (string const& mopipline)
{
  bool isparsed = false;

  // make a copy of the line
  char* linecopy = new char [mopipline.length () + 1];
  (void) strcpy (linecopy, mopipline.c_str ());

  // skip over the value type
  char* token = strtok (linecopy, " ");

  // set the enumerator, based on the sensor name
  string name = strtok (NULL, " ");
  if      (name == "CF1") enumerator = ColdFilterTemperature1;
  else if (name == "CF3") enumerator = ColdFilterTemperature3;
  else if (name == "CF5") enumerator = ColdFilterTemperature5;
  else if (name == "CF7") enumerator = ColdFilterTemperature7;
  else ReportNameError ("DiodeColdFilter", name);

  // read in the values
  int valuecount = 0;
  int coefcount = 0;
  while ((token = strtok (NULL, " =,[]\n")) != NULL && valuecount < PARAMETER_COUNT) {
    switch (valuecount) {
      case 0 : bank = (short) atoi (token);
               break;
      case 1 : bits = (short) atoi (token);
               break;
      case 2 : vH = atof (token);
               break;
      case 3 : vL = atof (token);
               break;
      case 4 : zU = atof (token);
               break;
      case 5 : zL = atof (token);
               break;
      case 6 : low = (float) atof (token);
               break;
      case 7 : high = (float) atof (token);
               break;
      default : c[coefcount] = atof (token);
                coefcount++;
                break;
    }

    valuecount++;
  }

  delete [] linecopy;

  if (valuecount == PARAMETER_COUNT)
    isparsed = true;
  else
    ReportValuesError ("DiodeColdFilter", name);

  return isparsed;
}
