/********** HDFVData.C ************************************************************************************************\

 $Header$

 REVISION HISTORY
   09/99   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include <stdarg.h>
#include "vg.h"
#include "HDFVData.h"
#include "DiagnosticReporter.h"

int const hdf_vdata::ATTACH_ERROR = 903;
int const hdf_vdata::CLOSE_ERROR  = 904;
int const hdf_vdata::CREATE_ERROR = 905;
int const hdf_vdata::DEFINE_ERROR = 906;
int const hdf_vdata::INSERT_ERROR = 907;
int const hdf_vdata::READ_ERROR   = 908;
int const hdf_vdata::SET_ERROR    = 909;
int const hdf_vdata::WRITE_ERROR  = 910;

extern diagnostic_reporter diagnosticreporter;

hdf_vdata :: hdf_vdata ()
{
  fields = 0;
  ishetero = true;
  bytecount = 0;
  records = 0;
  id = 0;
}


hdf_vdata :: ~hdf_vdata ()
{
  ;
}


void hdf_vdata :: CheckHeterogeneity (int fieldcount, int32 const datatypes [])
{
  // assume there is at least one datatype
  int32 datatype = datatypes [0];

  // the ishetero flag describes the heterogeneity of the data
  ishetero = false;

  // scan through the datatypes, and if they are not all the same, set the hetero flag to hetero
  for (int i = 1; i < fieldcount && ! ishetero; i++)
    if (datatypes [i] != datatype)
      ishetero = true;
}


int hdf_vdata :: HDFSizeOf (int32 hdfdatatype)
{
  // these are currently the only types used in the L1 processor
  if (hdfdatatype == DFNT_INT16 || hdfdatatype == DFNT_UINT16)
    return 2;
  else if (hdfdatatype == DFNT_INT32 || hdfdatatype == DFNT_UINT32 || hdfdatatype == DFNT_FLOAT32)
    return 4;
  else if (hdfdatatype == DFNT_FLOAT64)
    return 8;
  else
    return 0;
}


bool hdf_vdata :: Attach (int32 fileid, string const& dataname, string const& fieldnames)
{
  bool isattached = false;

  // set the hetero flag to true (default)
  ishetero = true;

  // find the vdata in the file
  int32 refindex = VSfind (fileid, (char*) dataname.c_str ());
  if (refindex > 0)

    // attach to the vdata
    if ((id = VSattach (fileid, refindex, "r")) != FAIL)

      // get the record count and byte count
      if (VSinquire (id, &records, NULL, NULL, &bytecount, NULL) != FAIL)

        // get the field count
        if ((fields = VFnfields (id)) != FAIL) {

          // if there are no records, stop, else set the fields for reading
          isattached = true;
          if (records > 0)
            if (! Set (fieldnames))
              isattached = false;
        }

  if (! isattached) {
    string message = "Could not attach to vdata ";
    message += dataname;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, ATTACH_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }

  return isattached;
}


bool hdf_vdata :: Attach (int32 fileid, string const& dataname, string const fieldnames [])
{
  bool isattached = false;

  // set the hetero flag to true (default)
  ishetero = true;

  // find the vdata in the file
  int32 refindex = VSfind (fileid, (char*) dataname.c_str ());
  if (refindex > 0)

    // attach to the vdata
    if ((id = VSattach (fileid, refindex, "r")) != FAIL)

      // get the record count and byte count
      if (VSinquire (id, &records, NULL, NULL, &bytecount, NULL) != FAIL)

        // get the field count
        if ((fields = VFnfields (id)) != FAIL) {

          // if there are no records, stop, else set the fields for reading
          isattached = true;
          if (records > 0)
            if (! Set (fields, fieldnames))
              isattached = false;
        }

  if (! isattached) {
    string message = "Could not attach to vdata ";
    message += dataname;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, ATTACH_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }

  return isattached;
}


bool hdf_vdata :: Close ()
{
  bool isclosed = true;

  // detach from the vdata
  if (VSdetach (id) == FAIL) {
    char idchars [10];
    (void) sprintf (idchars, "%d", id);
    string message = "Could not detach from vdata with id ";
    message += idchars;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, CLOSE_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
    isclosed = false;
  }

  return isclosed;
}


bool hdf_vdata :: Create (string const& dataname, string const& dataclass, int32 fileid)
{
  bool iscreated = false;

  // make a new vdata and attach to it
  if ((id = VSattach (fileid, -1, "w")) != FAIL)
    if (VSsetname (id, (char*) dataname.c_str ()) != FAIL)
      if (VSsetclass (id, (char*) dataclass.c_str ()) != FAIL)
        iscreated = true;

  if (! iscreated) {
    string message = "Could not make the hdf vdata ";
    message += dataname;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, CREATE_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }

  return iscreated;
}


bool hdf_vdata :: Define (int32 datatype, int fieldcount, const char* const fieldname1, ...)
{
  bool isdefined = true;

  // save the field count
  fields = fieldcount;

  // this routine is useable by only homogeneous fields
  ishetero = false;

  // start access to the variable argument list
  va_list argptr;
  va_start (argptr, fieldcount);

  // loop over the field names
  string fieldname;
  for (; fieldcount > 0 && isdefined; fieldcount--) {

    // extract the next field name, then define the field and accumulate the byte count
    fieldname = (char*) va_arg (argptr, const char*);
    if (VSfdefine (id, (char*) fieldname.c_str (), datatype, 1) == FAIL)
      isdefined = false;
    else
      bytecount += HDFSizeOf (datatype);
  }

  // end access to the argument list
  va_end (argptr);

  // if any problems, report it
  if (! isdefined)
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, DEFINE_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 "Could not define vdata fields");

  return isdefined;
}


bool hdf_vdata :: Define (int fieldcount, int32 const datatypes [], int32 const orders [], string const fieldnames [])
{
  bool isdefined = true;

  // save the field count
  fields = fieldcount;

  // set hetero flag
  CheckHeterogeneity (fieldcount, datatypes);

  // loop over the fields, defining each one (and accumulating the byte count)
  for (int i = 0; i < fieldcount && isdefined; i++)
    if (VSfdefine (id, (char*) (fieldnames [i]).c_str (), datatypes [i], orders [i]) == FAIL)
      isdefined = false;
    else
      bytecount += HDFSizeOf (datatypes [i]) * orders [i];

  // if any problems, report it
  if (! isdefined)
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, DEFINE_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 "Could not define vdata fields");

  return isdefined;
}


bool hdf_vdata :: Insert (int32 groupid)
{
  bool isinserted = true;

  // insert the vdata into the vgroup
  if (Vinsert (groupid, id) == FAIL) {
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, INSERT_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 "Could not insert the vdata into the vgroup");
    isinserted = false;
  }

  return isinserted;
}


bool hdf_vdata :: Read (int32 recordindex, void* bufferptrs [])
{
  bool isread = false;

  // check the record index
  if (recordindex >= 0 && recordindex < records) 

    // seek to the record position
    if (VSseek (id, recordindex) != FAIL) {

      // create space for the data, read in the data, unpack the data
      uint8* readbuffer = new uint8 [bytecount];
      if (VSread (id, readbuffer, 1, FULL_INTERLACE) == 1)
        if (VSfpack (id, _HDF_VSUNPACK, NULL, (VOIDP) readbuffer, bytecount, 1, NULL, bufferptrs) != FAIL)
          isread = true;
      delete [] readbuffer;
    }

  if (! isread) {
    char recordchars [10];
    (void) sprintf (recordchars, "%d", recordindex);
    string message = "Could not read vdata record ";
    message += recordchars;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, READ_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }

  return isread;
}


bool hdf_vdata :: Set (string const& fieldnames)
{
  bool isset = true;

  // set the names of the vdata fields
  if (VSsetfields (id, (char*) fieldnames.c_str ()) == FAIL) {
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, SET_ERROR, 0, 0, 0, 0, 0, 0, 0, 0, 
                                 "Could not set the vdata fields");
    isset = false;
  }

  return isset;
}


bool hdf_vdata :: Set (int fieldcount, string const fieldnames [])
{
  // assume there is at least one name and initialize the name string
  string names = fieldnames [0];

  // add the other names
  for (int i = 1; i < fieldcount; i++) {
    names += ",";
    names += fieldnames [i];
  }

  // set the fields
  return Set (names);
}


bool hdf_vdata :: Write (void* writevalues)
{
  bool iswritten = true;

  // move to the correct position
  if (records > 0) {
    uint16* readbuffer = new uint16 [bytecount];
    iswritten = false;
    if (VSseek (id, (records - 1)) != FAIL)
      if (VSread (id, (uint8*) readbuffer, 1, FULL_INTERLACE) != FAIL)
        iswritten = true;
    delete [] readbuffer;
  }

  // write the data
  if (iswritten)
    if (VSwrite (id, (uint8*) writevalues, 1, FULL_INTERLACE) == 1)
      records++;
    else
      iswritten = false;
   
  if (! iswritten) {
    char recordschars [10];
    (void) sprintf (recordschars, "%d", records);
    string message = "Could not write record ";
    message += recordschars;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, WRITE_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }

  return iswritten;
}


bool hdf_vdata :: Write (void* bufferptrs [])
{
  bool iswritten = false;

  // pack the data
  uint16* writebuffer = new uint16 [bytecount];
  if (VSfpack (id, _HDF_VSPACK, NULL, (VOIDP) writebuffer, bytecount, 1, NULL, bufferptrs) != FAIL) {

    // reset iswritten for further processing
    iswritten = true;

    // move to the correct position
    if (records > 0) {
      uint16* readbuffer = new uint16 [bytecount];
      iswritten = false;
      if (VSseek (id, (records - 1)) != FAIL)
        if (VSread (id, (uint8*) readbuffer, 1, FULL_INTERLACE) != FAIL)
          iswritten = true;
      delete [] readbuffer;
    }

    // write the data
    if (iswritten)
      if (VSwrite (id, (uint8*) writebuffer, 1, FULL_INTERLACE) == 1)
        records++;
      else
        iswritten = false;
  }

  delete [] writebuffer;

  if (! iswritten) {
    char recordschars [10];
    (void) sprintf (recordschars, "%d", records);
    string message = "Could not write record ";
    message += recordschars;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, WRITE_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }

  return iswritten;
}
