/********** HDFSwathDataset.C *****************************************************************************************\

 $Header$

 REVISION HISTORY
   11/99   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include "HDFSwathDataset.h"
#include "SwathConstants.h"
#include "DiagnosticReporter.h"
#include <string>
#include <iostream>

extern diagnostic_reporter diagnosticreporter;

int const hdf_swath_dataset::INITIALIZE_ERROR = 911;
int const hdf_swath_dataset::SET_ERROR        = 912;
int const hdf_swath_dataset::WRITE_ERROR      = 913;

hdf_swath_dataset :: hdf_swath_dataset ()
                   : name ()
{
  type = 0;
  dimensions = 0;
  dimensionsizes = NULL;
  dimensionnames = NULL;
  maxdimensionnames = NULL;
  start = NULL;
  trackindex = 0;
  //first = 0;
}


hdf_swath_dataset :: hdf_swath_dataset (string const& datasetname, int32 datasettype, int datasetdimensions, 
                                        hsize_t const datasetdimensionsizes [], string const datasetdimensionnames [])
{
  Initialize (datasetname, datasettype, datasetdimensions, datasetdimensionsizes, datasetdimensionnames);
  trackindex = 0;
}


hdf_swath_dataset :: ~hdf_swath_dataset ()
{
  delete [] dimensionsizes;
  delete [] dimensionnames;
  delete [] maxdimensionnames;
  delete [] start;
}


bool hdf_swath_dataset :: Initialize (string const& datasetname, int32 datasettype, int datasetdimensions, 
                                      hsize_t const datasetdimensionsizes [], string const datasetdimensionnames [])
{
  bool isinitialized = false;

  // save the dataset name, type and dimension count
  name = datasetname;
  type = datasettype;
  dimensions = datasetdimensions;

  // delete the old dimension info
  delete [] dimensionsizes;
  delete [] dimensionnames;
  delete [] maxdimensionnames;
  delete [] start;
  
  // there has to be at least one dimension
  if (dimensions < 1) {
    string message = "Could not initialize hdf swath dataset ";
    message += name;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, INITIALIZE_ERROR, 0, 0, 0, 0, 0, 0, 0, 0,
                                 message);
  }
  else {

    // save the new dimension info
    dimensionsizes = new hsize_t [dimensions];
    dimensionnames = new string [dimensions];
    maxdimensionnames = new string [dimensions];
    start = new hssize_t [dimensions];
    isinitialized = true;

    for (int i = 0; i < dimensions; i++) {
      dimensionsizes [i] = datasetdimensionsizes [i];
      dimensionnames [i] = datasetdimensionnames [i];
      maxdimensionnames [i] = datasetdimensionnames [i];
      start [i] = 0;
    }
    maxdimensionnames[0] = "Unlim";

    //cout << "======" << datasetname.c_str() << endl;
    //for (int i = 0; i < dimensions; i++) {
    //  cout<<i<<"  "<<dimensionnames[i].c_str()<<"  "<<dimensionsizes[i]<<"  "<<maxdimensionnames[i].c_str()<<endl;
    //}
  }

  return isinitialized;
}


bool hdf_swath_dataset :: Define (hid_t dataid)
{
  bool isdefined = false;
  hsize_t* sdimensizes;

  // if the fields have been initialized, define the dataset 
  if (dimensions > 0) {

    sdimensizes = new hsize_t [dimensions];
    sdimensizes [0] = 20;
    for (int i = 1; i < dimensions; i++) {
      sdimensizes [i] = dimensionsizes [i];
    }

    // change dimensionsizes [0] to 1 because dimensionsizes is no longer needed for definition,
    // just for writing (and the 1 signifies 1 track of writing)
    dimensionsizes [0] = 1;

    string dimnames = dimensionnames [0];
    string maxdimnames = maxdimensionnames [0];

    for (int i = 1; i < dimensions; i++) {
      dimnames += ",";
      dimnames += dimensionnames [i];
      maxdimnames += ",";
      maxdimnames += maxdimensionnames [i];
    }

    isdefined = SetField (dataid, name, dimensions, sdimensizes, dimnames, maxdimnames, type);

    
  }
  delete [] sdimensizes;
  return isdefined;
}

bool hdf_swath_dataset :: Define (hid_t dataid, string const& datasetname, int32 datasettype, int datasetdimensions,
                                  hsize_t const datasetdimensionsizes [], string const datasetdimensionnames [], string const units)
{
  bool isdefined = false;
  hsize_t cnt = 1; 
  
  if (Initialize (datasetname, datasettype, datasetdimensions, datasetdimensionsizes, datasetdimensionnames))
    if (Define (dataid)) {
      float missingvalue = SWATH_MISSING_VALUE;
      if (HE5_SWwritelocattr (dataid, (char*) datasetname.c_str(), "missing_nodata", H5T_NATIVE_FLOAT, &cnt, &missingvalue) != FAIL) {
	float invalidvalue = SWATH_INVALID_VALUE;
	if (HE5_SWwritelocattr (dataid, (char*) datasetname.c_str(), "missing_invalid", H5T_NATIVE_FLOAT, &cnt, &invalidvalue) != FAIL) {
	  cnt = units.size();
	  if (HE5_SWwritelocattr(dataid, (char*) datasetname.c_str(), "units", H5T_NATIVE_CHAR, &cnt, (char*) units.c_str()) != FAIL)
	    isdefined = true;
	}
      }
    }

  return isdefined;
}


bool hdf_swath_dataset :: Write (hid_t dataid, int newtrackindex, VOIDP data)
{
  bool iswritten = true;

  // the data has to have at least one dimension
  if (dimensions < 1)
    iswritten = false;
  else {

    // tell the write where to start
    start [0] = newtrackindex;
    if (HE5_SWwritefield (dataid, (char*) name.c_str (), start, NULL, dimensionsizes, data) == FAIL)
      iswritten = false;
  }

  if (! iswritten) {
    string message = "Could not write to dataset ";
    message += name;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, WRITE_ERROR, 0, (newtrackindex + 1), 0, 0,
                                 0, 0, 0, 0, message);
  }

  return iswritten;
}


bool hdf_swath_dataset :: Write (hid_t dataid, VOIDP data)
{
  bool iswritten = true;

  // the data has to have at least one dimension
  if (dimensions < 1)
    iswritten = false;
  else {

    // tell the write where to start
    start [0] = trackindex;
    if (HE5_SWwritefield (dataid, (char*) name.c_str (), start, NULL, dimensionsizes, data) == FAIL)
      iswritten = false;
  }

  if (iswritten)
    trackindex++;
  else {
    string message = "Could not write to dataset ";
    message += name;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_HDF_MODULE, WRITE_ERROR, 0, (trackindex + 1), 0, 0, 0,
                                 0, 0, 0, message);
  }

  return iswritten;
}
