/********** EngineeringData.C *****************************************************************************************\

 $Header$

 REVISION HISTORY
   02/99   Charles Cavanaugh
   06/99   Tom Lauren
   10/99   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include "EngineeringData.h"
#include "DiagnosticReporter.h"

extern diagnostic_reporter diagnosticreporter;

int const engineering_data::OVERLAP_WARNING = 804;
int const engineering_data::TRAIN_WARNING   = 805;

engineering_data :: engineering_data ()
                  : timedata (),
                    instrumentdata (),
                    bench1data (),
                    bench2data (),
                    train1data (),
                    train2data (),
                    train3data (),
                    train4data ()
{
  swathindex    = 0;
  trainindex[0] = 0;
  trainindex[1] = 0;
  trainindex[2] = 0;
  trainindex[3] = 0;
  for (int i = 0; i < ENGINEERING_LISTS; i++)
    summaryindex [i] = 0;
}


engineering_data :: ~engineering_data ()
{
  ;
}


void engineering_data :: WriteIndecesDiagnostic (mopitt_time const& starttime, mopitt_time const& stoptime) const
{
  string message = "No engineering blocks within ";
  message += starttime.GetUTC ();
  message += " to ";
  message += stoptime.GetUTC ();
  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_ENGINEERING_MODULE, OVERLAP_WARNING, 0, 0, 0, 0, 0, 0, 
                               0, 0, message);
}


void engineering_data :: AddBlock (engineering_block* engineeringblock)
{
  // add the block to all the members
  timedata.AddBlock       (engineeringblock);
  instrumentdata.AddBlock (engineeringblock);
  bench1data.AddBlock     (engineeringblock);
  bench2data.AddBlock     (engineeringblock);
  train1data.AddBlock     (engineeringblock);
  train2data.AddBlock     (engineeringblock);
  train3data.AddBlock     (engineeringblock);
  train4data.AddBlock     (engineeringblock);
}


bool engineering_data :: Close ()
{
  bool isclosed = false;

  if (timedata.Close ())
    if (instrumentdata.Close ())
      if (bench1data.Close ())
        if (bench2data.Close ())
          if (train1data.Close ())
            if (train2data.Close ())
              if (train3data.Close ())
                if (train4data.Close ())
                  isclosed = true;

  return isclosed;
}


bool engineering_data :: Define (int32 fileid, int32 engineeringid)
{
  bool isdefined = false;
  string const groupclass = "Engineering Group";

  if (timedata.Define (fileid, engineeringid, groupclass))
    if (instrumentdata.Define (fileid, engineeringid, groupclass))
      if (bench1data.Define (fileid, engineeringid, groupclass))
        if (bench2data.Define (fileid, engineeringid, groupclass))
          if (train1data.Define (fileid, engineeringid, groupclass))
            if (train2data.Define (fileid, engineeringid, groupclass))
              if (train3data.Define (fileid, engineeringid, groupclass))
                if (train4data.Define (fileid, engineeringid, groupclass))
                  isdefined = true;

  return isdefined;
}


float engineering_data :: GetEventTemperature (int train, mopitt_time const& starttime, mopitt_time const& stoptime,
                                               mopip_file const& mopipfile)
{
  // set the event temperature to the default value
  float eventtemperature = 0.0;

  // get the time indeces
  int stopindex;
  if (! timedata.GetTelemIndeces (starttime, stoptime, trainindex [train - 1], stopindex))
    WriteIndecesDiagnostic (starttime, stoptime);

  // call the appropriate function
  switch (train) {
    case 1 : eventtemperature = train1data.GetEventTemperature (trainindex [0], stopindex, mopipfile);
             break;
    case 2 : eventtemperature = train2data.GetEventTemperature (trainindex [1], stopindex, mopipfile);
             break;
    case 3 : eventtemperature = train3data.GetEventTemperature (trainindex [2], stopindex, mopipfile);
             break;
    case 4 : eventtemperature = train4data.GetEventTemperature (trainindex [3], stopindex, mopipfile);
             break;
    default : diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_SCIENCE_MODULE, TRAIN_WARNING, 0, 0, 0, 0,
                                           train, 0, 0, 0, "Incorrect train number for event temperature computation");
  }

  return eventtemperature;
}


void engineering_data :: GetSummaryData (mopitt_time const& starttime, mopitt_time const& stoptime, 
                                         mopip_file const& mopipfile,
                                         float summaryengineering [ENGINEERING_SUMMARY_VALUES][ENGINEERING_STATES])
{
  // Get the time indeces.
  int stopindex[ENGINEERING_LISTS];
  timedata.GetTelemIndeces  (starttime, stoptime, summaryindex [TELEM],  stopindex [TELEM]);
  timedata.GetBlock3Indeces (starttime, stoptime, summaryindex [BLOCK3], stopindex [BLOCK3]);
  timedata.GetBlock4Indeces (starttime, stoptime, summaryindex [BLOCK4], stopindex [BLOCK4]);
  timedata.GetBlock5Indeces (starttime, stoptime, summaryindex [BLOCK5], stopindex [BLOCK5]);
  timedata.GetBlock8Indeces (starttime, stoptime, summaryindex [BLOCK8], stopindex [BLOCK8]);

  // Get the references.
  float diodehighrefs [ENGINEERING_LISTS] [DIODE_BANKS] [ENGINEERING_STATES], 
        diodelowrefs [ENGINEERING_LISTS] [DIODE_BANKS] [ENGINEERING_STATES], 
        thermistorhighrefs [ENGINEERING_LISTS] [THERMISTOR_BANKS] [ENGINEERING_STATES], 
        thermistorlowrefs [ENGINEERING_LISTS] [THERMISTOR_BANKS] [ENGINEERING_STATES];
  for (int i = 0; i < ENGINEERING_LISTS; i++) {
    instrumentdata.GetReferences (summaryindex [i], stopindex [i], diodehighrefs [i], diodelowrefs [i], 
                                  thermistorhighrefs [i], thermistorlowrefs [i]);
  }

  // Get the summary data for telem and blocks 3, 4, 5, and 8.

  // Get the telem summary data.
  bench1data.GetTelemSummaryData (summaryindex[TELEM], stopindex[TELEM], mopipfile, thermistorhighrefs[TELEM], 
                                  thermistorlowrefs[TELEM], summaryengineering);
  bench2data.GetTelemSummaryData (summaryindex[TELEM], stopindex[TELEM], mopipfile, thermistorhighrefs[TELEM], 
				  thermistorlowrefs[TELEM], summaryengineering);
  train1data.GetTelemSummaryData (summaryindex[TELEM], stopindex[TELEM], mopipfile, diodehighrefs[TELEM], 
                                  diodelowrefs[TELEM], thermistorhighrefs[TELEM], thermistorlowrefs[TELEM], 
                                  summaryengineering);
  train2data.GetTelemSummaryData (summaryindex[TELEM], stopindex[TELEM], mopipfile, diodehighrefs[TELEM], 
                                  diodelowrefs[TELEM], thermistorhighrefs[TELEM], thermistorlowrefs[TELEM], 
                                  summaryengineering);
  train3data.GetTelemSummaryData (summaryindex[TELEM], stopindex[TELEM], mopipfile, diodehighrefs[TELEM], 
                                  diodelowrefs[TELEM], thermistorhighrefs[TELEM], thermistorlowrefs[TELEM], 
                                  summaryengineering);
  train4data.GetTelemSummaryData (summaryindex[TELEM], stopindex[TELEM], mopipfile, diodehighrefs[TELEM], 
                                  diodelowrefs[TELEM], thermistorhighrefs[TELEM], thermistorlowrefs[TELEM], 
                                  summaryengineering);

  // Get the block 3 summary data.
  instrumentdata.GetBlock3SummaryData (summaryindex[BLOCK3], stopindex[BLOCK3], mopipfile, summaryengineering);
  bench1data.GetBlock3SummaryData     (summaryindex[BLOCK3], stopindex[BLOCK3], mopipfile, summaryengineering);
  bench2data.GetBlock3SummaryData     (summaryindex[BLOCK3], stopindex[BLOCK3], mopipfile, summaryengineering);

  // Get the block 4 summary data.
  bench2data.GetBlock4SummaryData (summaryindex[BLOCK4], stopindex[BLOCK4], mopipfile, summaryengineering);
  train1data.GetBlock4SummaryData (summaryindex[BLOCK4], stopindex[BLOCK4], mopipfile, summaryengineering);
  train2data.GetBlock4SummaryData (summaryindex[BLOCK4], stopindex[BLOCK4], mopipfile, summaryengineering);
  train3data.GetBlock4SummaryData (summaryindex[BLOCK4], stopindex[BLOCK4], mopipfile, summaryengineering);
  train4data.GetBlock4SummaryData (summaryindex[BLOCK4], stopindex[BLOCK4], mopipfile, summaryengineering);

  // Get the block 5 summary data.
  instrumentdata.GetBlock5SummaryData (summaryindex[BLOCK5], stopindex[BLOCK5], mopipfile, thermistorhighrefs[BLOCK5], 
				       thermistorlowrefs[BLOCK5], summaryengineering);
  bench1data.GetBlock5SummaryData (summaryindex[BLOCK5], stopindex[BLOCK5], mopipfile, thermistorhighrefs[BLOCK5], 
				   thermistorlowrefs[BLOCK5], summaryengineering);
  bench2data.GetBlock5SummaryData (summaryindex[BLOCK5], stopindex[BLOCK5], mopipfile, thermistorhighrefs[BLOCK5], 
				   thermistorlowrefs[BLOCK5], summaryengineering);
  train1data.GetBlock5SummaryData (summaryindex[BLOCK5], stopindex[BLOCK5], mopipfile, thermistorhighrefs[BLOCK5], 
				   thermistorlowrefs[BLOCK5], summaryengineering);
  train2data.GetBlock5SummaryData (summaryindex[BLOCK5], stopindex[BLOCK5], mopipfile, thermistorhighrefs[BLOCK5], 
				   thermistorlowrefs[BLOCK5], summaryengineering);
  train3data.GetBlock5SummaryData (summaryindex[BLOCK5], stopindex[BLOCK5], mopipfile, thermistorhighrefs[BLOCK5], 
				   thermistorlowrefs[BLOCK5], summaryengineering);
  train4data.GetBlock5SummaryData (summaryindex[BLOCK5], stopindex[BLOCK5], mopipfile, thermistorhighrefs[BLOCK5], 
				   thermistorlowrefs[BLOCK5], summaryengineering);

  // Get the block 8 summary data.
  instrumentdata.GetBlock8SummaryData (summaryindex[BLOCK8], stopindex[BLOCK8], mopipfile, summaryengineering);
}


void engineering_data :: GetSwathData (mopitt_time const& starttime, mopitt_time const& stoptime, 
                                       mopip_file const& mopipfile,
                                       float swathengineering [ENGINEERING_SWATH_VALUES][ENGINEERING_STATES])
{
  // get the time indeces
  int stopindex;
  if (! timedata.GetTelemIndeces (starttime, stoptime, swathindex, stopindex))
    WriteIndecesDiagnostic (starttime, stoptime);

  // get the references
  float diodehighrefs [DIODE_BANKS] [ENGINEERING_STATES], diodelowrefs [DIODE_BANKS] [ENGINEERING_STATES], 
        thermistorhighrefs [THERMISTOR_BANKS] [ENGINEERING_STATES], 
        thermistorlowrefs [THERMISTOR_BANKS] [ENGINEERING_STATES];
  instrumentdata.GetReferences (swathindex, stopindex, diodehighrefs, diodelowrefs, thermistorhighrefs, 
                                thermistorlowrefs);

  // all the swath data can be found in the train lists
  train1data.GetSwathData (swathindex, stopindex, mopipfile, diodehighrefs, diodelowrefs, thermistorhighrefs, 
                           thermistorlowrefs, swathengineering);
  train2data.GetSwathData (swathindex, stopindex, mopipfile, diodehighrefs, diodelowrefs, thermistorhighrefs, 
                           thermistorlowrefs, swathengineering);
  train3data.GetSwathData (swathindex, stopindex, mopipfile, diodehighrefs, diodelowrefs, thermistorhighrefs, 
                           thermistorlowrefs, swathengineering);
  train4data.GetSwathData (swathindex, stopindex, mopipfile, diodehighrefs, diodelowrefs, thermistorhighrefs, 
                           thermistorlowrefs, swathengineering);
}


bool engineering_data :: Write ()
{
  bool iswritten = false;

  if (timedata.Write ())
    if (instrumentdata.Write ())
      if (bench1data.Write ())
        if (bench2data.Write ())
          if (train1data.Write ())
            if (train2data.Write ())
              if (train3data.Write ())
                if (train4data.Write ())
                  iswritten = true;

  return iswritten;
}
