/********** PostProcInL1.C **************************************************************************************\
 PURPOSE
   Noises daily mean calculated and 3D and 7D shifted for L1 processor.     

 ASSUMPTIONS ON ENTRY
   The MOP01 file is available.

 ASSUMPTIONS ON EXIT
   The MOP01 file has been updated.

 REVISION HISTORY
   06/02   Debbie Mao
   10/09   Debbie Mao
   08/15   Debbie Mao - add daily Gain Deviation
   03/20   Debbie Mao - validate ch7 radiance by invalidateTrack information
\****************************************************************************************************************/
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "PGS_PC.h"
#include "mfhdf.h"
#include "HE5_HdfEosDef.h"
#include "DiagnosticReporter.h"
#include "SwathConstants.h"

extern diagnostic_reporter diagnosticreporter;

void PostProcInL1 ()
{
  int const REFERENCE_ERROR     = 880;
  int32 const MAX_TRACK_NUMBER  = 10000;
  PGSt_PC_Logical const logical = 41600;
  PGSt_integer version = 1;
  char filename [PGSd_PC_FILE_PATH_MAX], ch7flaginfo[100];

  int   pixel, chan, stare, track, state;
  intn  errstat;
  hid_t fileid, swid;
  int32 trackcount = -1, ierr, ierro, ierror, ierror2, inmn, igmn, idif, imn1, imn2, imn3;
  hsize_t  edge[5],  edgen[3],  edgep[2];
  hssize_t start[5], startn[3], startp[2];

  float cal [MAX_TRACK_NUMBER][SWATH_PIXELS][SWATH_CHANNELS][SWATH_RADIANCE_STATES][SWATH_CALIBRATION_TERMS],
        dailymeannoise [SWATH_PIXELS][SWATH_CHANNELS][SWATH_RADIANCE_STATES],
        dailygaindev [SWATH_PIXELS][SWATH_CHANNELS][SWATH_RADIANCE_STATES],
        nmn [SWATH_PIXELS][SWATH_CHANNELS][SWATH_RADIANCE_STATES],
        gmn [SWATH_PIXELS][SWATH_CHANNELS][SWATH_RADIANCE_STATES],
        gdv [SWATH_PIXELS][SWATH_CHANNELS][SWATH_RADIANCE_STATES];
  int   pos [MAX_TRACK_NUMBER][SWATH_STARES], ch7flag[MAX_TRACK_NUMBER];
  float rad [MAX_TRACK_NUMBER][SWATH_STARES][SWATH_PIXELS][SWATH_CHANNELS][SWATH_RADIANCE_STATES],
        mndif[SWATH_PIXELS][SWATH_CHANNELS], dif1mn[SWATH_PIXELS][SWATH_CHANNELS], 
        dif2mn[SWATH_PIXELS][SWATH_CHANNELS], dif3mn[SWATH_PIXELS][SWATH_CHANNELS];

  // initailize
  for ( pixel = 0; pixel < SWATH_PIXELS; pixel++ ) 
    for ( chan = 0; chan < SWATH_CHANNELS; chan++ ) 
      for ( state = 0; state < SWATH_RADIANCE_STATES; state++ ) {
	dailymeannoise [pixel][chan][state] = SWATH_MISSING_VALUE;
	dailygaindev [pixel][chan][state] = SWATH_MISSING_VALUE;
      }
  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_MOP01_MODULE, REFERENCE_ERROR, 0, 0, 0, 0,  0, 0, 0, 0, "Get in PostProcInL1");

  // read MOP01 file
  if (PGS_PC_GetReference (logical, &version, filename) != PGS_S_SUCCESS)
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_MOP01_MODULE, REFERENCE_ERROR, logical, 0, 0, 0, 
				 0, 0, 0, 0, "Could not get filename");
  if ((fileid = HE5_SWopen ((char*) filename, H5F_ACC_RDWR) ) == FAIL) 
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_MOP01_MODULE, REFERENCE_ERROR, logical, 0, 0, 0,  
				 0, 0, 0, 0, "Could not open MOP01 file");  
  swid = HE5_SWattach (fileid, "MOP01");
  if (swid < (int32) 0) {
    (void) HE5_SWclose (fileid);
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_MOP01_MODULE, REFERENCE_ERROR, logical, 0, 0, 0, 
				 0, 0, 0, 0, "Could not attach file");
  }
  errstat = HE5_SWreadattr (swid, "TrackCount", (VOIDP) &trackcount);	
  if (errstat == FAIL || trackcount < (int32) 1){
    (void) HE5_SWdetach (swid);
    (void) HE5_SWclose (fileid);
    trackcount = -1;
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_MOP01_MODULE, REFERENCE_ERROR, logical, 0, 0, 0, 
				 0, 0, 0, 0, "Could not read track number");
  }	  
  start[0] = 0;           start[1] = 0;  start[2] = 0;  start[3] = 0;  start[4] = 0;
  edge [0] = trackcount;  edge [1] = 4;  edge [2] = 8;  edge [3] = 2;  edge [4] = 8;
  ierr = HE5_SWreadfield(swid,"CalibrationData",start, NULL,edge, (VOIDP)cal);
  
  start[0] = 0;          start[1] =  0; start[2] = 0; start[3] = 0; start[4] = 0;
  edge [0] = trackcount; edge [1] = 29; edge [2] = 4; edge [3] = 8; edge [4] = 2;
  ierro = HE5_SWreadfield(swid,"MOPITTRadiances",start, NULL,edge,(VOIDP) rad);

  startp[0] = 0;          startp[1] =  0; 
  edgep [0] = trackcount; edgep [1] = 29; 
  ierror = HE5_SWreadfield(swid,"PacketPositions",startp, NULL, edgep, pos);
  ierror2 = HE5_SWreadfield(swid,"Ch7Invalid",startp, NULL, edgep, ch7flag);
  
  if ( (ierr != 0) || ( ierro != 0) || ( ierror != 0) || ( ierror2 != 0) )
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_SYSTEM_MODULE, REFERENCE_ERROR, 
				 logical, 0, 0, 0, 0, 0, 0, 0, "Could not read MOPITT Cali/Radiances/PacketPos/Ch7Invalid");  

  // Check Ch7 Invalid flag to reset middle good rad to invalid value
  for ( track = 2; track < trackcount-3; track++ )
    if (ch7flag[track] == 0)
      if ( (ch7flag[track-2] == 1 || ch7flag[track-1] == 1) && (ch7flag[track+1] == 1 || ch7flag[track+2] == 1) )  {
	  (void) sprintf (ch7flaginfo, "%5d %5d %5d %5d %5d", ch7flag[track-2], ch7flag[track-1], ch7flag[track], ch7flag[track+1], ch7flag[track+2]);
	  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_MOP01_MODULE, REFERENCE_ERROR, 0, track+1, 0, 0,  0, 0, 0, 0, ch7flaginfo);
	  for ( stare = 0; stare < SWATH_STARES; stare++ )
	    for ( pixel = 0; pixel < SCIENCE_PIXELS; pixel++ ) 
	      for ( state = 0; state < SWATH_RADIANCE_STATES; state++ ) 
		rad[track][stare][pixel][6][state] = SWATH_INVALID_VALUE;
	}
  if ( ch7flag[0] == 1 && ch7flag[3] == 1 )    
    for ( stare = 0; stare < SWATH_STARES; stare++ )
      for ( pixel = 0; pixel < SCIENCE_PIXELS; pixel++ ) 
	for ( state = 0; state < SWATH_RADIANCE_STATES; state++ ) 
	  rad[2][stare][pixel][6][state] = SWATH_INVALID_VALUE;
  if ( ch7flag[trackcount-3] == 1 && ch7flag[trackcount-1] == 1 )    
    for ( stare = 0; stare < SWATH_STARES; stare++ )
      for ( pixel = 0; pixel < SCIENCE_PIXELS; pixel++ ) 
	for ( state = 0; state < SWATH_RADIANCE_STATES; state++ ) 
	  rad[trackcount-2][stare][pixel][6][state] = SWATH_INVALID_VALUE;    
  
  // calculation
  for ( pixel = 0; pixel < SCIENCE_PIXELS; pixel++ ) {
    // noise mean (npixels,nchan,nstate)
    for ( chan = 0; chan < SCIENCE_CHANNELS; chan++ ) 
      for ( state = 0; state < SWATH_RADIANCE_STATES; state++ ) {
	nmn[pixel][chan][state] = 0.0;
	inmn = 0;
	for ( track = 0; track < trackcount; track++ ) 
	  if ( cal[track][pixel][chan][state][2] > 0.0 && isnan(cal[track][pixel][chan][state][2]) == 0 ) {
	    nmn[pixel][chan][state] = nmn[pixel][chan][state] + cal[track][pixel][chan][state][2];
	    inmn++;
	  }
	if ( inmn > 0 ) dailymeannoise[pixel][chan][state] = nmn[pixel][chan][state] / inmn;
      }

    // gain deviation (npixels,nchan,nstate)
    for ( chan = 0; chan < SCIENCE_CHANNELS; chan=chan+2 ) 
      for ( state = 0; state < SWATH_RADIANCE_STATES; state++ ) {
	gmn[pixel][chan][state] = 0.0;
	gdv[pixel][chan][state] = 0.0;
	igmn = 0;
	for ( track = 0; track < trackcount; track++ ) 
	  if ( cal[track][pixel][chan][state][0] > 0.0 && isnan(cal[track][pixel][chan][state][0]) == 0 ) {
	    gmn[pixel][chan][state] = gmn[pixel][chan][state] + cal[track][pixel][chan][state][0];
	    igmn++;
	  }
	if ( igmn > 0 ) {
	  gmn[pixel][chan][state] = gmn[pixel][chan][state] / igmn;
	  for ( track = 0; track < trackcount; track++ )
	    if ( cal[track][pixel][chan][state][0] > 0.0 && isnan(cal[track][pixel][chan][state][0]) == 0 ) 
	      gdv[pixel][chan][state] += powf((cal[track][pixel][chan][state][0]-gmn[pixel][chan][state]), 2);
	  dailygaindev[pixel][chan][state] = sqrtf( gdv[pixel][chan][state] / igmn );
	}
      }

    // R3D, R7D mean at packet position 4&5, 1, 2, 3
    for ( chan = 2; chan < SWATH_CHANNELS; chan = chan + 4 ) { 		
      mndif [pixel][chan] = 0.0;
      dif1mn[pixel][chan] = 0.0;
      dif2mn[pixel][chan] = 0.0;
      dif3mn[pixel][chan] = 0.0;
      idif=0;
      imn1=0; 
      imn2=0;
      imn3=0;

      for ( track = 0; track < trackcount; track++ ) 
	for ( stare = 0; stare < SWATH_STARES; stare++ )
	  if ( rad[track][stare][pixel][chan][1] > 0.0 ) 
	    switch ( pos[track][stare] ) {
	    case 1: dif1mn[pixel][chan] = dif1mn[pixel][chan] + rad[track][stare][pixel][chan][1];
	      imn1++;
	      break;
	    case 2: dif2mn[pixel][chan] = dif2mn[pixel][chan] + rad[track][stare][pixel][chan][1];
	      imn2++;
	      break;
	    case 3: dif3mn[pixel][chan] = dif3mn[pixel][chan] + rad[track][stare][pixel][chan][1];
	      imn3++;
	      break;
	    case 4:
	    case 5: mndif[pixel][chan] = mndif[pixel][chan] + rad[track][stare][pixel][chan][1] ;
	      idif++;
	      break;
	    }
      if ( idif > 0 && imn1 > 0 && imn2 > 0 && imn3 > 0 ) {
	mndif  [pixel][chan] = mndif  [pixel][chan] / idif;
	dif1mn [pixel][chan] = dif1mn [pixel][chan] / imn1;
	dif2mn [pixel][chan] = dif2mn [pixel][chan] / imn2;
	dif3mn [pixel][chan] = dif3mn [pixel][chan] / imn3;
	dif1mn [pixel][chan] = dif1mn [pixel][chan] - mndif [pixel][chan];
	dif2mn [pixel][chan] = dif2mn [pixel][chan] - mndif [pixel][chan];
	dif3mn [pixel][chan] = dif3mn [pixel][chan] - mndif [pixel][chan];
	
	for ( track = 0; track < trackcount; track++ ) 
	  for ( stare = 0; stare < SWATH_STARES; stare++ )
	    switch ( pos[track][stare] ) {
	    case 1: 
	      rad[track][stare][pixel][chan][1] = rad[track][stare][pixel][chan][1] - dif1mn[pixel][chan];
	      break;
	    case 2: 
	      rad[track][stare][pixel][chan][1] = rad[track][stare][pixel][chan][1] - dif2mn[pixel][chan];
	      break;
	    case 3: 
	      rad[track][stare][pixel][chan][1] = rad[track][stare][pixel][chan][1] - dif3mn[pixel][chan];
	      break;
	    }
      }
    } // end for chan 3 & 7 loop
  } // end pf pix out loop

  
  // Add Daily Mean Noise ( nstate, nchan, npix ) in MOP01 file 
  startn[0] = 0;            startn[1] = 0;              startn[2] = 0; 
  edgen [0] = SWATH_PIXELS; edgen [1] = SWATH_CHANNELS; edgen [2] = SWATH_RADIANCE_STATES; 

  ierr = HE5_SWwritefield( swid, "DailyMeanNoise", startn, NULL, edgen, dailymeannoise );
  if ( ierr != 0 )
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_MOP01_MODULE, REFERENCE_ERROR,
				 logical, 0, 0, 0, 0, 0, 0, 0, "Could not add daily mean noise");

  // Add Daily Gain Deviation (nstate, nchan, npix) in MOP01 file 
  ierr = HE5_SWwritefield( swid, "DailyGainDev", startn, NULL, edgen, dailygaindev );
  if ( ierr != 0 )
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_MOP01_MODULE, REFERENCE_ERROR,
				 logical, 0, 0, 0, 0, 0, 0, 0, "Could not add Daily gain dev");

  // update MOPITT PMC D Radiances in MOP01 file
  ierr = HE5_SWwritefield( swid, "MOPITTRadiances", start, NULL, edge, rad);
  if ( ierr != 0 )
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_SYSTEM_MODULE, REFERENCE_ERROR, 
				 logical, 0, 0, 0, 0, 0, 0, 0, "Could not update MOPITT Radiances");

  // define nTrack dimension scale
  hsize_t ntrk = trackcount;
  int dtrk[trackcount];
  for ( track = 0; track < trackcount; track++ )
    dtrk[track] = track + 1;

  ierr = HE5_SWdefdimscale (swid, "ntrack", ntrk, H5T_NATIVE_INT, dtrk);
  if ( ierr != 0 )
    diagnosticreporter.AddEntry (DIAGNOSTICS_ERROR, DIAGNOSTICS_SYSTEM_MODULE, REFERENCE_ERROR, 
				 logical, 0, 0, 0, 0, 0, 0, 0, "Could not define ntrack dime scale");
    
  (void) HE5_SWdetach (swid); 
  (void) HE5_SWclose(fileid);  
  
} 
