/********** IngestTelemetry.C *****************************************************************************************\

 $Header$

 PURPOSE
   Telemetry ingest driver for L1 processor.

 INPUT ARGUMENTS
   None

 INPUT/OUTPUT ARGUMENTS
   None

 OUTPUT ARGUMENTS
   swath_list&   swathlist   The Earth-view swaths container.
   qa_list&      qalist      The QA anomalies container.

 RETURNS
   ON SUCCESS
     None
   ON FAILURE
     None

 ASSUMPTIONS ON ENTRY
   FILES
     The MOPITT Instrument Parameters, L0 Science, L0 Engineering and L0 Table collections are available.
     and staged.
   OBJECTS
     None
   MISCELLANEOUS
     None

 ASSUMPTIONS ON EXIT
   FILES
     The MOPITT Calibration History file for this granule has been created.
   OBJECTS
     None
   ERROR HANDLING
     None
   MISCELLANEOUS
     None

 REVISION HISTORY
   12/98   Charles Cavanaugh
   07/99   Tom Lauren
   09/01   Debbie Mao

 $Log$

\**********************************************************************************************************************/

#include "Drivers.h"
#include "TableCollection.h"
#include "EngineeringCollection.h"
#include "MOPCHWriteFile.h"
#include "MOPIPFile.h"
#include "ScienceCollection.h"
#include "SciencePacket.h"
#include "ScienceStare.h"
#include "ScienceTrain.h"
#include "CalibrationCollections.h"
#include "EarthviewCollections.h"
#include "MOPCHGlobalMetadata.h"
#include "ScienceGlobalSectorMeans.h"

extern processor_parameters processorparameters;

// construct the mopch global metadata container
mopch_global_metadata mopchglobalmetadata;

// construct the global lmc sector offset container
science_global_sector_means globalsectormeans;

void IngestTelemetry (swath_list& swathlist, processor_parameters& processorparameters)
{
  // initialize the mopch output file
  mopch_write_file mopchfile;

  // ingest the table file data and write the data to mopch
  table_collection tablecollection;
  tablecollection.Ingest ();
  tablecollection.Write (mopchfile.GetFileID (), mopchfile.GetTableID ());

  // ingest the engineering file data and write the data to mopch
  engineering_collection engineeringcollection;
  engineeringcollection.Ingest ();
  engineeringcollection.Write (mopchfile.GetFileID (), mopchfile.GetEngineeringID ());

  // read the instrument parameters
  mopip_file mopipfile;
  mopipfile.Read ();

  // read the run-time parameters
  processorparameters.Read ();

  // setup the calibration science collections
  calibration_collections calibrations (processorparameters, engineeringcollection, mopipfile, mopchfile);

  // setup the earth view science collections
  earthview_collections earthviews (processorparameters, engineeringcollection, mopipfile, swathlist);

  // ingest the science file data: retrieve science trains and route them to the appropriate collection
  science_collection sciencecollection (tablecollection,processorparameters);
  science_train* sciencetrain;
  while ((sciencetrain = sciencecollection.GetTrain ()) != NULL)
    switch (sciencetrain->GetType ()) {
      case EARTHVIEW_TRAIN :   earthviews.AddTrain (sciencetrain);
                               break;
      case CALIBRATION_TRAIN : calibrations.AddTrain (sciencetrain);
                               break;
      default : break;
    }
}
