/********** DiagnosticReporter.C **************************************************************************************\

 $Header$

 REVISION HISTORY
   09/99   Charles Cavanaugh
   12/99   Charles Cavanaugh
   10/00   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include "DiagnosticReporter.h"

enum diagnostics_error_code const diagnostic_reporter::DEFAULT_THRESHOLD = DIAGNOSTICS_TERSE;
PGSt_PC_Logical const             diagnostic_reporter::FILE_THRESHOLD_LOGICALS [MODULE_CODE_COUNT] =
                                               { 41910, 41912, 41914, 41916, 41918, 41920, 41922, 41924, 41926, 41928, 
                                                 41930, 41932, 41934, 41936, 41938, 41940, 41942, 41944, 41946, 41948 };
PGSt_PC_Logical const             diagnostic_reporter::SCREEN_THRESHOLD_LOGICALS [MODULE_CODE_COUNT] = 
                                               { 41911, 41913, 41915, 41917, 41919, 41921, 41923, 41925, 41927, 41929, 
                                                 41931, 41933, 41935, 41937, 41939, 41941, 41943, 41945, 41947, 41949 };


diagnostic_reporter :: diagnostic_reporter ()
{
  Initialize ();
}


diagnostic_reporter :: ~diagnostic_reporter ()
{
  ;
}


void diagnostic_reporter :: AddHeaders (enum diagnostics_error_code errorcode, enum diagnostics_module_code modulecode,
                                        int diagnosticscode, string& message) const
{
  // add the error code and module code
  message += GetErrorCodeString (errorcode);
  message += GetModuleCodeString (modulecode);
  
  // add the diagnostics code
  char diagchars [4];
  (void) sprintf (diagchars, "%3d", diagnosticscode);
  message += diagchars;
}


void diagnostic_reporter :: AddIndeces (int file, int track, int packet, int stare, int train, int channel, int pixel, 
                                        int block, string& message) const
{
  // add the appropriate indeces to the message string (packet and pixel subtract off 1 because 1 is added on before
  // the call to this function [packet 0 and pixel 0 are valid entries, but 0 means "missing"])
  if (file > 0)
    AddIndex ("Fi", file, 5, message);
  if (track > 0)
    AddIndex ("Tk", track, 5, message);
  if (packet > 0)
    AddIndex ("Pa", packet - 1, 5, message);
  if (stare > 0)
    AddIndex ("St", stare, 2, message);
  if (train > 0)
    AddIndex ("Tn", train, 1, message);
  if (channel > 0)
    AddIndex ("Ch", channel, 1, message);
  if (pixel > 0)
    AddIndex ("Px", pixel - 1, 1, message);
  if (block > 0)
    AddIndex ("Bl", block, 1, message); 
}


void diagnostic_reporter :: AddIndex (string const& type, int value, int length, string& message) const
{
  message += "  ";
  message += type;
  char* charvalue = new char [length + 1];
  (void) sprintf (charvalue, "%d", value);
  message += charvalue;
  delete [] charvalue;
}


void diagnostic_reporter :: Exit () const
{
  // do whatever to exit the processor
  exit (EXIT_FAILURE);
}


void diagnostic_reporter :: Initialize ()
{
  isabnormal = false;
  isopen = false;
  handle = NULL;
  for (int i = 0; i < MODULE_CODE_COUNT; i++) {
    filethresholds [i] = DEFAULT_THRESHOLD;
    screenthresholds [i] = DEFAULT_THRESHOLD;
  }
}


void diagnostic_reporter :: ReadLogical (PGSt_PC_Logical logical, enum diagnostics_error_code& threshold) const
{
  // read the runtime parameter value (which is read in as a character string)
  char logicalchars [PGSd_PC_VALUE_LENGTH_MAX];
  if (PGS_PC_GetConfigData (logical, logicalchars) == PGS_S_SUCCESS)
    threshold = (enum diagnostics_error_code) atoi (logicalchars); 
}


void diagnostic_reporter :: WriteEntry (enum diagnostics_error_code errorcode, 
                                        enum diagnostics_error_code filethreshold, 
                                        enum diagnostics_error_code screenthreshold, string const& message) const
{
  // if appropriate, write to file
  if (errorcode <= filethreshold) {
    fputs (message.c_str (), handle);
    fputc ('\n', handle);
  }

  // if appropriate, write to screen
  if (errorcode <= screenthreshold)
    cout << message.c_str () << endl;
}


void diagnostic_reporter :: AddEntry (enum diagnostics_error_code errorcode, enum diagnostics_module_code modulecode, 
                                      int diagnosticscode, int file, int track, int packet, int stare, int train, 
                                      int channel, int pixel, int block, string const& diagnostic)
{
  // make a container for the diagnostic message
  string message;

  // add the diagnostic error type, module and code
  AddHeaders (errorcode, modulecode, diagnosticscode, message);

  // add the relevant indeces
  AddIndeces (file, track, packet, stare, train, channel, pixel, block, message);

  // add the diagnostic
  message += "  ";
  message += diagnostic;

  // write the diagnostic
  WriteEntry (errorcode, filethresholds [modulecode], screenthresholds [modulecode], message);

  // if errorcode denotes error, shut down the processor
  if (errorcode == DIAGNOSTICS_ERROR) {
    isabnormal = true;
    Close ();
    Exit ();
  }
}


void diagnostic_reporter :: Close ()
{
  // make sure the file is open
  if (isopen) {

    // the last entry should be either normal or abnormal termination
    string message = "1System000  Normal Termination";
    if (isabnormal)
      message = "ESystem999  Abnormal Termination";
    WriteEntry (DIAGNOSTICS_STATIC, DIAGNOSTICS_VERBOSE, DIAGNOSTICS_VERBOSE, message);

    // close the file
    if (PGS_IO_Gen_Close (handle) == PGS_S_SUCCESS)
      isopen = false;
    else {
      string report = "ESystem990  Could not close diagnostic file";
      (void) PGS_SMF_GenerateStatusReport ((char*) report.c_str ());
      cout << report.c_str () << endl;
    }
  }
}


void diagnostic_reporter :: Open (PGSt_PC_Logical filelogical)
{
  if (PGS_IO_Gen_Open (filelogical, PGSd_IO_Gen_Write, &handle, 1) == PGS_S_SUCCESS)
    isopen = true;
  else {
    char report [80];
    (void) sprintf (report, "%s %d\n", "ESystem990  Could not open diagnostic file using logical ", filelogical);
    (void) PGS_SMF_GenerateStatusReport (report);
    cout << report;
    Exit ();
  }
}


void diagnostic_reporter :: ReadThresholds ()
{
  for (int i = 0; i < MODULE_CODE_COUNT; i++) {
    ReadLogical (FILE_THRESHOLD_LOGICALS [i],   filethresholds [i]);
    ReadLogical (SCREEN_THRESHOLD_LOGICALS [i], screenthresholds [i]);
  }
}
