/********** CalibrationLMCPMCEvent.C **********************************************************************************\

 $Header$

 REVISION HISTORY
   02/99   Charles Cavanaugh
   11/99   Charles Cavanaugh
   05/00   Charles Cavanaugh
   11/00   Charles Cavanaugh
   4/01    Daniel Ziskin
   02/02   Debbie Mao
   04/02   Debbie Mao
   05/02   Debbie Mao
   06/02   Debbie Mao
   08/02   Debbie Mao

 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include "SciencePMCSignal.h"
#include "CalibrationLMCPMCEvent.h"
#include "DiagnosticReporter.h"

extern diagnostic_reporter diagnosticreporter;

calibration_lmcpmc_event :: calibration_lmcpmc_event (int train, engineering_collection& engineering,
                                                      mopip_file const& mopip)
                          : calibration_event (train, engineering, mopip)
{
  ;
}


calibration_lmcpmc_event :: ~calibration_lmcpmc_event ()
{
  ;
}


void calibration_lmcpmc_event :: ComputePMCSignals (double signals [SCIENCE_PIXELS] [PMC_SIGNAL_TERMS],
                                 double coldsignals [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_SIGNAL_STATES],
				 double warmsignals [SCIENCE_PIXELS] [WARMCAL_STDDEV_TERMS] [SCIENCE_PACKETPOSITION])
{
  int pixel, position;

  // get the event means
  bool isvalid [SCIENCE_PIXELS];
  double openmeans [SCIENCE_PIXELS] [PMC_STROKES], closedmeans [SCIENCE_PIXELS] [PMC_STROKES];
  double coldposopenmeans [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES], 
         coldposclosedmeans [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES];
  double warmposopenmeans [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES],
         warmposclosedmeans [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES],
         warmposopenvariances [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES],
         warmposclosedvariances [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES];

  GetPMCEventMeans (openmeans, closedmeans, coldposopenmeans, coldposclosedmeans, warmposopenmeans,
		    warmposclosedmeans, isvalid);
  // calculate cold cal position signals
  for ( position = 0; position < SCIENCE_PACKETPOSITION; position++ )
    for ( pixel = 0; pixel < SCIENCE_PIXELS; pixel++) 
      if ( coldposopenmeans [position] [pixel][PMC_HIGH_STROKE] != CALIBRATION_MISSING_VALUE &&
	   coldposclosedmeans [position] [pixel][PMC_HIGH_STROKE] != CALIBRATION_MISSING_VALUE &&
	   coldposopenmeans [position] [pixel][PMC_LOW_STROKE] != CALIBRATION_MISSING_VALUE &&
	   coldposclosedmeans [position] [pixel][PMC_LOW_STROKE] != CALIBRATION_MISSING_VALUE ) {
        double highsignal = coldposopenmeans [position] [pixel][PMC_HIGH_STROKE] - 
                            coldposclosedmeans [position] [pixel][PMC_HIGH_STROKE];
        double lowsignal  = coldposopenmeans [position] [pixel][PMC_LOW_STROKE]  - 
	                    coldposclosedmeans [position] [pixel][PMC_LOW_STROKE];
        coldsignals [position] [pixel] [PMC_SIGNAL_AVERAGE]    = (highsignal + lowsignal) / 2.0;
        coldsignals [position] [pixel] [PMC_SIGNAL_DIFFERENCE] = highsignal - lowsignal;
      }
      else {
	coldsignals [position] [pixel] [PMC_SIGNAL_AVERAGE]    = CALIBRATION_MISSING_VALUE;
	coldsignals [position] [pixel] [PMC_SIGNAL_DIFFERENCE] = CALIBRATION_MISSING_VALUE;
      }

  // get the event variances
  double openvariances [SCIENCE_PIXELS] [PMC_STROKES], closedvariances [SCIENCE_PIXELS] [PMC_STROKES];
  GetPMCEventVariances (openmeans, closedmeans, openvariances, closedvariances,
			warmposopenmeans, warmposclosedmeans, warmposopenvariances, warmposclosedvariances);

  // compute the event signals
  science_pmc_signal pmcsignal;
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++) 
    pmcsignal.ComputeSignal (openmeans [pixel], openvariances [pixel], closedmeans [pixel], closedvariances [pixel],
                             signals [pixel]);

  // compute the warm event signals' stddev 
  for (  position = 0; position < SCIENCE_PACKETPOSITION; position++ )
    for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++) 
      if ( warmposopenvariances [position] [pixel][PMC_HIGH_STROKE] == CALIBRATION_MISSING_VALUE ||
	   warmposclosedvariances [position] [pixel][PMC_HIGH_STROKE] == CALIBRATION_MISSING_VALUE ||
           ( warmposopenvariances [position] [pixel][PMC_HIGH_STROKE] ==  0.0 &&
	     warmposclosedvariances [position] [pixel][PMC_HIGH_STROKE] == 0.0 ) ||
	   warmposopenvariances [position] [pixel][PMC_LOW_STROKE] == CALIBRATION_MISSING_VALUE ||
	   warmposclosedvariances [position] [pixel][PMC_LOW_STROKE] == CALIBRATION_MISSING_VALUE ||  
	   ( warmposopenvariances [position] [pixel][PMC_LOW_STROKE] == 0.0 &&
	   warmposclosedvariances [position] [pixel][PMC_LOW_STROKE] == 0.0 )  ) {
	warmsignals [pixel] [PMC_SIGNAL_AVERAGE] [position]    = CALIBRATION_MISSING_VALUE;
	warmsignals [pixel] [PMC_SIGNAL_DIFFERENCE] [position] = CALIBRATION_MISSING_VALUE;
      }
      else {
	double highsignalvariance = warmposopenvariances  [position] [pixel] [PMC_HIGH_STROKE] + 
	  warmposclosedvariances  [position] [pixel] [PMC_HIGH_STROKE];
	double lowsignalvariance  = warmposopenvariances  [position] [pixel] [PMC_LOW_STROKE]  + 
	  warmposclosedvariances  [position] [pixel] [PMC_LOW_STROKE];
	warmsignals [pixel][WARMCAL_AVERAGE_STDDEV] [position]   = (highsignalvariance + lowsignalvariance)/4.0;
	warmsignals [pixel][WARMCAL_DIFFERENCE_STDDEV][position] = highsignalvariance + lowsignalvariance;
      }

  // set invalid pixels to missing
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
    if (! isvalid [pixel]) {
      for (int term = 0; term < PMC_SIGNAL_TERMS; term++) 
        signals [pixel] [term] = CALIBRATION_MISSING_VALUE;
      for (  position = 0; position < SCIENCE_PACKETPOSITION; position++ ) {
	for (int state = 0; state < PMC_SIGNAL_STATES; state++) 
	  coldsignals [position] [pixel] [state] = CALIBRATION_MISSING_VALUE;
	for ( int warmterm = 0; warmterm < WARMCAL_STDDEV_TERMS; warmterm++ )
	   warmsignals [pixel] [warmterm] [position] = CALIBRATION_MISSING_VALUE;
      }      
    }
}


void calibration_lmcpmc_event :: GetPMCEventMeans (double openmeans [SCIENCE_PIXELS] [PMC_STROKES],
                    double closedmeans [SCIENCE_PIXELS] [PMC_STROKES], 
		    double coldpositionopenmeans   [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES],
		    double coldpositionclosedmeans [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES],
		    double warmposopenmeans   [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES],
		    double warmposclosedmeans [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES],
                    bool isvalid [SCIENCE_PIXELS] )
{
  int pixel, stroke, position, pos;
  int warmposcounter [SCIENCE_PACKETPOSITION][SCIENCE_PIXELS];

  // zero out the means and set validity to true
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++) 
    for (stroke = 0; stroke < PMC_STROKES; stroke++) {
      openmeans   [pixel] [stroke] = 0.0;
      closedmeans [pixel] [stroke] = 0.0;
      isvalid     [pixel] = true;
      for ( pos = 0; pos < SCIENCE_PACKETPOSITION; pos++ ) {
	warmposopenmeans   [pos] [pixel] [stroke] = 0.0;
	warmposclosedmeans [pos] [pixel] [stroke] = 0.0;
	warmposcounter     [pos] [pixel] = 0;

	coldpositionopenmeans   [pos] [pixel] [stroke] = CALIBRATION_MISSING_VALUE;
	coldpositionclosedmeans [pos] [pixel] [stroke] = CALIBRATION_MISSING_VALUE;
      }
    }

  // for each train, accumulate the open and closed means for each stroke and pixel
  bool trainvalid [SCIENCE_PIXELS];
  int counter [SCIENCE_PIXELS] = {0, 0, 0, 0};
  double trainopenmeans [SCIENCE_PIXELS] [PMC_STROKES], trainclosedmeans [SCIENCE_PIXELS] [PMC_STROKES];
  science_train* sciencetrain;
  trains.ResetNext ();
  while ((sciencetrain = trains.GetNext ()) != NULL) {

    position = sciencetrain->GetPacketPosition() - 1;
    sciencetrain->GetMeans (CHANNEL_I, (double*) trainopenmeans, (double*) trainclosedmeans, trainvalid);

    for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
      if (trainvalid [pixel]) {
	for (stroke = 0; stroke < PMC_STROKES; stroke++) {
	  openmeans   [pixel] [stroke] += trainopenmeans   [pixel] [stroke];
	  closedmeans [pixel] [stroke] += trainclosedmeans [pixel] [stroke]; 

	  // calculte cold cal position open/closed means
	  if ( GetTrainDataType (engineeringcollection, mopipfile) == COLD_DATA_TYPE &&
	       ( position >= 0 && position < 5 ) ) {
	    coldpositionopenmeans   [position] [pixel] [stroke] = trainopenmeans   [pixel] [stroke];
	    coldpositionclosedmeans [position] [pixel] [stroke] = trainclosedmeans [pixel] [stroke];
	  }
	  else{
	    coldpositionopenmeans   [position] [pixel] [stroke] = CALIBRATION_MISSING_VALUE;
	    coldpositionclosedmeans [position] [pixel] [stroke] = CALIBRATION_MISSING_VALUE;
	  }	    
	  
	  // calculate warm cal position open/closed means
	  if ( GetTrainDataType (engineeringcollection, mopipfile) == WARM_DATA_TYPE && 
	       ( position >= 0 && position < 5 ) ) {
	    warmposopenmeans  [position] [pixel] [stroke] += trainopenmeans   [pixel] [stroke];
	    warmposclosedmeans[position] [pixel] [stroke] += trainclosedmeans [pixel] [stroke];
	    }	 
	}
	counter [pixel] += 1;

	if ( GetTrainDataType (engineeringcollection, mopipfile) == WARM_DATA_TYPE && 
	     ( position >= 0 && position < 5 ) ) 
	  warmposcounter [position] [pixel] += 1;
      }
  }

  // turn the sums into means
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++) {
    if (counter [pixel] == 0)
      isvalid [pixel] = false;
    else
      for (stroke = 0; stroke < PMC_STROKES; stroke++) {
        openmeans   [pixel] [stroke] /= (double) counter [pixel];
        closedmeans [pixel] [stroke] /= (double) counter [pixel];
      }
    for (pos = 0; pos < SCIENCE_PACKETPOSITION; pos++ ) 
      if (warmposcounter [pos] [pixel] > 1) 
	for (stroke = 0; stroke < PMC_STROKES; stroke++) {
	  warmposopenmeans   [pos][pixel][stroke] /= (double) warmposcounter [pos][pixel];
	  warmposclosedmeans [pos][pixel][stroke] /= (double) warmposcounter [pos][pixel];
	}
  }
}


void calibration_lmcpmc_event :: GetPMCEventVariances (double const openmeans [SCIENCE_PIXELS] [PMC_STROKES],
                                     double const closedmeans [SCIENCE_PIXELS] [PMC_STROKES],
                                     double openvariances [SCIENCE_PIXELS] [PMC_STROKES],
                                     double closedvariances [SCIENCE_PIXELS] [PMC_STROKES],
				     double const warmposopenmeans [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES], 
                                     double const warmposclosedmeans [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES], 
				     double warmposopenvariances  [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES],
				     double warmposclosedvariances [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_STROKES] )
{
  int pixel, stroke, position, pos;
  int warmposcounter [SCIENCE_PACKETPOSITION][SCIENCE_PIXELS];

  // zero out the variances
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
    for (stroke = 0; stroke < PMC_STROKES; stroke++) {
      openvariances   [pixel] [stroke] = 0.0;
      closedvariances [pixel] [stroke] = 0.0;
      for (pos = 0; pos < SCIENCE_PACKETPOSITION; pos++ ) {
	warmposopenvariances   [pos][pixel] [stroke] = 0.0;
	warmposclosedvariances [pos][pixel] [stroke] = 0.0;
	warmposcounter [pos][pixel] = 0;
      }
    }

  // for each train, accumulate the open and closed mean differences for each stroke and pixel
  bool trainvalid [SCIENCE_PIXELS];
  int counter [SCIENCE_PIXELS] = {0, 0, 0, 0};
  double trainopenmeans [SCIENCE_PIXELS] [PMC_STROKES], trainclosedmeans [SCIENCE_PIXELS] [PMC_STROKES];
  science_train* sciencetrain;
  trains.ResetNext ();
  while ((sciencetrain = trains.GetNext ()) != NULL) {

      sciencetrain->GetMeans (CHANNEL_I, (double*) trainopenmeans, (double*) trainclosedmeans, trainvalid);
      position = sciencetrain->GetPacketPosition() - 1;

      for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
	if (trainvalid [pixel]) {
	  for (stroke = 0; stroke < PMC_STROKES; stroke++) {
	    openvariances   [pixel] [stroke] +=
	      pow ((openmeans   [pixel] [stroke] - trainopenmeans   [pixel] [stroke]), 2.0);
	    closedvariances [pixel] [stroke] +=
	      pow ((closedmeans [pixel] [stroke] - trainclosedmeans [pixel] [stroke]), 2.0);
	    
	    if ( GetTrainDataType (engineeringcollection, mopipfile) == WARM_DATA_TYPE &&
		 ( position >= 0 && position < 5 ) ) {
	      warmposopenvariances [position][pixel][stroke] += 
		pow ((warmposopenmeans [position][pixel][stroke] - trainopenmeans [pixel][stroke]), 2.0); 
	      warmposclosedvariances [position][pixel][stroke] += 
		pow ((warmposclosedmeans [position][pixel][stroke] - trainclosedmeans [pixel][stroke]), 2.0);
	    }
	  }
	  counter [pixel] += 1;
	  if ( GetTrainDataType (engineeringcollection, mopipfile) == WARM_DATA_TYPE &&
	       ( position >= 0 && position < 5 ) ) 
	    warmposcounter [position] [pixel] += 1;
	}
  }
  
  // turn the sums into variances
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++) {
    if (counter [pixel] > 2)
      for (stroke = 0; stroke < PMC_STROKES; stroke++) {
	openvariances   [pixel] [stroke] /= (double) (counter [pixel] - 1);
	closedvariances [pixel] [stroke] /= (double) (counter [pixel] - 1);	
      }
    for (pos = 0; pos < SCIENCE_PACKETPOSITION; pos++ ) 
      if (warmposcounter [pos] [pixel] > 1 ) 
	for (stroke = 0; stroke < PMC_STROKES; stroke++) {
	  warmposopenvariances   [pos] [pixel] [stroke]   /= (double) warmposcounter [pos][pixel];
	  warmposclosedvariances [pos] [pixel] [stroke] /= (double) warmposcounter [pos][pixel];
	}
      else 
	for (stroke = 0; stroke < PMC_STROKES; stroke++) {
	  warmposopenvariances   [pos] [pixel] [stroke] = CALIBRATION_MISSING_VALUE; 
	  warmposclosedvariances [pos] [pixel] [stroke] = CALIBRATION_MISSING_VALUE; 
	} 
  }
}


bool calibration_lmcpmc_event :: MakeRecord (science_calibration_record* record)
{
  bool issignalszero = false;

  // compute the signals
  double signalsI  [SCIENCE_PIXELS] [PMC_SIGNAL_TERMS];
  double signalsII [SCIENCE_PIXELS] [LMC_SIGNAL_TERMS];
  double sectorsignalsII  [SCIENCE_PIXELS] [SECTOR_SIGNAL_TERMS];
  double coldsignals  [SCIENCE_PACKETPOSITION] [SCIENCE_PIXELS] [PMC_SIGNAL_STATES];
  double warmsignals  [SCIENCE_PIXELS] [WARMCAL_STDDEV_TERMS] [SCIENCE_PACKETPOSITION] ;

  ComputePMCSignals (signalsI, coldsignals, warmsignals);
  ComputeLMCSignals (CHANNEL_II, signalsII, sectorsignalsII);

  // set the record
  record->SetRecordData (trains.GetCount (), 
                         engineeringcollection.GetEventTemperature (train, starttime, stoptime, mopipfile), 
                         starttime.GetDay (), starttime.GetMsecs (), stoptime.GetMsecs (), 
                         GetShadowStatus (starttime.GetDay (), starttime.GetMsecs (), stoptime.GetMsecs ()), 
                         signalsI, signalsII, sectorsignalsII, coldsignals,warmsignals );

  if ( (signalsI [0][0] == 0.0 && signalsI [0][1] == 0.0) || ( signalsII [0][0] ==  0.0 && signalsII [0][1] == 0.0) ||
       (signalsI [1][0] == 0.0 && signalsI [1][1] == 0.0) || ( signalsII [1][0] ==  0.0 && signalsII [1][1] == 0.0) ||
       (signalsI [2][0] == 0.0 && signalsI [2][1] == 0.0) || ( signalsII [2][0] ==  0.0 && signalsII [2][1] == 0.0) ||
       (signalsI [3][0] == 0.0 && signalsI [3][1] == 0.0) || ( signalsII [3][0] ==  0.0 && signalsII [3][1] == 0.0) )
    issignalszero = true;
  return issignalszero;
    
}


bool calibration_lmcpmc_event :: AddTrain (science_train* sciencetrain, science_train_group* traingroup)
{
  bool isadded = true;

  if (DoesTrainBelong (sciencetrain))
    UpdateEvent (sciencetrain);
  else {
    isadded = Write (traingroup);
    ReInitializeEvent (sciencetrain);
  }

  return isadded;
}


bool calibration_lmcpmc_event :: Write (science_train_group* traingroup)
{
  bool iswritten = false, issignalszero = false;

  // process only if there are enough stares
  if (! HasEnoughStares ())
    WriteStareCountDiagnostic ();
  else {
    science_calibration_record* record = new science_calibration_lmcpmc_record ();
    issignalszero = MakeRecord (record);

    if ( issignalszero != true )
      if ((iswritten = traingroup->AddRecord (GetTrainDataType (engineeringcollection, mopipfile), record))) {
	unsigned short starecount, dummyshort;
	float temperature;
	int dummy;
	record->GetHeaderInfo (starecount, temperature, dummy, dummy, dummy, dummyshort);
	WriteInfo (starecount, temperature);
      }
  }
  
  return iswritten;
}
