/********** CalibrationEvent.C ****************************************************************************************\

 $Header$

 REVISION HISTORY
   02/99   Charles Cavanaugh
   06/99   Tom Lauren
   11/99   Charles Cavanaugh
   11/00   Charles Cavanaugh
   04/01   Daniel Ziskin
   02/02   Debbie Mao
   04/02   Debbie Mao
   04/18  Daniel Ziskin
 $Log$

\**********************************************************************************************************************/

#include <math.h>
#include "PGS_CBP.h"
#include "PGS_CSC.h"
#include "CalibrationEvent.h"
#include "ScienceLMCSignal.h"
#include "DiagnosticReporter.h"
#include "ScienceGlobalSectorMeans.h"

extern diagnostic_reporter diagnosticreporter;
extern science_global_sector_means globalsectormeans;

float          const calibration_event::CALIBRATION_EVENT_HOT_MINIMUM_TEMPERATURE  = 440.0;
float          const calibration_event::CALIBRATION_EVENT_WARM_MAXIMUM_TEMPERATURE = 340.0;
unsigned short const calibration_event::CALIBRATION_EVENT_MINIMUM_TRAIN_COUNT      = 5;
unsigned short const calibration_event::CALIBRATION_EVENT_MAXIMUM_TRAIN_COUNT      = 400;
int            const calibration_event::CALIBRATION_EVENT_MSECS_GAP_TOLERANCE      = (STARE_MSECS_GAP * 3);
double         const calibration_event::SHADOWVECTOR_MINIMUM                       = .55;
int            const calibration_event::STARECOUNT_WARNING                         = 800;

calibration_event :: calibration_event (int trainnumber, engineering_collection& engineering, mopip_file const& mopip)
                   : starttime ()
                   , stoptime ()
                   , trains ()
                   , engineeringcollection (engineering)
                   , mopipfile (mopip)
{
  train = trainnumber;
  mirrorposition = 0;
}


calibration_event :: ~calibration_event ()
{
  ;
}


void calibration_event :: ComputeLMCSignals (enum science_channel_number channelnumber,
                                             double signals [SCIENCE_PIXELS] [LMC_SIGNAL_TERMS],
					     double secsignals [SCIENCE_PIXELS] [SECTOR_SIGNAL_TERMS])
{
  int pixel;

  // get the event means
  bool isvalid [SCIENCE_PIXELS];
  double openmeans [SCIENCE_PIXELS] [LMC_SECTORS], closedmeans [SCIENCE_PIXELS] [LMC_SECTORS];
  GetLMCEventMeans (channelnumber, openmeans, closedmeans, isvalid);

  // get the event variances
  double openvariances [SCIENCE_PIXELS] [LMC_SECTORS], closedvariances [SCIENCE_PIXELS] [LMC_SECTORS];
  GetLMCEventVariances (channelnumber, openmeans, closedmeans, openvariances, closedvariances);

  // compute the event signals
  science_lmc_signal lmcsignal;
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++) 
    lmcsignal.ComputeSignal (openmeans [pixel], openvariances [pixel], closedmeans [pixel], closedvariances [pixel], 
                             signals [pixel], secsignals [pixel] );

  // set invalid pixels to missing
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
    if (! isvalid [pixel]) {
      for (int term = 0; term < LMC_SIGNAL_TERMS; term++) 
        signals [pixel] [term] = CALIBRATION_MISSING_VALUE;
      for (int secterm = 0; secterm < SECTOR_SIGNAL_TERMS; secterm++) 
        secsignals [pixel] [secterm] = CALIBRATION_MISSING_VALUE;
    }      

  // set the global sector container
  if (channelnumber == CHANNEL_II && isvalid [0] && isvalid [1] && isvalid [2] && isvalid [3] &&
      GetShadowStatus (starttime.GetDay (), starttime.GetMsecs (), stoptime.GetMsecs())) {
    double sectormeans [SCIENCE_PIXELS] [LMC_SECTORS];
    double sectornoise [SCIENCE_PIXELS] [LMC_SECTORS];
    for (int pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
      for (int sector = 0; sector < LMC_SECTORS; sector++) {
        sectormeans [pixel] [sector] = openmeans [pixel] [sector] - closedmeans [pixel] [sector];
        sectornoise [pixel] [sector] = openvariances [pixel] [sector] + closedvariances [pixel] [sector];
      }
    globalsectormeans.Set ((train * 2), sectormeans, sectornoise);
  }
}


void calibration_event :: GetLMCEventMeans (enum science_channel_number channelnumber,  
                                            double openmeans [SCIENCE_PIXELS] [LMC_SECTORS],
                                            double closedmeans [SCIENCE_PIXELS] [LMC_SECTORS],
                                            bool isvalid [SCIENCE_PIXELS])
{
  int pixel, sector;

  // zero out the means and set validity to true
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
    for (sector = 0; sector < LMC_SECTORS; sector++) {
      openmeans   [pixel] [sector] = 0.0;
      closedmeans [pixel] [sector] = 0.0;
      isvalid     [pixel]          = true;
    }

  // for each train, accumulate the open and closed means for each sector and pixel
  bool trainvalid [SCIENCE_PIXELS];
  int counter [SCIENCE_PIXELS] = {0, 0, 0, 0};
  double trainopenmeans [SCIENCE_PIXELS] [LMC_SECTORS], trainclosedmeans [SCIENCE_PIXELS] [LMC_SECTORS];
  science_train* sciencetrain;
  trains.ResetNext ();
  while ((sciencetrain = trains.GetNext ()) != NULL) {

    sciencetrain->GetMeans (channelnumber, (double*) trainopenmeans, (double*) trainclosedmeans, trainvalid);
    for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
      if (trainvalid [pixel]) {
	for (sector = 0; sector < LMC_SECTORS; sector++) {
	  openmeans   [pixel] [sector] += trainopenmeans   [pixel] [sector];
	  closedmeans [pixel] [sector] += trainclosedmeans [pixel] [sector];
	}
	counter [pixel] += 1;
      }
  }

  // turn the sums into means
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
    if (counter [pixel] == 0)
      isvalid [pixel] = false;
    else
      for (sector = 0; sector < LMC_SECTORS; sector++) {
        openmeans   [pixel] [sector] /= (double) counter [pixel];
        closedmeans [pixel] [sector] /= (double) counter [pixel];
      }
}


void calibration_event :: GetLMCEventVariances (enum science_channel_number channelnumber,  
                                                double const openmeans [SCIENCE_PIXELS] [LMC_SECTORS],
                                                double const closedmeans [SCIENCE_PIXELS] [LMC_SECTORS],
                                                double openvariances [SCIENCE_PIXELS] [LMC_SECTORS],
                                                double closedvariances [SCIENCE_PIXELS] [LMC_SECTORS])
{
  int pixel, sector;

  // zero out the variances
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
    for (sector = 0; sector < LMC_SECTORS; sector++) {
      openvariances   [pixel] [sector] = 0.0;
      closedvariances [pixel] [sector] = 0.0;
    }

  // for each train, accumulate the open and closed mean differences for each sector and pixel
  bool trainvalid [SCIENCE_PIXELS];
  int counter [SCIENCE_PIXELS] = {0, 0, 0, 0};
  double trainopenmeans [SCIENCE_PIXELS] [LMC_SECTORS], trainclosedmeans [SCIENCE_PIXELS] [LMC_SECTORS];
  science_train* sciencetrain;
  trains.ResetNext ();
  while ((sciencetrain = trains.GetNext ()) != NULL) {

    sciencetrain->GetMeans (channelnumber, (double*) trainopenmeans, (double*) trainclosedmeans, trainvalid);
    for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
      if (trainvalid [pixel]) {
	for (sector = 0; sector < LMC_SECTORS; sector++) {
	  openvariances   [pixel] [sector] += 
	    pow ((openmeans   [pixel] [sector] - trainopenmeans   [pixel] [sector]), 2.0);
	  closedvariances [pixel] [sector] += 
	    pow ((closedmeans [pixel] [sector] - trainclosedmeans [pixel] [sector]), 2.0);
	}
	counter [pixel] += 1;
      }
  }

  // turn the sums into variances
  for (pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
    if (counter [pixel] > 2)
      for (sector = 0; sector < LMC_SECTORS; sector++) {
        openvariances   [pixel] [sector] /= (double) (counter [pixel] - 1);
        closedvariances [pixel] [sector] /= (double) (counter [pixel] - 1);
      }
}


bool calibration_event :: DoesTrainBelong (science_train* sciencetrain) const
{
  bool doesbelong = false;

  // if there are no trains in the event, the train belongs
  int starecount = trains.GetCount ();
  if (starecount == 0)
    doesbelong = true;

  // train belongs if train count is less than maximum and mirror positions match and time gap is within tolerance
  else if (starecount < CALIBRATION_EVENT_MAXIMUM_TRAIN_COUNT && 
           sciencetrain->GetMirrorPosition () == mirrorposition &&
           ((int) (((sciencetrain->GetTime ()).GetTAI () - stoptime.GetTAI ()) * 1000.0)) < 
            CALIBRATION_EVENT_MSECS_GAP_TOLERANCE)
    doesbelong = true;

  return doesbelong;
}


bool calibration_event :: GetShadowStatus (int day, int startmsecs, int stopmsecs) const
{
  // set default return value to false
  bool isinshadow = false;

  // call the toolkit routine that points from the spacecraft to the sun
  PGSt_double sunvectors [2][3];
  PGSt_double timeoffsets [2] = {0.0, ((stopmsecs - startmsecs) / 1000.0)};
  mopitt_time time1 (day, startmsecs);
  string time1utc = time1.GetUTC ();
  PGSt_SMF_status stat = PGS_CBP_Sat_CB_Vector (PGSd_EOS_AM, 2, (char*) time1utc.c_str (), timeoffsets, 11, sunvectors);

  // test against the unitized Z component 
  double mag0 = (PGSt_double) sqrt ((sunvectors [0][0] * sunvectors [0][0]) + (sunvectors [0][1] * sunvectors [0][1]) +
                                    (sunvectors [0][2] * sunvectors [0][2]));
  double mag1 = (PGSt_double) sqrt ((sunvectors [1][0] * sunvectors [1][0]) + (sunvectors [1][1] * sunvectors [1][1]) +
                                    (sunvectors [1][2] * sunvectors [1][2]));

  // DIVIDE BY ZERO CHECK ADDED BY ZISKIN Apr 2018 FOR V8  
  if ((mag0 != 0) && (mag1 != 0))
    if (((sunvectors [0][2] / mag0) > SHADOWVECTOR_MINIMUM) && ((sunvectors [1][2] / mag1) > SHADOWVECTOR_MINIMUM))
      isinshadow = true;


  // shadowflag diagnostics
  // char shagMsg[200];
  // (void) sprintf (shagMsg,
  //	  "SHADOW Calibration Event for train %d %d mirror, %3.6f mag0, %3.6f mag1, %d isinshadow",
  //                train, mirrorposition,(sunvectors[0][2]/mag0), (sunvectors[1][2]/mag1), isinshadow);
  //  string shadMsg = shagMsg; 
  // diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_CALIBRATIONS_MODULE, STARECOUNT_WARNING, 0, 0, 0, 0,
  //		       train, 0, 0, 0, shadMsg);

  return isinshadow;
}


enum science_train_data_type calibration_event :: GetTrainDataType (engineering_collection& engineeringcollection, 
                                                                    mopip_file const& mopipfile)
{
  enum science_train_data_type datatype;

  if (mirrorposition == SPACE_CALIBRATION_MIRROR_POSITION)
    datatype = COLD_DATA_TYPE;
  else {
    float temperature = engineeringcollection.GetEventTemperature (train, starttime, stoptime, mopipfile);
    if (temperature <= CALIBRATION_EVENT_WARM_MAXIMUM_TEMPERATURE)
      datatype = WARM_DATA_TYPE;
    else if (temperature >= CALIBRATION_EVENT_HOT_MINIMUM_TEMPERATURE)
      datatype = HOT_DATA_TYPE;
    else
      datatype = TEPID_DATA_TYPE;
  }

  return datatype;
}


void calibration_event :: PolyRegressionFit (double* const x, double* const y, int count, double* c)
{
  double x1 = 0;
  double x2 = 0;
  double x3 = 0;
  double x4 = 0;
  double y1 = 0;
  double x1y1 = 0;
  double x2y1 = 0;
  double N = 0;

  int i;

  // Since x[i] and y[i] are referenced alot, these two values are optimized for speed.
  register double xi;
  register double yi;

  for (i = 0; i < count; i++) {
    xi = x[i];
    yi = y[i];
    x1 += xi;
    x2 += xi * xi;
    x3 += xi * xi * xi;
    x4 += xi * xi * xi * xi;
    y1 += yi;
    x1y1 += xi * yi;
    x2y1 += xi * xi * yi;
    N++;
  }

  double d = Determ (N, x1, x2, x1, x2, x3, x2, x3, x4);

  // Note: May want to check if (fabs(d) < very small value) before using it as a divisor.
  // In general, (d == very small value) when count < 3.  

  c[0] = Determ (y1, x1, x2, x1y1, x2, x3, x2y1, x3, x4) / d;
  c[1] = Determ (N, y1, x2, x1, x1y1, x3, x2, x2y1, x4) / d;
  c[2] = Determ (N, x1, y1, x1, x2, x1y1, x2, x3, x2y1) / d;
}


void calibration_event :: ReInitializeEvent (science_train* sciencetrain)
{
  // clear out the old event and update the new event
  trains.Empty ();
  UpdateEvent (sciencetrain);
}


void calibration_event :: UpdateEvent (science_train* sciencetrain)
{
  // add the train
  trains.AddTrain (sciencetrain);

  // if this is the first train, set the start time and mirror position
  if (trains.GetCount () == 1) {
    starttime = sciencetrain->GetTime ();
    mirrorposition = sciencetrain->GetMirrorPosition ();
  }

  // adjust the stop time (to the end of the train)
  stoptime.Set (((sciencetrain->GetTime ()).GetTAI () + ((double) STARE_MSECS_GAP / 1000.0)));
  if (stoptime.GetMsecs () < starttime.GetMsecs ())
    stoptime.Set (starttime.GetDay (), 86400000);
}


void calibration_event :: WriteInfo (unsigned short starecount, float temperature) const
{
  char info [170];
  string startutc = starttime.GetUTC ();
  string stoputc = stoptime.GetUTC ();
  (void) sprintf (info,
                "Calibration Event for train %d : %d stares, %d mirror, %3.6f temperature, %s start time, %s stop time",
                  train, (int) starecount, mirrorposition, temperature, startutc.c_str (), stoputc.c_str ());
  (void) PGS_SMF_GenerateStatusReport (info);
}


void calibration_event :: WriteStareCountDiagnostic () const
{
  char countchars [10];
  (void) sprintf (countchars, "%d", trains.GetCount ());
  string message = "Not enough stares (";
  message += countchars;
  message += ") for calibration event with start time ";
  message += starttime.GetUTC ();
  diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_CALIBRATIONS_MODULE, STARECOUNT_WARNING, 0, 0, 0, 0,
                               train, 0, 0, 0, message);
}
