/********** CalibrationCollections.C **********************************************************************************\

 $Header$

 REVISION HISTORY
   12/98   Charles Cavanaugh
   10/00   Charles Cavanaugh

 $Log$

\**********************************************************************************************************************/

#include "CalibrationLMCEvent.h"
#include "CalibrationLMCPMCEvent.h"
#include "ScienceTrainLMCGroup.h"
#include "ScienceTrainLMCPMCGroup.h"
#include "CalibrationCollections.h"
#include "MOPCHGlobalMetadata.h"
#include "DiagnosticReporter.h"

extern diagnostic_reporter diagnosticreporter;

int const calibration_collections::TIME_TERSE = 100;

calibration_collections :: calibration_collections (processor_parameters const& parameters, 
                                                    engineering_collection& engineering, mopip_file const& mopip, 
                                                    mopch_write_file& mopch)
                         : processorparameters (parameters)
{
  Initialize (engineering, mopip, mopch);
}


calibration_collections :: ~calibration_collections ()
{
  Finalize ();
}


bool calibration_collections :: Initialize (engineering_collection& engineeringcollection, 
                                            mopip_file const& mopipfile, mopch_write_file& mopchfile)
{
  bool isinitialized = false;

  // initialize the calibration events
  calibrationevent [0] = new calibration_lmc_event    (1, engineeringcollection, mopipfile);
  calibrationevent [1] = new calibration_lmcpmc_event (2, engineeringcollection, mopipfile);
  calibrationevent [2] = new calibration_lmc_event    (3, engineeringcollection, mopipfile);
  calibrationevent [3] = new calibration_lmcpmc_event (4, engineeringcollection, mopipfile);

  // initialize the train groups
  traingroup [0] = new science_train_lmc_group    (1, mopchfile.GetScienceID (), mopchfile.GetFileID ());
  traingroup [1] = new science_train_lmcpmc_group (2, mopchfile.GetScienceID (), mopchfile.GetFileID ());
  traingroup [2] = new science_train_lmc_group    (3, mopchfile.GetScienceID (), mopchfile.GetFileID ());
  traingroup [3] = new science_train_lmcpmc_group (4, mopchfile.GetScienceID (), mopchfile.GetFileID ());

  // define the structure of the train groups
  isinitialized = true;
  for (int train = 0; train < SCIENCE_TRAINS; train++)
    if (! traingroup [train] ->Define ())
      isinitialized = false;

  return isinitialized;
}


bool calibration_collections :: Finalize ()
{
  bool isfinalized = true;

  // flush out the calibration events, extract metadata info, close the train groups and delete the objects
  for (int train = 0; train < SCIENCE_TRAINS; train++) {
    if (! calibrationevent [train] ->Write (traingroup [train]))
      isfinalized = false;
    SetEventCounts ((train + 1), traingroup [train]);
    if (! traingroup [train] ->Close ())
      isfinalized = false;
    delete calibrationevent [train];
    delete traingroup [train];
  }

  return isfinalized;
}


void calibration_collections :: SetEventCounts (int train, science_train_group* traingroup) const
{
  // get the event counts for the train group
  int coldcounts = traingroup->GetEventCount (COLD_DATA_TYPE);
  int warmcounts = traingroup->GetEventCount (WARM_DATA_TYPE);
  int hotcounts  = traingroup->GetEventCount (HOT_DATA_TYPE);

  // set the metadata
  extern mopch_global_metadata mopchglobalmetadata;
  mopchglobalmetadata.SetEventDesc ("Normal granule");
  switch (train) {
    case 1 : mopchglobalmetadata.SetCold1Cals (coldcounts);
             mopchglobalmetadata.SetWarm1Cals (warmcounts);
             mopchglobalmetadata.SetHot1Cals  (hotcounts);
             break;
    case 2 : mopchglobalmetadata.SetCold2Cals (coldcounts);
             mopchglobalmetadata.SetWarm2Cals (warmcounts);
             mopchglobalmetadata.SetHot2Cals  (hotcounts);
             break;
    case 3 : mopchglobalmetadata.SetCold3Cals (coldcounts);
             mopchglobalmetadata.SetWarm3Cals (warmcounts);
             mopchglobalmetadata.SetHot3Cals  (hotcounts);
             break;
    case 4 : mopchglobalmetadata.SetCold4Cals (coldcounts);
             mopchglobalmetadata.SetWarm4Cals (warmcounts);
             mopchglobalmetadata.SetHot4Cals  (hotcounts);
             break;
    default : break;
  }
}


bool calibration_collections :: AddTrain (science_train* sciencetrain)
{
  bool isadded = true;

  // get the train's time and id
  mopitt_time traintime = sciencetrain->GetTime ();
  int train = sciencetrain->GetID () - 1;

  // if the train time is within the processing time range, add it to the appropriate event
  if (traintime >= processorparameters.GetStartTime () && traintime <= processorparameters.GetStopTime ()) {
    if (! calibrationevent [train] ->AddTrain (sciencetrain, traingroup [train]))
      isadded = false;
  }
  else {
    delete sciencetrain;
    string message = "Science train time outside of processing boundaries : ";
    message += traintime.GetUTC ();
    diagnosticreporter.AddEntry (DIAGNOSTICS_TERSE, DIAGNOSTICS_CALIBRATIONS_MODULE, TIME_TERSE, 0, 0, 0, 0, 
                                 (train + 1), 0, 0, 0, message);
  }

  return isadded;
}
