/********** CalibrationChannelPMCHistory.C ****************************************************************************\

 $Header$

 REVISION HISTORY
   06/99   Charles Cavanaugh
   11/99   Charles Cavanaugh
   09/00   Charles Cavanaugh
   10/00   Charles Cavanaugh
   04/02   Debbie Mao
   05/02   Debbie Mao

 $Log$

\**********************************************************************************************************************/

#include "ScienceCalibrationLMCPMCRecord.h"
#include "CalibrationChannelPMCHistory.h"
#include "DiagnosticReporter.h"
 
 extern diagnostic_reporter diagnosticreporter;

calibration_channel_pmc_history :: calibration_channel_pmc_history ()
                                 : calibration_channel_history ()
{
  internalrecord = new science_calibration_lmcpmc_record ();
  spacerecord    = new science_calibration_lmcpmc_record ();
}


calibration_channel_pmc_history :: calibration_channel_pmc_history (int channel)
                                 : calibration_channel_history (channel)
{
  internalrecord = new science_calibration_lmcpmc_record ();
  spacerecord    = new science_calibration_lmcpmc_record ();
}


calibration_channel_pmc_history :: ~calibration_channel_pmc_history ()
{
  ;
}


void calibration_channel_pmc_history :: FillGainRecord (mopir_collection const& mopircollection, 
                                                        enum science_train_data_type datatype,
                                                        calibration_channel_record* gainrecord,
							calibration_channel_noise_record* noiserecord)
{
  // until we know how to compute the noise, just load in Jinxue's numbers for all pixels
  gainrecord->AddNoise (CALIBRATION_STATE_AVERAGE,    NOISEVALUES [channel - 1] [CALIBRATION_STATE_AVERAGE],    0.0);
  gainrecord->AddNoise (CALIBRATION_STATE_DIFFERENCE, NOISEVALUES [channel - 1] [CALIBRATION_STATE_DIFFERENCE], 0.0);

  // for each pixel in the channel, retrieve the internal average and difference radiances,
  // calculate the mean and variance in gain and store them in the new record
  float avgradiancemean, avgradiancevariance, difradiancemean, difradiancevariance, gainmean, gainvariance;
  float posnoise;

  for (int pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) {
    mopircollection.GetRadiance (internalrecord->GetTemperature (), channel, pixel, RADIANCE_AVERAGE, avgradiancemean);
    avgradiancevariance = 0.0;
    gainrecord->AddRadiance (pixel, CALIBRATION_STATE_AVERAGE, avgradiancemean, avgradiancevariance);
    CalculateGain (avgradiancemean, avgradiancevariance, 
                   internalrecord->GetPMCSignal (pixel, PMC_SIGNAL_AVERAGE_AVERAGE),
                   internalrecord->GetPMCSignal (pixel, PMC_SIGNAL_AVERAGE_VARIANCE),
                   spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_AVERAGE_AVERAGE),
                   spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_AVERAGE_VARIANCE), gainmean, gainvariance);
    gainrecord->AddGain (pixel, CALIBRATION_STATE_AVERAGE, gainmean, gainvariance);

    int position;
    // compute packet position average noise
    for ( position =0; position < CALIBRATION_PACKETPOSITION; position++ ) {
      CalculatePosNoise ( gainmean, internalrecord->GetWarmStddev ( position, pixel, WARMCAL_AVERAGE_STDDEV),
			  posnoise );
      noiserecord->AddPosNoise ( position, pixel, POSITION_NOISE_AVERAGE, posnoise );
    }

    mopircollection.GetRadiance (internalrecord->GetTemperature (), channel, pixel, RADIANCE_DIFFERENCE, 
                                 difradiancemean);
    difradiancevariance = 0.0;
    gainrecord->AddRadiance (pixel, CALIBRATION_STATE_DIFFERENCE, difradiancemean, difradiancevariance);
    CalculateGain (difradiancemean, difradiancevariance, 
                   internalrecord->GetPMCSignal (pixel, PMC_SIGNAL_DIFFERENCE_AVERAGE),
                   internalrecord->GetPMCSignal (pixel, PMC_SIGNAL_DIFFERENCE_VARIANCE),
                   spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_DIFFERENCE_AVERAGE),
                   spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_DIFFERENCE_VARIANCE), gainmean, gainvariance);
    gainrecord->AddGain (pixel, CALIBRATION_STATE_DIFFERENCE, gainmean, gainvariance);

    // compute packet position difference noise
    for ( position =0; position < CALIBRATION_PACKETPOSITION; position++ ) {
      CalculatePosNoise ( gainmean, internalrecord->GetWarmStddev(position,pixel,WARMCAL_DIFFERENCE_STDDEV),
			  posnoise );
      noiserecord->AddPosNoise ( position, pixel, POSITION_NOISE_DIFFERENCE, posnoise );
    }
  }
}


void calibration_channel_pmc_history :: InterpolateOffsetData (calibration_channel_record* prerecord,
                                                               calibration_channel_record* postrecord,
                                                               enum science_train_data_type datatype,
                                                               calibration_channel_record* offsetrecord)
{
  float gainmean, gainerror, offsetmean, offseterror;

  // get the before and after times, and the offset record time
  mopitt_time beforetime = prerecord->GetTime ();
  mopitt_time aftertime = postrecord->GetTime ();
  mopitt_time offsettime = spacerecord->GetStartTime ();

  // for each pixel/state, interpolate and store the gain, radiance and noise, then calculate the offset
  float beforemean, beforeerror, aftermean, aftererror, datapointmean, datapointerror;
  for (int pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) {
    // read the gain out of the records, interpolate to track time, save interpolated gain
    prerecord->GetGain (pixel, CALIBRATION_STATE_AVERAGE, beforemean, beforeerror);
    postrecord->GetGain  (pixel, CALIBRATION_STATE_AVERAGE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddGain (pixel, CALIBRATION_STATE_AVERAGE, datapointmean, datapointerror);
    prerecord->GetGain (pixel, CALIBRATION_STATE_DIFFERENCE, beforemean, beforeerror);
    postrecord->GetGain  (pixel, CALIBRATION_STATE_DIFFERENCE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddGain (pixel, CALIBRATION_STATE_DIFFERENCE, datapointmean, datapointerror);

    // read the radiance out of the records, interpolate to track time, save interpolated radiance
    prerecord->GetRadiance (pixel, CALIBRATION_STATE_AVERAGE, beforemean, beforeerror);
    postrecord->GetRadiance  (pixel, CALIBRATION_STATE_AVERAGE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddRadiance (pixel, CALIBRATION_STATE_AVERAGE, datapointmean, datapointerror);
    prerecord->GetRadiance (pixel, CALIBRATION_STATE_DIFFERENCE, beforemean, beforeerror);
    postrecord->GetRadiance  (pixel, CALIBRATION_STATE_DIFFERENCE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddRadiance (pixel, CALIBRATION_STATE_DIFFERENCE, datapointmean, datapointerror);

    //  get the interpolated gain, calculate the offset and noise, save the offset and noise
    offsetrecord->GetGain (pixel, CALIBRATION_STATE_AVERAGE, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_AVERAGE_AVERAGE),
                     spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_AVERAGE_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_AVERAGE, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_AVERAGE_VARIANCE), 
                    datapointmean, datapointerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_AVERAGE, datapointmean, datapointerror);

    offsetrecord->GetGain (pixel, CALIBRATION_STATE_DIFFERENCE, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_DIFFERENCE_AVERAGE),
                     spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_DIFFERENCE_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_DIFFERENCE, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_DIFFERENCE_VARIANCE), 
                    datapointmean, datapointerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_DIFFERENCE, datapointmean, datapointerror);
  }
}


void calibration_channel_pmc_history :: TransferOffsetData (calibration_channel_record* gainrecord,
                                                            enum science_train_data_type datatype,
                                                            calibration_channel_record* offsetrecord)
{
  // transfer the data from the gain record to the offset record
  float recorddata [CALIBRATION_PIXELS] [CALIBRATION_STATES] [CALIBRATION_TERMS];
  gainrecord->GetData (recorddata);
  offsetrecord->AddData (recorddata);

  // calculate the offsets
  float gainmean, gainerror, offsetmean, offseterror, noisemean, noiseerror;
  for (int pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) {
    gainrecord->GetGain (pixel, CALIBRATION_STATE_AVERAGE, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_AVERAGE_AVERAGE),
                     spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_AVERAGE_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_AVERAGE, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_AVERAGE_VARIANCE), 
                    noisemean, noiseerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_AVERAGE, noisemean, noiseerror);

    gainrecord->GetGain (pixel, CALIBRATION_STATE_DIFFERENCE, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_DIFFERENCE_AVERAGE),
                     spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_DIFFERENCE_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_DIFFERENCE, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetPMCSignal (pixel, PMC_SIGNAL_DIFFERENCE_VARIANCE), 
                    noisemean, noiseerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_DIFFERENCE, noisemean, noiseerror);
  }
}
