/********** CalibrationChannelLMCHistory.C ****************************************************************************\

 $Header$

 REVISION HISTORY
   06/99   Charles Cavanaugh
   11/99   Charles Cavanaugh
   09/00   Charles Cavanaugh
   10/00   Charles Cavanaugh
   11/01   Debbie Mao
   02/02   Debbie Mao

 $Log$

\**********************************************************************************************************************/

#include "ScienceCalibrationLMCRecord.h"
#include "ScienceCalibrationLMCPMCRecord.h"
#include "CalibrationChannelLMCHistory.h"
#include "DiagnosticReporter.h"

extern diagnostic_reporter diagnosticreporter;

calibration_channel_lmc_history :: calibration_channel_lmc_history ()
                                 : calibration_channel_history ()
{
  internalrecord = new science_calibration_lmc_record ();
  spacerecord    = new science_calibration_lmc_record ();
}


calibration_channel_lmc_history :: calibration_channel_lmc_history (int channel)
                                 : calibration_channel_history (channel)
{
  if (channel == 4 || channel == 8) {
    internalrecord = new science_calibration_lmcpmc_record ();
    spacerecord    = new science_calibration_lmcpmc_record ();
  }
  else {
    internalrecord = new science_calibration_lmc_record ();
    spacerecord    = new science_calibration_lmc_record ();
  }
}


calibration_channel_lmc_history :: ~calibration_channel_lmc_history ()
{
  ;
}


void calibration_channel_lmc_history :: FillGainRecord (mopir_collection const& mopircollection, 
                                                        enum science_train_data_type datatype,
                                                        calibration_channel_record* gainrecord)
{
  // until we know how to compute the noise, just load in Jinxue's numbers for all pixels
  gainrecord->AddNoise (CALIBRATION_STATE_AVERAGE, NOISEVALUES [channel - 1] [CALIBRATION_STATE_AVERAGE],  0.0);
  gainrecord->AddNoise (CALIBRATION_STATE_DIFFERENCE, NOISEVALUES [channel - 1] [CALIBRATION_STATE_DIFFERENCE], 0.0);

  // for each pixel in the channel, retrieve the internal average and difference radiances,
  // calculate the mean and variance in gain and store them in the new record
  float avgradiancemean, avgradiancevariance, difradiancemean, difradiancevariance, gainmean, gainvariance;
  float longmean, longvariance,shortmean, shortvariance;

  for (int pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) {
    mopircollection.GetRadiance (internalrecord->GetTemperature (), channel, pixel, RADIANCE_AVERAGE, avgradiancemean);
    avgradiancevariance = 0.0;
    mopircollection.GetRadiance (internalrecord->GetTemperature(), channel, pixel, RADIANCE_DIFFERENCE,difradiancemean);
    difradiancevariance = 0.0;
    gainrecord->AddRadiance (pixel, CALIBRATION_STATE_AVERAGE, avgradiancemean, avgradiancevariance);
    gainrecord->AddRadiance (pixel, CALIBRATION_STATE_DIFFERENCE, difradiancemean, difradiancevariance);

    //transfer internal radiance from A/D to L/S
    if ( avgradiancemean <= CALIBRATION_MISSING_VALUE || difradiancemean <= CALIBRATION_MISSING_VALUE ) {
      longmean  = CALIBRATION_MISSING_VALUE;
      shortmean = CALIBRATION_MISSING_VALUE;
    }
    else {
      longmean = avgradiancemean - difradiancemean / 2.0;
      shortmean = avgradiancemean + difradiancemean / 2.0;
    }
    longvariance = 0.0;
    shortvariance = 0.0;

    // GAIN_AVERAGE 
    CalculateGain (avgradiancemean, avgradiancevariance, 
                   internalrecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_AVERAGE_MEAN),
                   internalrecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_AVERAGE_VARIANCE),
                   spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_AVERAGE_MEAN),
                   spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_AVERAGE_VARIANCE), gainmean, gainvariance);
    gainrecord->AddGain (pixel, CALIBRATION_STATE_AVERAGE, gainmean, gainvariance);

    // GAIN_DIFFERENCE
    CalculateGain (difradiancemean, difradiancevariance, 
                   internalrecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_DIFFERENCE_MEAN),
                   internalrecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_DIFFERENCE_VARIANCE),
                   spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_DIFFERENCE_MEAN),
                   spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_DIFFERENCE_VARIANCE), gainmean, gainvariance);
    gainrecord->AddGain (pixel, CALIBRATION_STATE_DIFFERENCE, gainmean, gainvariance);

    // GAIN_A_SECTOR
    CalculateGain (longmean, longvariance, 
                   internalrecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_A_SECTOR_MEAN),
                   internalrecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_A_SECTOR_VARIANCE),
                   spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_A_SECTOR_MEAN),
                   spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_A_SECTOR_VARIANCE), gainmean, gainvariance);
    gainrecord->AddGain (pixel, CALIBRATION_STATE_A_SECTOR, gainmean, gainvariance);

    // GAIN_B_SECTOR
    CalculateGain (shortmean, shortvariance, 
                   internalrecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_B_SECTOR_MEAN),
                   internalrecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_B_SECTOR_VARIANCE),
                   spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_B_SECTOR_MEAN),
                   spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_B_SECTOR_VARIANCE), gainmean, gainvariance);
    gainrecord->AddGain (pixel, CALIBRATION_STATE_B_SECTOR, gainmean, gainvariance);

    // GAIN_C_SECTOR
    CalculateGain (longmean, longvariance, 
                   internalrecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_C_SECTOR_MEAN),
                   internalrecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_C_SECTOR_VARIANCE),
                   spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_C_SECTOR_MEAN),
                   spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_C_SECTOR_VARIANCE), gainmean, gainvariance);
    gainrecord->AddGain (pixel, CALIBRATION_STATE_C_SECTOR, gainmean, gainvariance);

    // GAIN_D_SECTOR
    CalculateGain (shortmean, shortvariance, 
                   internalrecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_D_SECTOR_MEAN),
                   internalrecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_D_SECTOR_VARIANCE),
                   spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_D_SECTOR_MEAN),
                   spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_D_SECTOR_VARIANCE), gainmean, gainvariance);
    gainrecord->AddGain (pixel, CALIBRATION_STATE_D_SECTOR, gainmean, gainvariance);

  }
}


void calibration_channel_lmc_history :: InterpolateOffsetData (calibration_channel_record* prerecord,
                                                               calibration_channel_record* postrecord,
                                                               enum science_train_data_type datatype, 
                                                               calibration_channel_record* offsetrecord)
{
  // get the before and after times, and the offset record time
  mopitt_time beforetime = prerecord->GetTime ();
  mopitt_time aftertime = postrecord->GetTime ();
  mopitt_time offsettime = spacerecord->GetStartTime ();

  // for each pixel/state, interpolate and store the gain and radiance, then calculate the offset and Noise
  float beforemean, beforeerror, aftermean, aftererror, datapointmean, datapointerror;
  for (int pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) {

    // read the gain out of the records, interpolate to track time, save interpolated gain
    prerecord->GetGain (pixel, CALIBRATION_STATE_AVERAGE, beforemean, beforeerror);
    postrecord->GetGain  (pixel, CALIBRATION_STATE_AVERAGE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddGain (pixel, CALIBRATION_STATE_AVERAGE, datapointmean, datapointerror);

    prerecord->GetGain (pixel, CALIBRATION_STATE_DIFFERENCE, beforemean, beforeerror);
    postrecord->GetGain  (pixel, CALIBRATION_STATE_DIFFERENCE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddGain (pixel, CALIBRATION_STATE_DIFFERENCE, datapointmean, datapointerror);

    prerecord->GetGain (pixel, CALIBRATION_STATE_A_SECTOR, beforemean, beforeerror);
    postrecord->GetGain  (pixel, CALIBRATION_STATE_A_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddGain (pixel, CALIBRATION_STATE_A_SECTOR, datapointmean, datapointerror);

    prerecord->GetGain (pixel, CALIBRATION_STATE_B_SECTOR, beforemean, beforeerror);
    postrecord->GetGain  (pixel, CALIBRATION_STATE_B_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddGain (pixel, CALIBRATION_STATE_B_SECTOR, datapointmean, datapointerror);

    prerecord->GetGain (pixel, CALIBRATION_STATE_C_SECTOR, beforemean, beforeerror);
    postrecord->GetGain  (pixel, CALIBRATION_STATE_C_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddGain (pixel, CALIBRATION_STATE_C_SECTOR, datapointmean, datapointerror);

    prerecord->GetGain (pixel, CALIBRATION_STATE_D_SECTOR, beforemean, beforeerror);
    postrecord->GetGain  (pixel, CALIBRATION_STATE_D_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddGain (pixel, CALIBRATION_STATE_D_SECTOR, datapointmean, datapointerror);

    // read the radiance out of the records, interpolate to track time, save interpolated radiance
    prerecord->GetRadiance (pixel, CALIBRATION_STATE_AVERAGE, beforemean, beforeerror);
    postrecord->GetRadiance  (pixel, CALIBRATION_STATE_AVERAGE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddRadiance (pixel, CALIBRATION_STATE_AVERAGE, datapointmean, datapointerror);
    prerecord->GetRadiance (pixel, CALIBRATION_STATE_DIFFERENCE, beforemean, beforeerror);
    postrecord->GetRadiance  (pixel, CALIBRATION_STATE_DIFFERENCE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, offsettime,
                           datapointmean, datapointerror);
    offsetrecord->AddRadiance (pixel, CALIBRATION_STATE_DIFFERENCE, datapointmean, datapointerror);

    //  get the interpolated gain, calculate the offset and noise, save the offset and noise
    float gainmean, gainerror, offsetmean, offseterror;
    offsetrecord->GetGain (pixel, CALIBRATION_STATE_AVERAGE, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_AVERAGE_MEAN),
                     spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_AVERAGE_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_AVERAGE, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_AVERAGE_VARIANCE), 
                    datapointmean, datapointerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_AVERAGE, datapointmean, datapointerror);

    offsetrecord->GetGain (pixel, CALIBRATION_STATE_DIFFERENCE, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_DIFFERENCE_MEAN),
                     spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_DIFFERENCE_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_DIFFERENCE, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_DIFFERENCE_VARIANCE), 
                    datapointmean, datapointerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_DIFFERENCE, datapointmean, datapointerror);

    offsetrecord->GetGain (pixel, CALIBRATION_STATE_A_SECTOR, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_A_SECTOR_MEAN),
                     spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_A_SECTOR_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_A_SECTOR, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_A_SECTOR_VARIANCE), 
                    datapointmean, datapointerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_A_SECTOR, datapointmean, datapointerror);

    offsetrecord->GetGain (pixel, CALIBRATION_STATE_B_SECTOR, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_B_SECTOR_MEAN),
                     spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_B_SECTOR_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_B_SECTOR, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_B_SECTOR_VARIANCE), 
                    datapointmean, datapointerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_B_SECTOR, datapointmean, datapointerror);

    offsetrecord->GetGain (pixel, CALIBRATION_STATE_C_SECTOR, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_C_SECTOR_MEAN),
                     spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_C_SECTOR_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_C_SECTOR, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_C_SECTOR_VARIANCE), 
                    datapointmean, datapointerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_C_SECTOR, datapointmean, datapointerror);

    offsetrecord->GetGain (pixel, CALIBRATION_STATE_D_SECTOR, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_D_SECTOR_MEAN),
                     spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_D_SECTOR_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_D_SECTOR, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_D_SECTOR_VARIANCE), 
                    datapointmean, datapointerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_D_SECTOR, datapointmean, datapointerror);
  }
}


void calibration_channel_lmc_history :: TransferOffsetData (calibration_channel_record* gainrecord,
                                                            enum science_train_data_type datatype,
                                                            calibration_channel_record* offsetrecord)

{ 
  // transfer the data from the gain record to the offset record
  float recorddata [CALIBRATION_PIXELS] [CALIBRATION_STATES] [CALIBRATION_TERMS];
  gainrecord->GetData (recorddata);
  offsetrecord->AddData (recorddata);

  // calculate the offsets
  float gainmean, gainerror, offsetmean, offseterror, noisemean, noiseerror;
  for (int pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) {

    gainrecord->GetGain (pixel, CALIBRATION_STATE_AVERAGE, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_AVERAGE_MEAN),
                     spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_AVERAGE_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_AVERAGE, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_AVERAGE_VARIANCE), 
                    noisemean, noiseerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_AVERAGE, noisemean, noiseerror);

    gainrecord->GetGain (pixel, CALIBRATION_STATE_DIFFERENCE, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_DIFFERENCE_MEAN),
                     spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_DIFFERENCE_VARIANCE), offsetmean, 
                     offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_DIFFERENCE, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSignal (datatype, pixel, LMC_SIGNAL_DIFFERENCE_VARIANCE), 
                    noisemean, noiseerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_DIFFERENCE, noisemean, noiseerror);

    gainrecord->GetGain (pixel, CALIBRATION_STATE_A_SECTOR, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_A_SECTOR_MEAN),
                     spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_A_SECTOR_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_A_SECTOR, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_A_SECTOR_VARIANCE), 
                    noisemean, noiseerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_A_SECTOR, noisemean, noiseerror);
 
    gainrecord->GetGain (pixel, CALIBRATION_STATE_B_SECTOR, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_B_SECTOR_MEAN),
                     spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_B_SECTOR_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_B_SECTOR, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_B_SECTOR_VARIANCE), 
                    noisemean, noiseerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_B_SECTOR, noisemean, noiseerror);

    gainrecord->GetGain (pixel, CALIBRATION_STATE_C_SECTOR, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_C_SECTOR_MEAN),
                     spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_C_SECTOR_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_C_SECTOR, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_C_SECTOR_VARIANCE), 
                    noisemean, noiseerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_C_SECTOR, noisemean, noiseerror);

    gainrecord->GetGain (pixel, CALIBRATION_STATE_D_SECTOR, gainmean, gainerror);
    CalculateOffset (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_D_SECTOR_MEAN),
                     spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_D_SECTOR_VARIANCE), offsetmean, offseterror);
    offsetrecord->AddOffset (pixel, CALIBRATION_STATE_D_SECTOR, offsetmean, offseterror);
    CalculateNoise (gainmean, gainerror, spacerecord->GetLMCSectorSignal (datatype, pixel, LMC_SIGNAL_D_SECTOR_VARIANCE), 
                    noisemean, noiseerror);
    offsetrecord->AddNoise (pixel, CALIBRATION_STATE_D_SECTOR, noisemean, noiseerror);
  }
}


void calibration_channel_lmc_history :: GetBracketingTimes (mopitt_time const& tracktime, mopitt_time& pretime,
                                                            mopitt_time& posttime)
{
  calibration_channel_record* prerecord = NULL;
  calibration_channel_record* postrecord = NULL;

  GetBracketingRecords (tracktime, offsetlist, &prerecord, &postrecord);
  if (prerecord == NULL)
    pretime.Set (0.0);
  else
    pretime = prerecord->GetTime ();
  if (postrecord == NULL)
    posttime.Set (0.0);
  else
    posttime = postrecord->GetTime ();
}
