/********** CalibrationChannelHistory.C *******************************************************************************\

 $Header$

 REVISION HISTORY
   06/99   Charles Cavanaugh
   11/99   Charles Cavanaugh
   06/00   Charles Cavanaugh
   09/00   Charles Cavanaugh
   10/00   Charles Cavanaugh
   10/01   Debbie Mao
   02/02   Debbie Mao
   04/02   Debbie Mao
   05/02   Debbie Mao
   06/02   Debbie Mao
   07/02   Debbie Mao
   04/03   Debbie Mao -- add missing value for sector internal radiances

 $Log$

\**********************************************************************************************************************/

#include <iomanip>
#include <math.h>
#include "CalibrationChannelHistory.h"
#include "DiagnosticReporter.h"

extern diagnostic_reporter diagnosticreporter;

int   const calibration_channel_history::INTERPOLATION_WARNING                                   = 800;
float const calibration_channel_history::NOISEVALUES [CALIBRATION_CHANNELS] [CALIBRATION_STATES] = { 
                             3.686e-6, 7.373e-6, 6.660e-7, 1.330e-6, 6.863e-6, 1.373e-5, 1.669e-6, 3.338e-6,
                             5.750e-6, 1.150e-5, 6.520e-7, 1.303e-6, 4.713e-6, 9.425e-6, 1.792e-6, 3.583e-6 };

calibration_channel_history :: calibration_channel_history ()
                             : gainlist ()
                             , offsetlist ()
{
  channel = 1;
  internalrecord = NULL;
  spacerecord = NULL;
  mopchfile = NULL;
}


calibration_channel_history :: calibration_channel_history (int channelnumber)
                             : gainlist ()
                             , offsetlist ()
{
  channel = channelnumber;
  internalrecord = NULL;
  spacerecord = NULL;
  mopchfile = NULL;
}


calibration_channel_history :: ~calibration_channel_history ()
{
  delete internalrecord;
  delete spacerecord;
  delete mopchfile;
}


bool calibration_channel_history :: ActivateTrainGroup (PGSt_PC_Logical filelogical, 
                                                        enum science_train_data_type datatype, 
                                                        science_train_group* traingroup)
{
  // initialize the mopch file
  mopchfile = new mopch_read_file (filelogical);

  // initialize the train group
  traingroup->SetIDs (mopchfile->GetScienceID (), mopchfile->GetFileID ());
  return traingroup->Attach ();
}


bool calibration_channel_history :: InactivateTrainGroup (science_train_group* traingroup)
{
  // shut down access to the train group
  bool isinactive = traingroup->Close ();

  // shut down access to the mopch file
  delete mopchfile;
  mopchfile = NULL;

  return isinactive;
}


void calibration_channel_history :: InterpolateTrackData (mopitt_time const& tracktime, 
                       calibration_channel_record* prerecord, calibration_channel_record* postrecord,
                       float trackdata [CALIBRATION_PIXELS] [CALIBRATION_STATES] [CALIBRATION_TERMS],
		       double trackcoldcal [CALIBRATION_PACKETPOSITION] [CALIBRATION_PIXELS] [PMC_SIGNAL_STATES])
{
  // get the before and after times, and the offset record time
  mopitt_time beforetime = prerecord->GetTime ();
  mopitt_time aftertime = postrecord->GetTime ();

  float beforemean, beforeerror, aftermean, aftererror;
  double precoldcal  [CALIBRATION_PACKETPOSITION] [CALIBRATION_PIXELS] [PMC_SIGNAL_STATES],
         postcoldcal [CALIBRATION_PACKETPOSITION] [CALIBRATION_PIXELS] [PMC_SIGNAL_STATES];
  for (int pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) {
    prerecord->GetGain  (pixel, CALIBRATION_STATE_AVERAGE, beforemean, beforeerror);
     postrecord->GetGain   (pixel, CALIBRATION_STATE_AVERAGE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_AVERAGE] [CALIBRATION_TERM_GAIN_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_AVERAGE] [CALIBRATION_TERM_GAIN_ERROR]);

    prerecord->GetGain  (pixel, CALIBRATION_STATE_DIFFERENCE, beforemean, beforeerror);
    postrecord->GetGain   (pixel, CALIBRATION_STATE_DIFFERENCE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_DIFFERENCE] [CALIBRATION_TERM_GAIN_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_DIFFERENCE] [CALIBRATION_TERM_GAIN_ERROR]);

    prerecord->GetGain  (pixel, CALIBRATION_STATE_A_SECTOR, beforemean, beforeerror);
    postrecord->GetGain   (pixel, CALIBRATION_STATE_A_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_A_SECTOR] [CALIBRATION_TERM_GAIN_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_A_SECTOR] [CALIBRATION_TERM_GAIN_ERROR]);

    prerecord->GetGain  (pixel, CALIBRATION_STATE_B_SECTOR, beforemean, beforeerror);
    postrecord->GetGain   (pixel, CALIBRATION_STATE_B_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_B_SECTOR] [CALIBRATION_TERM_GAIN_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_B_SECTOR] [CALIBRATION_TERM_GAIN_ERROR]);

    prerecord->GetGain  (pixel, CALIBRATION_STATE_C_SECTOR, beforemean, beforeerror);
    postrecord->GetGain   (pixel, CALIBRATION_STATE_C_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_C_SECTOR] [CALIBRATION_TERM_GAIN_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_C_SECTOR] [CALIBRATION_TERM_GAIN_ERROR]);

    prerecord->GetGain  (pixel, CALIBRATION_STATE_D_SECTOR, beforemean, beforeerror);
    postrecord->GetGain   (pixel, CALIBRATION_STATE_D_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_D_SECTOR] [CALIBRATION_TERM_GAIN_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_D_SECTOR] [CALIBRATION_TERM_GAIN_ERROR]);

    prerecord->GetRadiance (pixel, CALIBRATION_STATE_AVERAGE, beforemean, beforeerror);
    postrecord->GetRadiance  (pixel, CALIBRATION_STATE_AVERAGE, aftermean,  aftererror);
    InterpolateDataPoints    (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			      trackdata [pixel] [CALIBRATION_STATE_AVERAGE] [CALIBRATION_TERM_RADIANCE_MEAN],
			      trackdata [pixel] [CALIBRATION_STATE_AVERAGE] [CALIBRATION_TERM_RADIANCE_ERROR]);

    prerecord->GetRadiance (pixel, CALIBRATION_STATE_DIFFERENCE, beforemean, beforeerror);
    postrecord->GetRadiance  (pixel, CALIBRATION_STATE_DIFFERENCE, aftermean,  aftererror);
    InterpolateDataPoints    (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			      trackdata [pixel] [CALIBRATION_STATE_DIFFERENCE] [CALIBRATION_TERM_RADIANCE_MEAN],
			      trackdata [pixel] [CALIBRATION_STATE_DIFFERENCE] [CALIBRATION_TERM_RADIANCE_ERROR]);

    prerecord->GetNoise (pixel, CALIBRATION_STATE_AVERAGE, beforemean, beforeerror);
    postrecord->GetNoise  (pixel, CALIBRATION_STATE_AVERAGE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_AVERAGE] [CALIBRATION_TERM_NOISE_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_AVERAGE] [CALIBRATION_TERM_NOISE_ERROR]);

    prerecord->GetNoise (pixel, CALIBRATION_STATE_DIFFERENCE, beforemean, beforeerror);
    postrecord->GetNoise  (pixel, CALIBRATION_STATE_DIFFERENCE, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_DIFFERENCE] [CALIBRATION_TERM_NOISE_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_DIFFERENCE] [CALIBRATION_TERM_NOISE_ERROR]);

    prerecord->GetNoise (pixel, CALIBRATION_STATE_A_SECTOR, beforemean, beforeerror);
    postrecord->GetNoise  (pixel, CALIBRATION_STATE_A_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_A_SECTOR] [CALIBRATION_TERM_NOISE_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_A_SECTOR] [CALIBRATION_TERM_NOISE_ERROR]);

    prerecord->GetNoise (pixel, CALIBRATION_STATE_B_SECTOR, beforemean, beforeerror);
    postrecord->GetNoise  (pixel, CALIBRATION_STATE_B_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_B_SECTOR] [CALIBRATION_TERM_NOISE_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_B_SECTOR] [CALIBRATION_TERM_NOISE_ERROR]);

    prerecord->GetNoise (pixel, CALIBRATION_STATE_C_SECTOR, beforemean, beforeerror);
    postrecord->GetNoise  (pixel, CALIBRATION_STATE_C_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_C_SECTOR] [CALIBRATION_TERM_NOISE_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_C_SECTOR] [CALIBRATION_TERM_NOISE_ERROR]);

    prerecord->GetNoise (pixel, CALIBRATION_STATE_D_SECTOR, beforemean, beforeerror);
    postrecord->GetNoise  (pixel, CALIBRATION_STATE_D_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			   trackdata [pixel] [CALIBRATION_STATE_D_SECTOR] [CALIBRATION_TERM_NOISE_MEAN],
			   trackdata [pixel] [CALIBRATION_STATE_D_SECTOR] [CALIBRATION_TERM_NOISE_ERROR]);

    prerecord->GetOffset (pixel, CALIBRATION_STATE_AVERAGE, beforemean, beforeerror);
    postrecord->GetOffset  (pixel, CALIBRATION_STATE_AVERAGE, aftermean,  aftererror);
    InterpolateDataPoints  (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			    trackdata [pixel] [CALIBRATION_STATE_AVERAGE] [CALIBRATION_TERM_OFFSET_MEAN],
			    trackdata [pixel] [CALIBRATION_STATE_AVERAGE] [CALIBRATION_TERM_OFFSET_ERROR]);

    prerecord->GetOffset (pixel, CALIBRATION_STATE_DIFFERENCE, beforemean, beforeerror);
    postrecord->GetOffset  (pixel, CALIBRATION_STATE_DIFFERENCE, aftermean,  aftererror);
    InterpolateDataPoints  (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			    trackdata [pixel] [CALIBRATION_STATE_DIFFERENCE] [CALIBRATION_TERM_OFFSET_MEAN],
			    trackdata [pixel] [CALIBRATION_STATE_DIFFERENCE] [CALIBRATION_TERM_OFFSET_ERROR]);

    prerecord->GetOffset (pixel, CALIBRATION_STATE_A_SECTOR, beforemean, beforeerror);
    postrecord->GetOffset  (pixel, CALIBRATION_STATE_A_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints  (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			    trackdata [pixel] [CALIBRATION_STATE_A_SECTOR] [CALIBRATION_TERM_OFFSET_MEAN],
			    trackdata [pixel] [CALIBRATION_STATE_A_SECTOR] [CALIBRATION_TERM_OFFSET_ERROR]);

    prerecord->GetOffset (pixel, CALIBRATION_STATE_B_SECTOR, beforemean, beforeerror);
    postrecord->GetOffset  (pixel, CALIBRATION_STATE_B_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints  (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			    trackdata [pixel] [CALIBRATION_STATE_B_SECTOR] [CALIBRATION_TERM_OFFSET_MEAN],
			    trackdata [pixel] [CALIBRATION_STATE_B_SECTOR] [CALIBRATION_TERM_OFFSET_ERROR]);

    prerecord->GetOffset (pixel, CALIBRATION_STATE_C_SECTOR, beforemean, beforeerror);
    postrecord->GetOffset  (pixel, CALIBRATION_STATE_C_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints  (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			    trackdata [pixel] [CALIBRATION_STATE_C_SECTOR] [CALIBRATION_TERM_OFFSET_MEAN],
			    trackdata [pixel] [CALIBRATION_STATE_C_SECTOR] [CALIBRATION_TERM_OFFSET_ERROR]);

    prerecord->GetOffset (pixel, CALIBRATION_STATE_D_SECTOR, beforemean, beforeerror);
    postrecord->GetOffset  (pixel, CALIBRATION_STATE_D_SECTOR, aftermean,  aftererror);
    InterpolateDataPoints  (beforemean, beforeerror, aftermean, aftererror, beforetime, aftertime, tracktime,
			    trackdata [pixel] [CALIBRATION_STATE_D_SECTOR] [CALIBRATION_TERM_OFFSET_MEAN],
			    trackdata [pixel] [CALIBRATION_STATE_D_SECTOR] [CALIBRATION_TERM_OFFSET_ERROR]);
  }

  if ( channel == 3 || channel == 7 ) {  
    prerecord->GetColdCals  (precoldcal);
    postrecord->GetColdCals (postcoldcal);
    for ( int pos = 0; pos < SCIENCE_PACKETPOSITION; pos++ )
      for (int pixel = 0; pixel < SCIENCE_PIXELS; pixel++) 
        for ( int state = 0; state < PMC_SIGNAL_STATES; state++) {
	  if ( precoldcal[pos][pixel][state] == CALIBRATION_MISSING_VALUE && 
	       postcoldcal[pos][pixel][state] == CALIBRATION_MISSING_VALUE )
	    trackcoldcal[pos][pixel][state] = CALIBRATION_MISSING_VALUE;
	  else if ( precoldcal[pos][pixel][state] == CALIBRATION_MISSING_VALUE )
	    trackcoldcal[pos][pixel][state] = postcoldcal[pos][pixel][state];
	  else if ( postcoldcal[pos][pixel][state] == CALIBRATION_MISSING_VALUE )
	    trackcoldcal[pos][pixel][state] = precoldcal[pos][pixel][state];
	  else
	    trackcoldcal[pos][pixel][state] = ( precoldcal[pos][pixel][state] + postcoldcal[pos][pixel][state]) / 2.0;
	  }
  }
  else 
   for ( int pos = 0; pos < SCIENCE_PACKETPOSITION; pos++ )
     for (int pixel = 0; pixel < SCIENCE_PIXELS; pixel++)
       for ( int state = 0; state < PMC_SIGNAL_STATES; state++)
         trackcoldcal[pos][pixel][state] = CALIBRATION_MISSING_VALUE;
}
  

void calibration_channel_history :: InvalidateOffsetData (calibration_channel_record* offsetrecord)
{
  offsetrecord->AddNoise (CALIBRATION_STATE_AVERAGE, CALIBRATION_MISSING_VALUE, CALIBRATION_MISSING_VALUE);
  offsetrecord->AddNoise (CALIBRATION_STATE_DIFFERENCE, CALIBRATION_MISSING_VALUE, CALIBRATION_MISSING_VALUE);
  offsetrecord->AddNoise (CALIBRATION_STATE_A_SECTOR, CALIBRATION_MISSING_VALUE, CALIBRATION_MISSING_VALUE);
  offsetrecord->AddNoise (CALIBRATION_STATE_B_SECTOR, CALIBRATION_MISSING_VALUE, CALIBRATION_MISSING_VALUE);
  offsetrecord->AddNoise (CALIBRATION_STATE_C_SECTOR, CALIBRATION_MISSING_VALUE, CALIBRATION_MISSING_VALUE);
  offsetrecord->AddNoise (CALIBRATION_STATE_D_SECTOR, CALIBRATION_MISSING_VALUE, CALIBRATION_MISSING_VALUE);

  for (int pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) {
    offsetrecord->AddRadiance (pixel, CALIBRATION_STATE_AVERAGE,    CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddRadiance (pixel, CALIBRATION_STATE_DIFFERENCE, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);

    offsetrecord->AddGain     (pixel, CALIBRATION_STATE_AVERAGE,    CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddGain     (pixel, CALIBRATION_STATE_DIFFERENCE, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddGain     (pixel, CALIBRATION_STATE_A_SECTOR, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddGain     (pixel, CALIBRATION_STATE_B_SECTOR, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddGain     (pixel, CALIBRATION_STATE_C_SECTOR, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddGain     (pixel, CALIBRATION_STATE_D_SECTOR, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);

    offsetrecord->AddOffset   (pixel, CALIBRATION_STATE_AVERAGE,    CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddOffset   (pixel, CALIBRATION_STATE_DIFFERENCE, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddOffset   (pixel, CALIBRATION_STATE_A_SECTOR, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddOffset   (pixel, CALIBRATION_STATE_B_SECTOR, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddOffset   (pixel, CALIBRATION_STATE_C_SECTOR, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
    offsetrecord->AddOffset   (pixel, CALIBRATION_STATE_D_SECTOR, CALIBRATION_MISSING_VALUE, 
			       CALIBRATION_MISSING_VALUE);
  }
}


void calibration_channel_history :: CalculateGain (float radiancemean, float radianceerror,
                                                   double internalsignalmean, double internalsignalerror,
                                                   double spacesignalmean, double spacesignalerror,
                                                   float& gainmean, float& gainerror) const
{
  if ((int) (radiancemean       + 0.5) == CALIBRATION_MISSING_INT || 
      (int) (internalsignalmean + 0.5) == CALIBRATION_MISSING_INT ||
      (int) (spacesignalmean    + 0.5) == CALIBRATION_MISSING_INT || 
      (internalsignalmean - spacesignalmean) == 0.0  ) {
    gainmean  = CALIBRATION_MISSING_VALUE;
    gainerror = CALIBRATION_MISSING_VALUE;
  }
  else {
    double dgainmean = radiancemean / (internalsignalmean - spacesignalmean);
    double signaldiff = pow ((internalsignalmean - spacesignalmean), 2.0);
    double dgainerror = (radianceerror / signaldiff) + ((internalsignalerror + spacesignalerror) *
                                                       pow ((radiancemean / signaldiff), 2.0));
    gainmean = (float) dgainmean;
    gainerror = (float) dgainerror;
  }
}


void calibration_channel_history :: CalculateNoise (float gainmean, float gainerror, double offseterror,
                                                    float& noisemean, float& noiseerror) const
{
  if ((int) (gainmean    + 0.5) == CALIBRATION_MISSING_INT ||
      (int) (offseterror + 0.5) == CALIBRATION_MISSING_INT) {
    noisemean  = CALIBRATION_MISSING_VALUE;
    noiseerror = CALIBRATION_MISSING_VALUE;
  }
  else {
    noisemean = sqrt (offseterror) * gainmean;
    noiseerror = powf (gainmean, 2.0) / (4.0 * offseterror) + offseterror * gainerror;
  }
}


void calibration_channel_history :: CalculatePosNoise ( float gainmean, double warmstddev, 
							float& posnoise ) const
{
  if ((int) (gainmean + 0.5)   == CALIBRATION_MISSING_INT || gainmean < 0.0 ||
      (int) (warmstddev + 0.5) == CALIBRATION_MISSING_INT || warmstddev < 0.0 ) 
    posnoise  = CALIBRATION_MISSING_VALUE;
  else
    posnoise =  sqrt( warmstddev ) * gainmean;
}

void calibration_channel_history :: CalculateOffset (float gainmean, float gainerror, double spacesignalmean,
                                                     double spacesignalerror, float& offsetmean, 
                                                     float& offseterror) const
{
  if ((int) (gainmean        + 0.5) == CALIBRATION_MISSING_INT || 
      (int) (spacesignalmean + 0.5) == CALIBRATION_MISSING_INT) {
    offsetmean  = CALIBRATION_MISSING_VALUE;
    offseterror = CALIBRATION_MISSING_VALUE;
  }
  else {
   double doffsetmean = -gainmean * spacesignalmean;
   double doffseterror = (spacesignalmean * spacesignalmean * gainerror) + (gainmean * gainmean * spacesignalerror);
   offsetmean = (float) doffsetmean;
   offseterror = (float) doffseterror;
  }
}


void calibration_channel_history :: FillOffsetRecord (enum science_train_data_type datatype,
                                                      calibration_channel_record* offsetrecord)
{
  // find the bracketing gain records
  calibration_channel_record* prerecord;
  calibration_channel_record* postrecord;
  GetBracketingRecords (spacerecord->GetStartTime (), gainlist, &prerecord, &postrecord);

  // if gains bracket this record, interpolate data to record time and compute offset; else if only one gain available,
  // compute offset from the one record; else if no gains available, set to all missing
  if (prerecord != NULL && postrecord != NULL)
    InterpolateOffsetData (prerecord, postrecord, datatype, offsetrecord);
  else if (prerecord != NULL)
    TransferOffsetData (prerecord, datatype, offsetrecord);
  else if (postrecord != NULL)
    TransferOffsetData (postrecord, datatype, offsetrecord);
  else
    InvalidateOffsetData (offsetrecord);

  // set the sector internal radiances to missing values
  for ( int pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) {
    offsetrecord->AddRadiance (pixel,CALIBRATION_STATE_A_SECTOR,CALIBRATION_MISSING_VALUE,CALIBRATION_MISSING_VALUE);
    offsetrecord->AddRadiance (pixel,CALIBRATION_STATE_B_SECTOR,CALIBRATION_MISSING_VALUE,CALIBRATION_MISSING_VALUE);
    offsetrecord->AddRadiance (pixel,CALIBRATION_STATE_C_SECTOR,CALIBRATION_MISSING_VALUE,CALIBRATION_MISSING_VALUE);
    offsetrecord->AddRadiance (pixel,CALIBRATION_STATE_D_SECTOR,CALIBRATION_MISSING_VALUE,CALIBRATION_MISSING_VALUE);
  }

  // get PMC cold cal signals and add them to offsetlist
  if ( channel == 3 || channel == 7 ) {
    double coldcals [CALIBRATION_PACKETPOSITION] [CALIBRATION_PIXELS] [PMC_SIGNAL_STATES];
    spacerecord->GetPMCColdSignals ( coldcals );
    offsetrecord->AddColdCals ( coldcals );
  }
}


void calibration_channel_history :: GetBracketingRecords (mopitt_time const& searchtime,
                                                          calibration_record_list& recordlist,
                                                          calibration_channel_record** prerecord,
                                                          calibration_channel_record** postrecord)
{
  // initialize the pre container
  *prerecord = NULL;

  // get the last record that occurs before the search time
  calibration_channel_record* record = NULL;
  mopitt_time recordtime (0.0);
  recordlist.ResetNext ();
  if ((record = recordlist.GetNext ()) != NULL)
    recordtime = record->GetTime ();
  while (record != NULL && recordtime < searchtime) {
    *prerecord = record;
     if ((record = recordlist.GetNext ()) != NULL)
       recordtime = record->GetTime ();
  }

  // the current record will be either the first post-search time record or NULL (which means there are no post records)
  *postrecord = record;
}


void calibration_channel_history :: InterpolateDataPoints (float beforemean, float beforeerror, float aftermean,
                                                           float aftererror, mopitt_time const& beforetime,
                                                           mopitt_time const& aftertime, mopitt_time const& interptime,
                                                           float& interpmean, float& interperror)
{
  int intbeforemean = (int) (beforemean + 0.5);
  int intaftermean  = (int) (aftermean + 0.5);

  if (intbeforemean == CALIBRATION_MISSING_INT && intaftermean == CALIBRATION_MISSING_INT) {
    interpmean = CALIBRATION_MISSING_VALUE;
    interperror = CALIBRATION_MISSING_VALUE;
  }
  else if (intbeforemean == CALIBRATION_MISSING_INT) {
    interpmean  = aftermean;
    interperror = aftererror;
  }
  else if (intaftermean == CALIBRATION_MISSING_INT) {
    interpmean  = beforemean;
    interperror = beforeerror;
  }
  else {
    double timediff = (interptime.GetTAI () - beforetime.GetTAI ()) / (aftertime.GetTAI () - beforetime.GetTAI ());
    double dinterpmean = beforemean + ((aftermean - beforemean) * timediff);
    double dinterperror = (pow ((1.0 - timediff), 2.0) * beforeerror) + (pow (timediff, 2.0) * aftererror);
    interpmean = (float) dinterpmean;
    interperror = (float) dinterperror;
  }
}


void calibration_channel_history :: ReadCurrentGranuleGains (PGSt_PC_Logical filelogical,
                                                             enum science_train_data_type datatype,
                                                             science_train_group* traingroup, 
                                                             mopir_collection const& mopircollection)
{
  calibration_channel_record* gainrecord = NULL;
  calibration_channel_noise_record* noiserecord = NULL;

  // activate the MOPCH train group
  if (ActivateTrainGroup (filelogical, datatype, traingroup)) {

    // for each "datatype" record
    for (int index = 0; index < traingroup->GetEventCount (datatype); index++)
      if (traingroup->GetRecord (index, datatype, internalrecord)) {

        // get the closest cold record
        if (traingroup->GetClosestRecord (internalrecord->GetStartTime (), COLD_DATA_TYPE, spacerecord)) {

          // make a new gain record, fill it and add it to the gain list
	  gainrecord = new calibration_channel_record (internalrecord->GetStartTime ());
	  noiserecord = new calibration_channel_noise_record ();
	  if ( channel == 3 || channel == 7 ) 
	    FillGainRecord (mopircollection, datatype, gainrecord, noiserecord);
	  else
	    FillGainRecord (mopircollection, datatype, gainrecord);	    
          gainlist.AddRecord (gainrecord);
          noiselist.AddRecord (noiserecord);
        }
     }

    // inactive the MOPCH train group
    InactivateTrainGroup (traingroup);
  }
}


void calibration_channel_history :: ReadCurrentGranuleOffsets (PGSt_PC_Logical filelogical, 
                                                               enum science_train_data_type datatype,
                                                               science_train_group* traingroup)
{
  calibration_channel_record* offsetrecord = NULL;

  // activate the MOPCH train group
  if (ActivateTrainGroup (filelogical, COLD_DATA_TYPE, traingroup)) {

    // for each record
    for (int index = 0; index < traingroup->GetEventCount (COLD_DATA_TYPE); index++)
      if (traingroup->GetRecord (index, COLD_DATA_TYPE, spacerecord)) {

        // make a new offset record, fill it and add it to the offset list
        offsetrecord = new calibration_channel_record (spacerecord->GetStartTime ());
        FillOffsetRecord (datatype, offsetrecord);
        offsetlist.AddRecord (offsetrecord);
      }

    // inactive the MOPCH train group
    InactivateTrainGroup (traingroup);
  }
}


void calibration_channel_history :: ReadPostGranuleGains (PGSt_PC_Logical filelogical,
                                                          enum science_train_data_type datatype,
                                                          science_train_group* traingroup, 
                                                          mopir_collection const& mopircollection)
{
  // activate the MOPCH train group
  if (ActivateTrainGroup (filelogical, datatype, traingroup)) {

    // get count of "datatype" events
    int events = traingroup->GetEventCount (datatype);

    // get the first "datatype" record
    if (events > 0)
      if (traingroup->GetRecord (0, datatype, internalrecord))

        // get the closest cold record
        if (traingroup->GetClosestRecord (internalrecord->GetStartTime (), COLD_DATA_TYPE, spacerecord)) {

          // make a new gain record, fill it and add it to the gain list
          calibration_channel_record* gainrecord = new calibration_channel_record (internalrecord->GetStartTime ());
	  calibration_channel_noise_record* noiserecord = new calibration_channel_noise_record ();
	  if ( channel == 3 || channel == 7 ) 
	    FillGainRecord (mopircollection, datatype, gainrecord, noiserecord);
	  else
	    FillGainRecord (mopircollection, datatype, gainrecord);
          gainlist.AddRecord (gainrecord);
	  noiselist.AddRecord (noiserecord);
        }

    // repeat for second record (we want two gain records because the gain will be interpolated to the offset time)
    if (events > 1)
      if (traingroup->GetRecord (1, datatype, internalrecord))

        // get the closest cold record
        if (traingroup->GetClosestRecord (internalrecord->GetStartTime (), COLD_DATA_TYPE, spacerecord)) {

          // make a new gain record, fill it and add it to the gain list
          calibration_channel_record* gainrecord = new calibration_channel_record (internalrecord->GetStartTime ());
	  calibration_channel_noise_record* noiserecord = new calibration_channel_noise_record ();
	  if ( channel == 3 || channel == 7 ) 
	    FillGainRecord (mopircollection, datatype, gainrecord, noiserecord);
	  else
	    FillGainRecord (mopircollection, datatype, gainrecord);
          gainlist.AddRecord (gainrecord);
	  noiselist.AddRecord (noiserecord);
        }

    // inactive the MOPCH train group
    InactivateTrainGroup (traingroup);
  }
}


void calibration_channel_history :: ReadPostGranuleOffsets (PGSt_PC_Logical filelogical, 
                                                            enum science_train_data_type datatype,
                                                            science_train_group* traingroup)
{
  // activate the MOPCH train group
  if (ActivateTrainGroup (filelogical, COLD_DATA_TYPE, traingroup)) {

    // get count of cold events
    int events = traingroup->GetEventCount (COLD_DATA_TYPE);

    // get the first cold record
    if (events > 0)
      if (traingroup->GetRecord (0, COLD_DATA_TYPE, spacerecord)) {

        // make a new offset record, fill it and add it to the offset list
        calibration_channel_record* offsetrecord = new calibration_channel_record (spacerecord->GetStartTime ());
        FillOffsetRecord (datatype, offsetrecord);
        offsetlist.AddRecord (offsetrecord);
      }

    // inactive the MOPCH train group
    InactivateTrainGroup (traingroup);
  }
}


void calibration_channel_history :: ReadPreGranuleGains (PGSt_PC_Logical filelogical,
                                                         enum science_train_data_type datatype,
                                                         science_train_group* traingroup, 
                                                          mopir_collection const& mopircollection)
{
  calibration_channel_record* gainrecord = NULL;

  // activate the MOPCH train group
  if (ActivateTrainGroup (filelogical, datatype, traingroup)) {

    // get count of "datatype" events
    int events = traingroup->GetEventCount (datatype);

    // get the penultimate "datatype" record
    if (events > 1) 
      if (traingroup->GetRecord ((events - 2), datatype, internalrecord))

        // get the closest cold record
        if (traingroup->GetClosestRecord (internalrecord->GetStartTime (), COLD_DATA_TYPE, spacerecord)) {

          // make a new gain record, fill it and add it to the gain list
          gainrecord = new calibration_channel_record (internalrecord->GetStartTime ());
	  calibration_channel_noise_record* noiserecord = new calibration_channel_noise_record ();
	  if ( channel == 3 || channel == 7 ) 
	    FillGainRecord (mopircollection, datatype, gainrecord, noiserecord); 
          else
	    FillGainRecord (mopircollection, datatype, gainrecord);
	  gainlist.AddRecord (gainrecord); 
	  noiselist.AddRecord (noiserecord); 
        }

    // repeat for ultimate record (we want two gain records because the gain will be interpolated to the offset time)
    if (events > 0) 
      if (traingroup->GetRecord ((events - 1), datatype, internalrecord))

        // get the closest cold record
        if (traingroup->GetClosestRecord (internalrecord->GetStartTime (), COLD_DATA_TYPE, spacerecord)) {

          // make a new gain record, fill it and add it to the gain list
          gainrecord = new calibration_channel_record (internalrecord->GetStartTime ());
	  calibration_channel_noise_record* noiserecord = new calibration_channel_noise_record ();
	  if ( channel == 3 || channel == 7 ) 
	    FillGainRecord (mopircollection, datatype, gainrecord, noiserecord);
	  else
	    FillGainRecord (mopircollection, datatype, gainrecord);
          gainlist.AddRecord (gainrecord);
	  noiselist.AddRecord (noiserecord); 
        }

    // inactive the MOPCH train group
    InactivateTrainGroup (traingroup);
  }
}


void calibration_channel_history :: ReadPreGranuleOffsets (PGSt_PC_Logical filelogical, 
                                                           enum science_train_data_type datatype,
                                                           science_train_group* traingroup)
{
  // activate the MOPCH train group
  if (ActivateTrainGroup (filelogical, COLD_DATA_TYPE, traingroup)) {

    // get count of cold events
    int events = traingroup->GetEventCount (COLD_DATA_TYPE);

    // get the last cold record
    if (events > 0)
      if (traingroup->GetRecord ((events - 1), COLD_DATA_TYPE, spacerecord)) {

        // make a new offset record, fill it and add it to the offset list
        calibration_channel_record* offsetrecord = new calibration_channel_record (spacerecord->GetStartTime ());
        FillOffsetRecord (datatype, offsetrecord);
        offsetlist.AddRecord (offsetrecord);
      }

    // inactive the MOPCH train group
    InactivateTrainGroup (traingroup);
  }
}


bool calibration_channel_history :: GetTrackData (mopitt_time const& tracktime,
           float  trackdata [CALIBRATION_PIXELS] [CALIBRATION_STATES] [CALIBRATION_TERMS],
	   double trackcoldcal [CALIBRATION_PACKETPOSITION] [CALIBRATION_PIXELS] [PMC_SIGNAL_STATES])
{
  int pixel, state;
  bool isretrieved = true;
  calibration_channel_record* prerecord = NULL;
  calibration_channel_record* postrecord = NULL;
  
  GetBracketingRecords (tracktime, offsetlist, &prerecord, &postrecord);
  if (prerecord != NULL && postrecord != NULL)
    InterpolateTrackData (tracktime, prerecord, postrecord, trackdata, trackcoldcal);
  else if (prerecord != NULL)
    prerecord->GetData (trackdata, trackcoldcal);
  else if (postrecord != NULL)
    postrecord->GetData (trackdata, trackcoldcal);
  else {
    for (int pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) {
      for (int state = 0 ; state < CALIBRATION_STATES; state++)
        for (int term = 0; term < CALIBRATION_TERMS; term++)
          trackdata [pixel][state][term] = CALIBRATION_MISSING_VALUE;
      for ( int pos = 0; pos < CALIBRATION_PACKETPOSITION; pos++ )
	for ( int pmcstate = 0; pmcstate < PMC_SIGNAL_STATES; pmcstate++ )
	  trackcoldcal [pos][pixel][pmcstate] = CALIBRATION_MISSING_VALUE; 
    }
    isretrieved = false;
    string message = "No calibration data for track time ";
    message += tracktime.GetUTC ();
    if ( channel > 4 ) {
      diagnosticreporter.AddEntry (DIAGNOSTICS_WARNING, DIAGNOSTICS_CALIBRATIONS_MODULE, INTERPOLATION_WARNING, 0, 0, 0,
				   0, 0, channel, 0, 0, message);
    }
  }
  // set sector interanl raiance to missing values
  for ( pixel = 0; pixel < CALIBRATION_PIXELS; pixel++) 
    for ( state = 2 ; state < CALIBRATION_STATES; state++) {
      trackdata [pixel][state][CALIBRATION_TERM_RADIANCE_MEAN]  = CALIBRATION_MISSING_VALUE;
      trackdata [pixel][state][CALIBRATION_TERM_RADIANCE_ERROR] = CALIBRATION_MISSING_VALUE;
    }
  return isretrieved;
}

void calibration_channel_history :: GetNoiseData (
			 float noisedata [CALIBRATION_PIXELS] [POSITION_NOISE_STATES] [CALIBRATION_PACKETPOSITION] )
{
  float noise [CALIBRATION_PIXELS] [POSITION_NOISE_STATES] [CALIBRATION_PACKETPOSITION],
        posnoise [CALIBRATION_PIXELS] [POSITION_NOISE_STATES] [CALIBRATION_PACKETPOSITION];
  int count [CALIBRATION_PIXELS] [POSITION_NOISE_STATES] [CALIBRATION_PACKETPOSITION];
  int pixel, state, position;

  calibration_channel_noise_record* record = NULL; 
  for ( pixel = 0; pixel < CALIBRATION_PIXELS; pixel++ ) 
    for ( state = 0; state < POSITION_NOISE_STATES; state++ )
      for ( position = 0; position < CALIBRATION_PACKETPOSITION; position++ ){
	posnoise [pixel] [state] [position] = 0.0;
	count    [pixel] [state] [position] = 0;
      }
  noiselist.ResetNext ();

  while ((record = noiselist.GetNext ()) != NULL)  {
    record->GetData ( noise );

    for ( pixel = 0; pixel < CALIBRATION_PIXELS; pixel++ ) 
      for ( state = 0; state < POSITION_NOISE_STATES; state++ ) 
        for ( position = 0; position < CALIBRATION_PACKETPOSITION; position++ )

	  if ( noise [pixel] [state] [position] != CALIBRATION_MISSING_VALUE &&
	       noise [pixel] [state] [position] > 0.0 &&
	       isnan(noise [pixel] [state] [position]) == 0 ) { 
	    posnoise [pixel] [state] [position] += noise [pixel] [state] [position]; 
	    count    [pixel] [state] [position] += 1; 
	  }
  }

  for ( pixel = 0; pixel < CALIBRATION_PIXELS; pixel++ ) 
    for ( state = 0; state < POSITION_NOISE_STATES; state++ )
      for ( position = 0; position < CALIBRATION_PACKETPOSITION; position++ )
        if ( count [pixel][state][position] > 1 )
          noisedata [pixel] [state] [position] = posnoise [pixel] [state] [position] /count [pixel] [state] [position];
        else
	  noisedata [pixel] [state] [position] = 0.0;
}
  

void calibration_channel_history :: Read (PGSt_PC_Logical preinternallogical, PGSt_PC_Logical prespacelogical,
                                          PGSt_PC_Logical currentlogical, PGSt_PC_Logical postinternallogical,
                                          PGSt_PC_Logical postspacelogical, enum science_train_data_type datatype,
                                          science_train_group* traingroup, mopir_collection const& mopircollection,
					  processor_parameters const& processorparameters )
{
  // skip channel 1 to channel 4 for post-anomaly data 
  if ( ( (processorparameters.GetCh7Info()) == 1 && (channel > 4) ) ||
       ( (processorparameters.GetCh7Info()) == 0)  ) {
    // read the gain files and generate the gains, then read the offsets and generate offsets and interpolate gains
    ReadPreGranuleGains       (preinternallogical, datatype, traingroup, mopircollection );
    ReadCurrentGranuleGains   (currentlogical, datatype, traingroup, mopircollection);
    ReadPostGranuleGains      (postinternallogical, datatype, traingroup, mopircollection);
    ReadPreGranuleOffsets     (prespacelogical, datatype, traingroup);
    ReadCurrentGranuleOffsets (currentlogical, datatype, traingroup);
    ReadPostGranuleOffsets    (postspacelogical, datatype, traingroup);
    
    // delete the gains list (all gain info has been interpolated and added to the offset list)
    gainlist.Empty ();
  }
}
