/********** TimeConvertNCARToTAI.C ************************************************************************************\
 
 !C
 
 $Header$
 
 PURPOSE
   Converts NCAR-form time fields to TAI-form time.
 
 INPUT ARGUMENTS
   ncartime   const ncar_time*   The address of the NCAR-format time container
 
 INPUT/OUTPUT ARGUMENTS
   None
 
 OUTPUT ARGUMENTS
   taitime   double*   The address of the TAI-format time container

 RETURNS
   None
 
 ASSUMPTIONS ON ENTRY
   FILES
     None
   MISCELLANEOUS
     SCTIME_BYTES is 8
     The ncartime and taitime addresses are valid
 
 ASSUMPTIONS ON EXIT
   FILES
     None
   ERROR HANDLING
     Errors are handle via the SMF
   MISCELLANEOUS
     None
 
 ADDITIONAL COMMENTS
   None
 
 ORIGINAL VERSION
   07/24/96  Charles Cavanaugh

 MODIFIED VERSION
   5/99  Tom Lauren

 REVISION HISTORY
   $Log$

 !END
\**********************************************************************************************************************/
#include "PGS_SMF.h"
#include "PGS_TD.h"
#include "Time.h"
#include "System.h"

system_code TimeConvertNCARToTAI (const ncar_time *ncartime, double *taitime)
{
  system_code returncode = SYSTEM_SUCCESS;      // Return value indicating (un)successful operation.
  PGSt_SMF_status smfcode;
  PGSt_double offsets;
  PGSt_scTime sctime [SCTIME_BYTES];
  utc_string utctime;

  /*** convert the ints to scTimes ***/
  sctime[0] = (PGSt_scTime) (ncartime->day / 256);
  sctime[1] = (PGSt_scTime) (ncartime->day % 256);
  sctime[2] = (PGSt_scTime) (ncartime->msecs / 16777216);
  sctime[3] = (PGSt_scTime) ((ncartime->msecs % 16777216) / 65536);
  sctime[4] = (PGSt_scTime) ((ncartime->msecs % 65536) / 256);
  sctime[5] = (PGSt_scTime) (ncartime->msecs % 256);
  sctime[6] = 0;
  sctime[7] = 0;

  // Use the toolkit function to convert from spacecraft time format to UTC format.
  smfcode = PGS_TD_SCtime_to_UTC (PGSd_EOS_AM, &sctime, 1, utctime, &offsets);

  // If code is SUCCESS or PRED_LEAPS, convert to TAI time, else return a failure.
  if (smfcode == PGS_S_SUCCESS || smfcode == PGSTD_W_PRED_LEAPS) {
    if (TimeConvertUTCToTAI (utctime, taitime) != SYSTEM_SUCCESS)
      returncode = SYSTEM_FAILURE;
  } else
    returncode = SYSTEM_FAILURE;

  return returncode;
}
