/********* ScienceRemapStare.C ***************************************************************************************\

 !C

 $Header$

 PURPOSE
   Corrects the mirror information for a stare, if possible.

 INPUT ARGUMENTS
   sciencepacket   const science_packet*   The address of the science packet container.
   stare           int                     The stare of the packet.

 INPUT/OUTPUT ARGUMENTS
   outhandle       FILE* The handle for the output file.

 OUTPUT ARGUMENTS
   staredata       stare_data*             The address of the science stare container.

 RETURNS
   ON SUCCESS
     SYSTEM_SUCCESS
   ON FAILURE
     SYSTEM_FAILURE

 ASSUMPTIONS ON ENTRY
   The science packet address is valid and points to valid data.
   The stare number is less than the number of possible stares in a packet.
   The handle points to the output file.

 ASSUMPTIONS ON EXIT
   Either the mirror information for the stare was corrected, a '?' was placed in the id field of the 
     stare's header indicating that the stare's mirror information is unknown, or the stare was unmodified
     because of a bad header or a failed attempt to retrive the stare's time.

 ORIGINAL VERSION
   5/99  Tom Lauren

 REVISION HISTORY
   $Log$

 !END
\**********************************************************************************************************************/
#include <stdio.h>         // sprintf(), fwrite(), fputc(), ferror().
#include "PGS_IO.h"
#include "Science.h"
#include "System.h"
#include "Message.h"

system_code ScienceRemapStare (const science_packet *sciencepacket, int stare, long &starenumber, 
                               FILE* outhandle, int mirrorcorrectoff, int timecorrectoff)
{
  const double TOLERANCE1 = 0.550;                           // Time tolerance for a earth-earth, space-space, or 
                                                             //  black-black mirror position change.
  const double TOLERANCE2 = 1.000;                           // Time tolerance for a earth-space, space-black, or 
                                                             //  space-earth mirror position change.
  const double TOLERANCE3 = 1.450;                           // Time tolerance for a black-earth mirror position change.
                                                              
  const double MIN_TOLERANCE = 0.350;                        // Minimum tolerance to ensure monotomic data.

  static char previousmirrorinfo[MIRROR_INFO_SIZE];          // Mirror information for previous stare.
  static char previouspreviousmirrorinfo[MIRROR_INFO_SIZE];  // Mirror information for previous previous stare.
  static double previoustaitime = 0.0;                       // Previous stare's TAI time.
  static bool previouspreviousunknown = true;                // Flag used to note that the previous previous stare's mirror
                                                             //  information could not be determined.

  static double maximumtime = -1000.0;                       // The maximum time reached thus far in seconds.

  char mirrorinfo[MIRROR_INFO_SIZE];                         // Mirror information for current stare.
  system_code returncode = SYSTEM_SUCCESS;                   // Return value indicating (un)successful operation.
  double taitime;                                            // TAI time for the current stare.
  double timedifference;                                     // The time diffference between this stare and the previous 
                                                             //  stare.
  double gap;                                                // The time difference between this stare and the maximum
                                                             //  time thus far.
  int mirrorbyteindex;                                       // Index for a byte of mirror information in a stare header.
  char tempinfobyte;                                         // Temporary byte used for swapping.
  PGSt_IO_L0_Packet *stareptr;                               // Pointer to the stare data.
  int previouspreviouspos[5];                                // Previous previous mirror positions.
  int previouspos[5];                                        // Previous mirror positions.
  bool unknown = false;                                      // Flag used to note that this stare's mirror information
                                                             //   could not be determined (either to it being the first
                                                             //   stare in the file, the first stare after a time delay,
                                                             //   an invalid header id, or the TAI time for the stare
                                                             //   not being determined correctly).
  bool lowincrease = false;                                  // Flag used to note that a stare time increase was below the 
                                                             //  required threshold.
  char message[280];                                         // A warning or error message to be sent to the LogReport
                                                             //  file and/or screen.

  // Grabs the mirror information from the stare header.
  void GrabMirrorInfo (const PGSt_IO_L0_Packet*, char*);

  // Converts two characters to an integer;
  int Unpack2Bytes (const char*);

  // Given the mirror information bytes, determines the mirror positions for each of the 4 mirrors.
  void UnpackMirrorPositions (char*, int*);

  // Returns true if all 4 mirrors are in an earthview position.
  bool IsEarth (int*);

  // Returns true if all 4 mirrors are in a spaceview position.
  bool IsSpace (int*);

  // Returns true if all 4 mirrors are in a blackbody view position.
  bool IsBlack (int*);


  // Increment the stare number.
  starenumber++;

  // Initialize the stare pointer to the beginning of the next stare header.
  stareptr = (PGSt_IO_L0_Packet *) sciencepacket->data + CCSDS_HEADER_SIZE + (stare * STARE_SIZE);

  // Grab the mirror information from the stare header.
  GrabMirrorInfo (stareptr, mirrorinfo);

  // Check the stare header id.  If not S, T, U, V, or W, just copy the stare to the output file without 
  //  saving the stare's tai time or mirror information.  If the next stare is valid, a time gap will be
  //  detected and that stare will be treated accordingly.  Set the previouspreviousunknown
  //  flag to true to let the next stare know that the mirror information for the previous previous stare
  //  is not known.
  if ((stareptr[0] != 'S') && (stareptr[0] != 'T') && (stareptr[0] != 'U') && (stareptr[0] != 'V') && (stareptr[0] != 'W')) {
    // Generate message.
    sprintf(message, 
	    "Stare # %ld which begins at byte offset %ld had the hex value %x in its id instead of the ASCII equivalent of %c.", 
	    starenumber, MessageStareToByte(starenumber), stareptr[0], (char)(0x53 + stare));
    MessageOut(message);
    // Write the entire stare (including the header) and check for error.
    if (fwrite(stareptr, 1, STARE_SIZE, outhandle) != STARE_SIZE) {
      // Generate message.
      sprintf(message, 
	      "An error occurred trying to write stare # %ld which begins at byte offset %ld to the output file.", 
	      starenumber, MessageStareToByte(starenumber));
      MessageOut(message);
      returncode = SYSTEM_FAILURE;
    }
    previouspreviousunknown = true;
  }

  else {
    // Get the TAI time of the stare.  If an error occurs, just copy the stare to the output file without 
    //  saving the stare's tai time or mirror information.  If the next stare is valid, a time gap will be
    //  detected and that stare will be treated accordingly.  Set the previouspreviousunknown
    //  flag to true to let the next stare know that the mirror information for the previous previous stare
    //  is not known.
    if (ScienceGetTime (stareptr, &sciencepacket->time, &taitime) != SYSTEM_SUCCESS) {
      // Generate message.
      sprintf(message, 
	      "The stare time for stare # %ld which begins at byte offset %ld was not retrieved correctly.", 
	      starenumber, MessageStareToByte(starenumber));
      MessageOut(message);
      // Write the entire stare (including the header) and check for error.
      if (fwrite(stareptr, 1, STARE_SIZE, outhandle) != STARE_SIZE) {
	// Generate message.
	sprintf(message, 
		"An error occurred trying to write stare # %ld which begins at byte offset %ld to the output file.", 
		starenumber, MessageStareToByte(starenumber));
	MessageOut(message);
        returncode = SYSTEM_FAILURE;
      }
      previouspreviousunknown = true;
    }

    else {
      // Check if there is a large time gap between this stare and the previous stare.  Use a different tolerance
      //  for the three general cases ([ee,ss,bb], [es,sb,se], [be]).  To determine which tolerance is
      //  acceptable, we need to determine the mirror postions for both the previous and previous previous stare, if
      //  they are known.
      timedifference = taitime - previoustaitime;
      if (timedifference > TOLERANCE1) {
	if (previouspreviousunknown) unknown = true;
        else {
          UnpackMirrorPositions(previousmirrorinfo, previouspos);
          UnpackMirrorPositions(previouspreviousmirrorinfo, previouspreviouspos);
          if ((IsEarth(previouspreviouspos) && IsEarth(previouspos)) ||
	      (IsSpace(previouspreviouspos) && IsSpace(previouspos)) ||
	      (IsBlack(previouspreviouspos) && IsBlack(previouspos)))
	    unknown = true;
          else {
            if (timedifference > TOLERANCE2) {
	      if ((IsEarth(previouspreviouspos) && IsSpace(previouspos)) ||
		  (IsSpace(previouspreviouspos) && IsBlack(previouspos)) ||
		  (IsSpace(previouspreviouspos) && IsEarth(previouspos)))
		unknown = true;
	      else
		if (timedifference > TOLERANCE3)
		  unknown = true;
	    }
	  }
	}
      }
      // Also ensure that the stare time is greater than the previous maximum stare time plus a tolerance 
      //  and update the maximum stare time.
      if ((gap = (taitime - maximumtime)) < MIN_TOLERANCE) 
        lowincrease = true;
      else
        maximumtime = taitime;        

      // If a time gap greater than the appropriate tolerance was detected (and mirror position correction was not turned off)
      //  or time did not increase by atleast a threshold amount (and non-monotomic time correction was not turned off), 
      //  write the stare to the output file,  
      //  modified only by a '?' assigned to the id member of the stare's header.  Also, save the current stare's 
      //  mirror info and taitime, and set the previouspreviousunknown flag to true to let the next stare know that 
      //  the mirror information for the previous previous stare is not known.
      if ((unknown && !mirrorcorrectoff) || (lowincrease && !timecorrectoff)) {
	// Generate message.
        if (unknown && !mirrorcorrectoff) 
	  sprintf(message, 
		  "A large time gap of %g seconds was detected before stare # %ld which begins at byte offset %ld.", 
		  timedifference, starenumber, MessageStareToByte(starenumber));
        else
          sprintf(message,
                  "The time gap of %g seconds before stare # %ld which begins at byte offset %ld was below the minimum threshold of %g seconds.",
                  gap, starenumber, MessageStareToByte(starenumber), MIN_TOLERANCE); 
	MessageOut(message);
        // Write the modified stare id and check for error.
        fputc('?', outhandle);
        if (ferror(outhandle)) {
	  // Generate message.
	  sprintf(message, 
		  "An error occurred trying to write a '?' to the id of stare # %ld which begins at byte offset %ld to the output file.", 
		  starenumber, MessageStareToByte(starenumber));
	  MessageOut(message);
	  returncode = SYSTEM_FAILURE;
	}
        // Write the remainder of the stare (the 1-byte id is already written) and check for error.
        if (fwrite(stareptr + ID_SIZE, 1, STARE_SIZE - ID_SIZE, outhandle) != (STARE_SIZE - ID_SIZE)) {
	  // Generate message.
	  sprintf(message, 
		  "An error occurred trying to write stare # %ld which begins at byte offset %ld to the output file.", 
		  starenumber, MessageStareToByte(starenumber));
	  MessageOut(message);
          returncode = SYSTEM_FAILURE;
        }
	for (mirrorbyteindex = 0; mirrorbyteindex < MIRROR_INFO_SIZE; mirrorbyteindex++) {
	  previousmirrorinfo[mirrorbyteindex] = mirrorinfo[mirrorbyteindex];
	}
        previoustaitime = taitime;
        previouspreviousunknown = true;
      } 

      // Otherwise (stare id ok, tai time retrieved ok, no time gap) if mirror positioin correction was not turned off,
      //  swap the previous stare's mirror 
      //  info with the current stare's mirror info, save the new previous previous mirror info, save the current 
      //  stare's tai time, and write the stare with the corrected mirror information to the output file.
      //  Also, set the previouspreviousunknown flag to false to let the next stare know 
      //  that the mirror information for the previous previous stare is known.
      else { 
	for (mirrorbyteindex = 0; mirrorbyteindex < MIRROR_INFO_SIZE; mirrorbyteindex++) {
	  tempinfobyte = mirrorinfo[mirrorbyteindex];
	  mirrorinfo[mirrorbyteindex] = previousmirrorinfo[mirrorbyteindex];
	  previouspreviousmirrorinfo[mirrorbyteindex] = previousmirrorinfo[mirrorbyteindex];
	  previousmirrorinfo[mirrorbyteindex] = tempinfobyte;
      	}
        previoustaitime = taitime;
        // Write the unmodified id, bitmask, word count, time, and DMA portion of the stare header to the output file,
        //   checking for error.
        if (fwrite(stareptr, 1, ID_SIZE + BITMASK_SIZE + WORDCOUNT_SIZE + TIME_SIZE + DMA_SIZE, 
		   outhandle) != ID_SIZE + BITMASK_SIZE + WORDCOUNT_SIZE + TIME_SIZE + DMA_SIZE){
	  // Generate message.
	  sprintf(message, 
  "An error occurred trying to write the unmodified header portion of stare # %ld which begins at byte offset %ld to the output file.", 
		  starenumber, MessageStareToByte(starenumber));
	  MessageOut(message);
	  returncode = SYSTEM_FAILURE;
        }
	// If mirror position correction was turned off, write the unmodified mirror information and check for error.
        if (mirrorcorrectoff) {
          if (fwrite (stareptr + STARE_HEADER_SIZE - MIRROR_INFO_SIZE, 1, MIRROR_INFO_SIZE, outhandle) != MIRROR_INFO_SIZE) {
            sprintf (message, 
                     "An error occurred trying to write the unmodified mirror information for stare # %ld which begins at byte offset %ld to the output file.",
                  starenumber, MessageStareToByte(starenumber));
            MessageOut(message);
            returncode = SYSTEM_FAILURE;
          }
        }
        // otherwise (mirror position correction not turned off) write the modified mirror info and check for error.
        else {
	  for (mirrorbyteindex = 0; mirrorbyteindex < MIRROR_INFO_SIZE; mirrorbyteindex++) {
	    if (!ferror(outhandle)) fputc(mirrorinfo[mirrorbyteindex], outhandle); 
	  }
	  if (ferror(outhandle)) {
	    sprintf(message, 
 "An error occurred trying to write the modified mirror information for stare # %ld which begins at byte offset %ld to the output file.", 
		  starenumber, MessageStareToByte(starenumber));
	    MessageOut(message);
	    returncode = SYSTEM_FAILURE;
	  }
	}
        // Write the stare body and check for error.
	if (fwrite(stareptr + STARE_HEADER_SIZE, 1, STARE_SIZE - STARE_HEADER_SIZE, 
		   outhandle) != STARE_SIZE - STARE_HEADER_SIZE) {
	  sprintf(message, 
 "An error occurred trying to write the unmodified stare body of stare # %ld which begins at byte offset %ld to the output file.", 
		  starenumber, MessageStareToByte(starenumber));
	  MessageOut(message);
	  returncode = SYSTEM_FAILURE;
	}
        previouspreviousunknown = false;
      }
    }
  }

  return returncode;
}

// Grabs the mirror information from the stare header.
void GrabMirrorInfo (const PGSt_IO_L0_Packet *stareptr, char *mirrorinfo)
{
  int mirrorbyteindex;

  for (mirrorbyteindex = 0; mirrorbyteindex < MIRROR_INFO_SIZE; mirrorbyteindex++) {
    mirrorinfo[mirrorbyteindex] = *(stareptr + MIRROR_INFO_OFFSET + mirrorbyteindex);
  }
}

// Converts two characters to an integer;
int Unpack2Bytes (const char* chptr)
{
  short unpackedvalue;

  unpackedvalue = 0;
  unpackedvalue |= *(chptr + 0);
  unpackedvalue <<=8;
  unpackedvalue |= *(chptr + 1);
  return (int) unpackedvalue;
}

// Given the mirror information bytes, determines the mirror positions for each of the 4 mirrors.
void UnpackMirrorPositions (char *info, int *pos)
{
  pos[1] = Unpack2Bytes(info);
  pos[2] = Unpack2Bytes(info + 2);
  pos[3] = Unpack2Bytes(info + 4);
  pos[4] = Unpack2Bytes(info + 6);
}

// Returns true if all 4 mirrors are in an earthview position.
bool IsEarth (int *pos)
{
  if ((pos[1] >= -14) && (pos[1] <= 14) && (pos[2] >= -14) && (pos[2] <= 14) && 
      (pos[3] >= -14) && (pos[3] <= 14) && (pos[4] >= -14) && (pos[4] <= 14))
    return true;
  else
    return false;
}

// Returns true if all 4 mirrors are in a spaceview position.
bool IsSpace (int *pos)
{
  if ((pos[1] == -50) && (pos[2] == -50) && (pos[3] == -50) && (pos[4] == -50))
    return true;
  else
    return false;
}

// Returns true if all 4 mirrors are in a blackbody view position.
bool IsBlack (int *pos)
{
  if ((pos[1] == -100) && (pos[2] == -100) && (pos[3] == -100) && (pos[4] == -100))
    return true;
  else
    return false;
}
