/********** ScienceGetTime.C ****************************************************************************************\

 !C

 $Header$

 PURPOSE
   Retrieves the time of a stare in TAI format.

 INPUT ARGUMENTS
   timeptr       const PGSt_IO_L0_Packet*      The address of the packet data.
   packettime    const ncar_time*              The address of the packet time container.

 INPUT/OUTPUT ARGUMENTS
   None

 OUTPUT ARGUMENTS
   taitime   double*   The address of the converted time.

 RETURNS
   ON SUCCESS
     SYSTEM_SUCCESS
   ON FAILURE
     SYSTEM_FAILURE

 ASSUMPTIONS ON ENTRY
   The addresses are all valid and the packettime container has valid data.

 ASSUMPTIONS ON EXIT
   taitime points to the correct double value.

 ADDITIONAL COMMENTS
   None

 ORIGINAL VERSION
   01/23/98  Charles Cavanaugh

 MODIFIED VERSION
   5/99  Tom Lauren

 REVISION HISTORY
   $Log$

 !END
\**********************************************************************************************************************/

#include "Science.h"
#include "System.h"
#include "Time.h"

system_code ScienceGetTime (const PGSt_IO_L0_Packet *timeptr, const ncar_time *packettime, double *taitime)
{
  system_code returncode = SYSTEM_SUCCESS;           // Return value indicating (un)successful operation.
  int packetday, packetmsecs, telemday, telemmsecs;  // Day and msec values for the packet time and stare time.
  ncar_time telemtime;                               // NCAR time for the stare.

  // Get the packet day and time.
  TimeGetNCAR (packettime, &packetday, &packetmsecs);

  // Build the stare milliseconds.
  telemmsecs = (*(timeptr + STARETIME_MSW_MSB_OFFSET) * 16777216) +
               (*(timeptr + STARETIME_MSW_LSB_OFFSET) *    65536) +
               (*(timeptr + STARETIME_LSW_MSB_OFFSET) *      256) +
               (int) *(timeptr + STARETIME_LSW_LSB_OFFSET);

  /*** If the number of milliseconds of of a stare is greater than the number of milliseconds  ***
   *** possible in a day, then subtract the maximum number of milliseconds in a day from the   ***
   *** time.                                                                                   ***/

  if (telemmsecs > 86400000)
    telemmsecs -= 86400000;

  // Build the stare day.
  /*** The packet time contains the time of the end of the last stare in the packet, so if a  ***
   *** stare time is greater than the packet time, that means that this packet overlaps a     ***
   *** day boundary, and the stare day is actually the day before the packet day.             ***/

  if (telemmsecs > packetmsecs)
    telemday = packetday - 1;
  else
    telemday = packetday;

  // Save the stare day and msecs as NCAR time.
  TimeSetNCAR (telemday, telemmsecs, &telemtime);

  // Convert the stare's NcAR time to TAI time.
  if (TimeConvertNCARToTAI (&telemtime, taitime) != SYSTEM_SUCCESS)
    returncode = SYSTEM_FAILURE;

  return returncode;
}
