/********* ScienceGetPacket.C *****************************************************************************************\

 !C

 $Header$

 PURPOSE
   Gets the next packet from the Level 0 science telemetry file.

 INPUT ARGUMENTS
   None

 INPUT/OUTPUT ARGUMENTS
   handle   sci_handle*   The address of the science telemetry file info container

 OUTPUT ARGUMENTS
   scipacket   science_packet*   The address of the science packet data container

 RETURNS
   ON SUCCESS
     SYSTEM_SUCCESS
   ON FAILURE
     SYSTEM_FAILURE

 ASSUMPTIONS ON ENTRY
     The science telemetry L0 file has been opened, and handle is a handle to it.
     The science packet address is valid.

 ASSUMPTIONS ON EXIT
     The science packet contains the latest packet read from the science telemetry file.

 ORIGINAL VERSION
   09/25/97  Charles Cavanaugh

 MODIFIED VERSION
   5/11  Tom Lauren

 REVISION HISTORY
   $Log$

 !END
\**********************************************************************************************************************/

#include <string.h>       // memset().
#include "PGS_SMF.h"
#include "PGS_IO.h"
#include "Science.h"
#include "Time.h"
#include "System.h"

system_code ScienceGetPacket (science_inhandle *inhandle, science_packet *sciencepacket)
{
  system_code returncode = SYSTEM_SUCCESS;  // Return value indicating (un)successful operation.
  PGSt_SMF_status packstat;

  // Initialize the packet holder to all zeros.
  (void) memset ((void *) &sciencepacket->data, 0, SCI_PACKET_SIZE);

  // Get a packet of data from the L0 science telemetry stream.
  packstat = PGS_IO_L0_GetPacket (ScienceFileID(inhandle), SCI_PACKET_SIZE, sciencepacket->data);

  // If the data was not read successfully...
  if (packstat != PGSIO_W_L0_END_OF_VIRTUAL_DS && packstat != PGSIO_M_L0_HEADER_CHANGED && packstat != PGS_S_SUCCESS) {
    returncode = SYSTEM_FAILURE;
  }
  else {

    // If an EOI was encountered, set the eoiflag.
    if (packstat == PGSIO_W_L0_END_OF_VIRTUAL_DS) 
      ScienceSetInputFlag(inhandle);

    // Convert the CCSDS header time to NCAR time.
    TimeConvertPacketToNCAR ((sciencepacket->data + PACKET_TIME_OFFSET), &sciencepacket->time);
  }

  return returncode;
}
