/**********Preprocessor.C *********************************************************************************************\

 PURPOSE
   Corrects mirror positions in L0 science files by assigning the previous stare's mirror position to a current 
             stare's mirror position.  Assigns a '?' to a stare header if a large time gap is found before a stare.  
   In addition to correcting the mirror positions, assigns a '?' to a stare header if the time of the current stare
             is not greater than the maximum time reached thus far (this discards stares with non-monotonic time issues).
             Uses the PCF file to find the L0 science files, but writes the modified L0 science files as
             <input filelogical number>.remapped.
   Accepts command-line argument flags to tell the preprocessor not to correct mirror positions or not to handle
             non-monotonic time issues.


 INPUT ARGUMENTS
   1) Flag to turn off mirror position correction (Non-zero = turn off mirror position correction).
   2) Flag to turn off non-monotomic time correction (Non-zero = turn off non-monotomic time correction).

   ex:  Preprocessor 1 0 

        This turns off mirror position correction but leaves on non-monotomic time correction.

   The default (no parameters passed to Preprocessor) is for both flags to be zero (nothing turned off).

   If only one flag is passed to Preprocessor, it assumed to be the mirror position correction flag.

 INPUT/OUTPUT ARGUMENTS
   None

 OUTPUT ARGUMENTS
   None

 RETURNS
   None

 ASSUMPTIONS ON ENTRY
   The PGE PCF has staged all required files.

 ASSUMPTIONS ON EXIT
   FILES
     The mirror information for each stare has been corrected, and stares with unknown mirror information
       have been tagged with a '?' in their respective stare header id fields.
   OBJECTS
     None
   ERROR HANDLING
     Errors detected during PGE run have been written to the LogStatus file.
   MISCELLANEOUS
     PGE information has been written to the LogReport file.
     A running status log has been written to the LogReport file.

 ORIGINAL VERSION
   5/99   Tom Lauren

\**********************************************************************************************************************/
#include <stdlib.h>    // exit(), EXIT_FAILURE, EXIT_SUCCESS.
#include <stdio.h>     // sprintf().
#include "Science.h"  
#include "Message.h"
#include "System.h"

int main (int argc, char* argv[])
{
  const int NUMBER_OF_FILES = 12;                     // The number of files to be remapped.
  const int FILE_LOGICAL_BASE = 41000;                // The value of the first file logical to be remapped.

  int                     filenumber;                 // Index for the file number being remapped.
  PGSt_PC_Logical         infilelogical;              // File logical for level 0 science data input file.
  science_inhandle        inhandle;                   // Handle for level 0 science data input file.
  FILE*                   outhandle;                  // Handle for level 0 science data output file.

  long           packetnumber = 0;    // A packet's position within the entire file, with 1 being the first packet, etc.
  long           starenumber = 0;     // A stare's position within the entire file, with 1 being the first stare, etc.
  int            stare;               // Index for the 5 stares of a packet.
  science_packet sciencepacket;       // A file packet structure consisting of the file packet data and NCAR time.
  char           message[280];        // A warning or error message to be sent to the LogReport
                                      //  file and/or screen.
  int            mirrorcorrectoff = 0;    // Flag to turn off mirror position correction.
  int            timecorrectoff = 0;      // Flag to turn off non-monotomic time correction.
 
  // Get the command-line arguments, if any.
  if (argc > 3) 
    printf ("\nUsage:  L0Remap [mirror_correction_off_flag] [non-monotomic_time_correction_off_flag]\n\n");
  else if (argc == 3) {
    mirrorcorrectoff = atoi (argv[1]);
    timecorrectoff = atoi (argv[2]); 
  }
  else if (argc == 2) 
    mirrorcorrectoff = atoi (argv[1]);
  
  if (mirrorcorrectoff)
    MessageOut ("Mirror positions have been selected to not be corrected.\n");
  if (timecorrectoff)
    MessageOut ("Non-monotomic timing problems have been selected to not be corrected.\n");

  // Repeat for each of the files to be remapped.
  for (filenumber = 0; filenumber < NUMBER_OF_FILES; filenumber++) {

    // Assign the file logical.
    infilelogical = (PGSt_PC_Logical)(FILE_LOGICAL_BASE + filenumber);

    // Generate message noting the file logical being remapped.
    sprintf(message, "Beginning to remap filelogical %d.", infilelogical);
    MessageOut(message);

    // Open the level 0 input files.
    if (ScienceOpenFiles (infilelogical, &inhandle, &outhandle) != SYSTEM_SUCCESS) {
      // Generate message.
      MessageOut("Could not open L0 file.  Program aborted.");
      exit (EXIT_FAILURE);
    }

    
    // Check if there is another packet to be processed.
    while (SciencePacketsPresent (&inhandle) == SYSTEM_TRUE) {
      
      packetnumber++;

      // Get a packet from the input file, checking for error.
      if (ScienceGetPacket (&inhandle, &sciencepacket) != SYSTEM_SUCCESS) {
        // Generate message.
        sprintf(message, 
		"Packet # %ld which begins at byte offset %ld was not retrieved and was replaced with a dummy packet.", 
		packetnumber, (packetnumber - 1) * SCI_PACKET_SIZE);
        MessageOut(message);
        starenumber +=5;
        // Write a dummy packet to the output file, checking for error.
        if (ScienceWriteDummyPacket(outhandle, packetnumber) != SYSTEM_SUCCESS) {
	  // Generate message.
	  MessageOut("Could not write to output file.  Program aborted.");
	  exit (EXIT_FAILURE);
        }
      }
      else {
        // Write the packet header to the output file, checking for error.
        if (ScienceWritePacketHeader (&sciencepacket, outhandle, packetnumber) != SYSTEM_SUCCESS) {
	  // Generate message.
	  MessageOut("Could not write to output file.  Program aborted.");
	  exit (EXIT_FAILURE);
        }
        else {
	  // For each stare in the packet...
	  for (stare = 0; stare < STARES_PER_PACKET; stare++) {
	    
	    // Correct the mirror information, if possible.
	    if (ScienceRemapStare (&sciencepacket, stare, starenumber, outhandle, mirrorcorrectoff, timecorrectoff) 
                 != SYSTEM_SUCCESS) {
	      // Generate message.
	      MessageOut("Could not write to output file.  Program aborted.");
	      exit (EXIT_FAILURE);
	    }
	  }
        }
      }
    } 
    
    if (ScienceCloseFiles (&inhandle, &outhandle) != SYSTEM_SUCCESS) {
      // Generate message.
      MessageOut("Could not close files.  Program aborted.");
      exit (EXIT_FAILURE);
    }

    // Generate success message noting the filelogical of the file that was remapped.
    sprintf(message, "Filelogical %d was successfully remapped.", infilelogical);
    MessageOut(message);
  }

  // Generate success message noting the number of files remapped.
  sprintf(message, "All %d files were successfully remapped.", filenumber);
  MessageOut(message);
  return EXIT_SUCCESS;
}
