pro gather4, file=file, outfile=outfile, plt=plt, fltr=fltr, maxrms=maxrms, incldErr=incldErr

close,/all
;******************************************************************************************************************
;
; Purpose: This program is designed to collect information from multiple sfit4 fits for storing in IDL save file and
;          eventualy HDF file for submission. The *.lst file dictates which sfit4 fits to process
;
;
;
; Keywords:
;   file       --  Name of list file which contains a list of data directories
;   outfile    --  Name of output IDL sav file
;   plt        --  Command to do intermediate plots
;   fltr       --  Command to filter data
;   maxrms     --  Maximum value of rms to use for filtering
;   incldErr   --  Command to include error analysis in structure
;
;
; Notes:
;  From list file we get the following parameters:
;     1) Directory of output
;     2) Date
;     3) Time
;
;  From spectral database file we get the following parameters:
;     1) Date
;     2) Time
;     3) Opus filename
;     4) SNR -- singal to noise ratio
;     5) Latitude [North]
;     6) Longitude [West]
;     7) Altitude [m]
;     8) Solar azmith angle
;     9) Solar zenith angle
;     10) Scan duration
;
;
;******************************************************************************************************************

;-------------------------------------------------
; Initialize subroutines/functions used in program
;-------------------------------------------------
forward_function bldstrct

funcs = [ 'usesite', 'usemol', 'readrfmd4', 'readnxn4', 'readprfs4', 'readlayr4', 'readsum4', 'readstat4', 'bldstrct','readnxn4e','readerrsum']
resolve_routine, funcs, /either
resolve_all
;---------------------------------
; Check user input in program call
;---------------------------------
if( ~keyword_set(file) ) then begin
   print, ' example usage : gather4, file="/home/usr/mlo_1_1_13.lst" [, outfile="/home/usr/MLO_2013.sav", /plt] '
   stop
endif

;---------------------------------
; If user does not specify outfile
; set default
;---------------------------------
if ~keyword_set(outfile) then begin
  pos = strpos( file, '.' )
  outfile =  '/' + strmid( file, 0, pos ) + '.sav'
  cd, current=wrkDir
  outfile = wrkDir + outfile
endif

;-------------------------
; Open and read *.lst file
;-------------------------
; Determine number of lines in file
nlines = file_lines(file)

; Open file
openr, fid_lstFile, file, /get_lun, error=ioerr
if( ioerr ne 0 ) then begin
  printf, -2, !err_string
  free_lun, fid_lstFile
  stop
endif

;---------------------------------------------
; Read header and count lines. From the header
; get the following information:
; 1) Primary Gas
; 2) Spectral DB file
; 3) Location
; 4) Station layers file
;---------------------------------------------
buf = ''
i   = 0
while ~strcmp(buf, 'Date', 4, /fold_case )  do begin
  readf, fid_lstFile, buf
  if strcmp(buf, 'statnLyrs_file', 14, /fold_case) then begin
    subs       = strsplit( buf,' ',/extract, count=nitms )
    stlyrFile  = subs[nitms-1]
  endif
  if strcmp(buf, 'primGas', 7, /fold_case)         then begin
    subs = strsplit( buf,' ',/extract, count=nitms )
    mol  = subs[nitms-1]
  endif
  if strcmp(buf, 'specDBfile', 10, /fold_case)     then begin
    subs       = strsplit( buf,' ',/extract, count=nitms )
    specDBfile = subs[nitms-1]
  endif
  if strcmp(buf, 'site', 4, /fold_case)            then begin
    subs = strsplit( buf,' ',/extract, count=nitms )
    site = subs[nitms-1]
  endif
  i++
endwhile

; Determine remaining lines
nsize = nlines-i
lDates   =  strarr(nsize)                  ; Create array for reading in Dates
lTstamps =  strarr(nsize)                  ; Create array for reading in timestamps
lDirs    =  strarr(nsize)                  ; Create array for reading in directories

; Read relavent data
for j = 0, nsize-1 do begin
  readf, fid_lstFile, buf
  subs = strsplit( buf,' ',/extract, count=nitms)
  lDates[j]   = subs[0]
  lTstamps[j] = subs[1]
  lDirs[j]    = subs[2]
endfor

; Close file
free_lun, fid_lstFile

; based on the site, set a few more variables
lcmol = STRLOWCASE(mol)
ucmol = STRUPCASE(mol)
ucsite= STRUPCASE(site)

;-----------------------------------------------------------
; Gather information on the station location and primary gas
;-----------------------------------------------------------
usemol, ucsite, ucmol, Ag     ; Returns Ag
usesite, ucsite, As, Ag       ; Returns As

;------------------------------------------
; Search for duplicate entries in list file
; and remove
;------------------------------------------
newDate    = (lDates + 0D)*1000000 + (lTstamps + 0D)
unq_ind    = uniq(newDate,sort(newDate))

;for i=0, nsize-1 do print, i, unq_ind[i]

if ( nsize ne size(unq_ind,/n_elements) ) then begin    ; Found duplicate entries
  lDates   = lDates[unq_ind]
  lTstamps = lTstamps[unq_ind]
  lDirs    = lDirs[unq_ind]
  nsize    = size(lDates,/n_elements)
endif

;-------------------------------------------
; Get layering info from station.layers file
;-------------------------------------------
klay    = readlayr4( grd, stlyrFile )
nlayers = klay-1

;--------------------------------------------------
; Now that we have the number of spectra and number
; of layers for the spectra we can initialize the
; datastructure to read data to
;--------------------------------------------------
ds = bldstrct( nsize, nlayers )

;-------------------------------------
; Open and read spectral database file
;-------------------------------------
; Open file
openr, fid_specDBfile, specDBfile, /get_lun, error=ioerr
if( ioerr ne 0 ) then begin
  printf, -2, !err_string
  stop
endif

; Determine number of lines in file
nlines = file_lines(specDBfile)

; First line of spectral DB is the header
buf = ''
readf, fid_specDBfile, buf
header = strsplit(buf,' ', /extract, count=nheaders)

;---------------------------------------------
; Determine if this is a coadded database file
;---------------------------------------------
if strcmp(header[0],'Filename1') then coaddFlg = 1 else coaddFlg = 0

;---------------------------------------------------
; Headers for data to extract from spectral DB file
; Coadded database has an extra column in the header
;---------------------------------------------------
if not coaddFlg then begin
  getHdrs = ['Filename','TStamp','Date','SNR','N_Lat','W_Lon','Alt','SAzm','SZen','Dur','HouseRH','ExtStatRH','Ext_Solar_Sens','Quad_Sens','Det_Intern_T_Swtch']
  ;Hdr ind      0         1        2      3     4       5       6     7      8      9      10         11           12             13               14
  indoff = 0
endif else begin
  getHdrs = ['Filename1','Filename2', 'TStamp','Date','SNR','N_Lat','W_Lon','Alt','SAzm','SZen','Dur','HouseRH','ExtStatRH','Ext_Solar_Sens','Quad_Sens','Det_Intern_T_Swtch']
  ;Hdr ind      0                1        2      3     4       5       6     7      8      9      10         11           12             13         14        15
  indoff = 1
endelse

nhdrs    = intarr(size(getHdrs,/n_elements))

; Find the indices for each header to extract
for i = 0, (size(getHdrs,/n_elements)-1) do begin
  nhdrs[i] = where(header eq getHdrs[i])
endfor

; Read data in spectral database
buf = ''
j = 0
for i = 0, (nlines-2) do begin

;print, i, j, nlines, nsize
  readf, fid_specDBfile, buf
  subs = strsplit( buf, ' ', /extract, count=nitms )

  tempDate   = subs[nhdrs[indoff+2]]
  tempTstamp = subs[nhdrs[indoff+1]]

  ; If date and timestamp from DB match an entry in list file grab data
  testMatch = strmatch(lDates, tempDate) + strmatch(lTstamps, tempTstamp)
  ;help, testmatch

  indMatch  = where(testMatch eq 2)
  ;print, indmatch

  if ( size(indMatch,/n_elements) gt 1 ) then begin
    print, 'Duplicate Entries in list file found'
    exit
  endif

  if indMatch ge 0 then begin
    ; Coadded database has two spectral files for each retrieval
    if not coaddFlg then begin
      ds[j].spectraname    = subs[nhdrs[0]]
    endif else begin
      ds[j].spectraname    = subs[nhdrs[0]] + '_' + subs[nhdrs[1]]
    endelse
    ds[j].hhmmss         = subs[nhdrs[indoff+1]]
    ds[j].yyyymmdd       = subs[nhdrs[indoff+2]]
    ds[j].snr            = subs[nhdrs[indoff+3]]  + 0.0D0
    ds[j].latitude       = subs[nhdrs[indoff+4]]  + 0.0D0
    ds[j].longitude      = subs[nhdrs[indoff+5]]  + 0.0D0
    ds[j].alt_instrument = subs[nhdrs[indoff+6]]  + 0.0D0
    ds[j].azi            = subs[nhdrs[indoff+7]]  + 0.0D0
    ds[j].sza            = subs[nhdrs[indoff+8]]  + 0.0D0
    ds[j].int_time       = subs[nhdrs[indoff+9]]  + 0.0D0
    ds[j].external_solar_sensor       = subs[nhdrs[indoff+12]] + 0.0D0
    ds[j].guider_quad_sensor_sum      = subs[nhdrs[indoff+13]] + 0.0D0
    ds[j].detector_intern_temp_switch = subs[nhdrs[indoff+14]] + 0.0D0
    ds[j].directory      = lDirs[indMatch]

    ; Inputs related to station.layers file
    ds[j].alt_index = indgen(nlayers, /long) + 1
    ds[j].altitude  = grd.midp[0:nlayers-1]

    ;----------------------------------------
    ; Determine Outside RH. First choice is
    ; external station RH. If this is missing
    ; default to house log RH
    ;----------------------------------------
    HouseRH  = subs[nhdrs[indoff+10]] + 0.0D0
    ExtStaRH = subs[nhdrs[indoff+11]] + 0.0D0

    if ( ExtStaRH ge 0 ) then begin
      ds[j].outside_relative_humidity = ExtStaRH
    endif else begin
      ds[j].outside_relative_humidity = HouseRH
    endelse
    j++
  endif

;print, ds[j-1].yyyymmdd, '  ', ds[j-1].hhmmss, j-1

endfor

;-----------------------------
; Close spectral database file
;-----------------------------
free_lun, fid_specDBfile

;------------------------------------------------
; Manipulate Date and Tstamps to get YYYY, MM, DD
; DOY, etc and convert Date and Tstamp to int
;------------------------------------------------
jd2000   = julday(1,1,2000, 00, 00, 00)

for i = 0, nsize-1 do begin
  ds[i].year  = strmid(ds[i].yyyymmdd,0,4) + 0
  ;print, i, '  ', ds[i].yyyymmdd,  '  ', ds[i].year, '  ', ds[i].hhmmss
  ds[i].month = strmid(ds[i].yyyymmdd,4,2) + 0
  ds[i].day   = strmid(ds[i].yyyymmdd,6,2) + 0
  hh          = strmid(ds[i].hhmmss,0,2)   + 0
  mm          = strmid(ds[i].hhmmss,2,2)   + 0
  ss          = strmid(ds[i].hhmmss,4,2)   + 0

  ; date conversions

  ds[i].doy      = julday( ds[i].month, ds[i].day, ds[i].year, hh, mm, ss )  - julday( 1, 1, ds[i].year, 0, 0, 0 ) +1
  ds[i].hrs      = double(hh) + (double(mm) + double(ss) /60.0d) /60.0d
  daysinyear     = julday( 12, 31, ds[i].year, 0, 0, 0 )  - julday( 1, 1, ds[i].year, 0, 0, 0 ) +1
  ds[i].tyr      = ds[i].year + (ds[i].doy + ds[i].hrs/24.d)/daysinyear
  ds[i].datetime = julday( ds[i].month, ds[i].day, ds[i].year, hh, mm, ss ) - jd2000
  ds[i].jday     = julday( ds[i].month, ds[i].day, ds[i].year, hh, mm, ss )
endfor

;--------------------------------------------
; Create Dummy variables. These are variables
; that are to be stored in the save files;
; however, are not currently read in. Set
; to missing value -999
;--------------------------------------------
ds.zcorrect = make_array(nsize,/float, value=-999)

                ;--------------------------------------------------;
                ; Loop through observations to collect output data ;
                ;--------------------------------------------------;
for i = 0, nsize-1 do begin

   ;print, 'Working directory : ', i, '  ', ds[i].directory
   ;----------------------------------------
   ; Read summary file from output directory
   ;----------------------------------------
   ; Set indicator to find missing entries
   ds[i].iterations = -999
   ;print, i, ds[i].directory
   dum = readsum4( sumf, ds[i].directory + 'summary' )
   if( dum NE 0 )then continue

   ds[i].iterations = sumf.itr
   ds[i].rms        = sumf.fitrms
   ds[i].dofs       = sumf.dofstrg

  ;----------------------------------------
  ; Read statvec file from output directory
  ;----------------------------------------
  statvecFname = ds[i].directory + 'statevec'
  dum = readstat4( statvecData, statvecFname )

  ds[i].p      = statvecData.p[0:nlayers-1]
  ds[i].t      = statvecData.t[0:nlayers-1]
  ds[i].aprtc  = statvecData.col[0,0]
  ds[i].rettc  = statvecData.col[1,0]
  ds[i].aprvmr = statvecData.vmr[0,0,0:nlayers-1]
  ds[i].retvmr = statvecData.vmr[1,0,0:nlayers-1]

  dum = where((statvecData.pnam[0:statvecData.nprm-1] eq 'BckGrdSlp'), count)
  if ( count ge 1 ) then ds[i].bslope = mean(statvecData.prm[1,dum])

  dum = where(((statvecData.pnam[0:statvecData.nprm-1] eq 'SWNumShft' ) or ( statvecData.pnam[0:statvecData.nprm-1] eq 'IWNumShft')), count)
  if ( count ge 1 ) then ds[i].wshift = mean(statvecData.prm[1,dum])

  dum = where((statvecData.pnam[0:statvecData.nprm-1] eq 'ZeroLev'), count)
  if ( count ge 1 ) then ds[i].zerolev = mean(statvecData.prm[1,dum])

  dum = where((statvecData.pnam[0:statvecData.nprm-1] eq 'SPhsErr'), count)
  if ( count ge 1 ) then ds[i].phase = mean(statvecData.prm[1,dum])

  ds[i].alt_boundaries[0,*] = grd.alts[1:nlayers]
  ds[i].alt_boundaries[1,*] = grd.alts[0:nlayers-1]

  ;--------------------------------------------------------------
  ; Read surface pressure and temperature from reference.prf file
  ;--------------------------------------------------------------

  dum = readrfmd4( refm, ds[i].directory + 'reference.prf', /zpt )

  ; Determine if alt decreasing: updn=1 or alt increasing: updn=0
  if ( refm.updn eq 1 ) then begin
    ds[i].surface_pressure    = refm.prfs[1,-1]
    ds[i].surface_temperature = refm.prfs[2,-1]
  endif else if ( ref.updn eq 2 ) then begin
    ds[i].surface_pressure    = refm.prfs[1,0]
    ds[i].surface_temperature = refm.prfs[2,0]
  endif

  ;-------------------------------------
  ; Read Averaging Kernal data AK.target
  ;-------------------------------------
  dum = readnxn4( akData, ds[i].directory + 'ak.target' )
  if (akData.n ne nlayers ) then stop, 'akData.n and nlayers NOT equal'

  ; Normalize Averaging Kernal relative to a priori
  for ii = 0, nlayers-1  do begin
    for jj = 0, nlayers-1 do begin
      ; Because the rows are stored in second indicies we multiple by apriori[jj]/apriori[ii] rather than apriori[ii]/apriori[jj]
      ds[i].ak[ii,jj] = akData.mat[ii,jj] * ( ds[i].aprvmr[jj] / ds[i].aprvmr[ii] )
    endfor
  endfor

  ;-----------------------------------------------
  ; Calculate the total column averaging kernel.
  ; Need to get retrieved airmass for total column
  ; averaging kernel from rprs.table
  ;-----------------------------------------------
  dum = readprfs4( rprfs, ds[i].directory + 'rprfs.table' )
  tmpsum   = fltarr(nlayers)
  ds[i].ms = rprfs.a                                    ; Air-mass profile (same in a prior prf table or retreived prf table)

  ;--------------------------
  ; Get retrieved H2O profile
  ;--------------------------
  H2Oind        = where(rprfs.name eq 'H2O', /null)
  ds[i].h2o_vmr = rprfs.vmr[*,H2Oind]
  ds[i].h2o_tc  = total( rprfs.vmr[*,H2Oind] * rprfs.a )

  for jj = 0, nlayers-1 do begin
    for ii = 0, nlayers-1 do begin
      tmpsum[ii] = rprfs.a[ii] * ds[i].ak[ii,jj]
    endfor
    ds[i].aktc[jj] = ( total(tmpsum) ) / rprfs.a[jj]
    ds[i].sens[jj] = total( ds[i].ak[*,jj] )            ; This is suming all layers without airmass weighting => sensitivity.
  endfor

  ;----------------------
  ; Intermediate plotting
  ;----------------------
  if( keyword_set( plt ) ) then begin
    tek_color
    set_plot, 'x'
    device, decompose=0
    window, 2
    !p.multi=[0,2,2]
    plot,  ds[i].ak[0,*], ds[i].altitude, xrange=[-2,2]
    for ii = 0, nlayers-1 do oplot, ds[i].ak[ii,*], ds[i].altitude, color=ii+2
    oplot, ds[i].sens,    ds[i].altitude
    plot,  ds[i].aprvmr,  ds[i].altitude
    oplot, ds[i].retvmr,  ds[i].altitude, color=2
    plot,  ds[i].aktc,    ds[i].altitude
    ;device,/close
  endif

  ;----------------------
  ; Get a priori profiles
  ;----------------------
  dum = readprfs4( aprfs, ds[i].directory + 'aprfs.table' )

  ; A priori
  ;ds[i].aprh2ovmr = aprfs.vmr[*,0]                       ; H2O profile
  ;ds[i].aprh2otc  = total( aprfs.vmr[*,0] * rprfs.a )    ; Total column H2O
  ds[i].aprlaycol = ds[i].aprvmr * rprfs.a               ;

  ; Retrieved
  ds[i].retlaycol = ds[i].retvmr * rprfs.a               ;
  
  ;-------------------------------------------
  ; Read relavent Error data for datastructure
  ;-------------------------------------------
  ; Covariance error matricies
  if keyword_set(incldErr) then begin
    dum = readnxn4e( errmatrx, ds[i].directory + 'Stotal.vmr.output' )
    indSys = where(errmatrx.name EQ 'Systematic')
    indRan = where(errmatrx.name EQ 'Random')
    
    ds[i].rand_covar = errmatrx.mat[indRan,*,*]
    ds[i].sys_covar  = errmatrx.mat[indSys,*,*]
    
    ; Total Column errors
    dum = readerrsum(errsum, ds[i].directory + 'Errorsummary.output' )
    
    ds[i].tot_rand_err = errsum.tot_rand
    ds[i].tot_sys_err  = errsum.tot_sys
  endif
  

endfor ; next entry

print, 'Total number of retrievals before filtering and valid retrieval checking: ',nsize

;--------------------------------------------------
; Filter out invalid retrievals. Invalid retrievals
; occure where files (like summary) are missing
;--------------------------------------------------
ds = ds[where( ds.iterations NE -999, cnt, /null )]
ds.totobs = cnt
print, 'Number of invalid retrievels (i.e. missing files): ',(nsize - cnt)

;-------------------------------------------------------------------
; Filtering Data. Since IDL does not have set operations, we have to
; re-create the data structure after every filter
;-------------------------------------------------------------------
if( keyword_set(fltr) ) then begin

  ; Find retrievals where profile vmr has negative values (inds1 are good values)
  inds  = indgen(nsize)
  inds1 = where( ds.retvmr LT 0.0, count )
  if (count gt 0) then begin
    inds1 = inds1/nlayers
    ; Find unique set of indicies and filter
    unq_ind = inds1(uniq(inds1,sort(inds1)))
    for i = 0,n_elements(unq_ind)-1 do inds = inds(where(inds ne unq_ind[i],/null))
    ds      = ds[inds]
    ds.fltr_negvmr = n_elements(unq_ind)
    print, 'Number of retrievals filtered for negative vmrs: ',n_elements(unq_ind)
  endif

  ; Find retrievals where total column is 0 (inds2 are good values)
  dum = where( ds.rettc eq 0, count, complement=inds2)
  if (count gt 0) then begin
    ; Find unique set of indicies and filter
    unq_ind = inds2( uniq( inds2, sort(inds2) ) )
    ds      = ds[unq_ind]
    ds.fltr_tc0 = count
    print, 'Number of retrievals filtered for total column eq 0: ',count
  endif

  ; Find values below maxrms (inds3 are good values)
  if( keyword_set(maxrms) ) then begin
    dum = where( ds.rms gt maxrms, count, complement=inds3)
    if (count gt 0) then begin
      ; Find unique set of indicies and filter
      unq_ind = inds3( uniq( inds3, sort(inds3) ) )
      ds      = ds[unq_ind]
      ds.fltr_maxrms = count
      print, 'Number of retrievals filtered based on maxrms: ',count
    endif
  endif

endif

ds.nobs = n_elements(ds)
print, 'Total number of retrievals after filtering: ',n_elements(ds)


;---------------------------
; Save data to IDL save file
;---------------------------
print, 'ready to save'

;reorder first by datetime so that we ensure chronological order
ds = ds( sort( ds.datetime ) )

save, ds, filename=outfile
print,'Data structure for all retrievals saved as ', outfile

end


;___________________________________________________________________________
;                            SUBROUTINES
;___________________________________________________________________________

function bldstrct, npnts, nlayers
;--------------------------------------------
; Subroutine to initialize data structure for
; save file. Structure is initialized with 
; -90000 which is NDACC missing value
;--------------------------------------------

print, 'Building data structure : ', npnts, ' retrievals by : ', nlayers, ' layers.'

return, datastructure = REPLICATE({h224, $                                              ; 
        totobs                     :-90000,$                                            ; Total number of observations before filtering
        fltr_negvmr                :-90000,$                                            ; Number of retrievals filtered for negative vmrs
        fltr_tc0                   :-90000,$                                            ; Number of retrievals filtered for total column eq 0
        fltr_maxrms                :-90000,$                                            ; Number of retrievals filtered based on max rms
        nobs                       :-90000,$                                            ; Number of retrievals left after filtering
        spectraname                :'',$                                                ; OPUS file name
        directory                  :'',$                                                ; Output directory location
        hhmmss                     :'',$                                                ; Time stamp of observation HHMMSS     [String!!!]
        yyyymmdd                   :'',$                                                ; Date stamp of observation YYYYMMDD   [String!!!]
        snr                        :-90000.0,$                                          ; Signal to noise ratio as given by OPUS file
        p                          :make_array(nlayers,/FLOAT,value=-90000.),$          ; Pressure levels [mbar]
        t                          :make_array(nlayers,/FLOAT,value=-90000.),$          ; Temperature profile [K] 
        ms                         :make_array(nlayers,/FLOAT,value=-90000.),$          ;
        rms                        :-90000.0,$                                          ; Fit RMS
        dofs                       :-90000.0,$                                          ; Degrees of freedom
        sza                        :-90000.D0,$                                         ; Solar zenith angle
        azi                        :-90000.D0,$                                         ; Solar Azimuth angle
        iterations                 :-90000,$                                            ; Number of iterations for retrieval
        zcorrect                   :-90000.D0,$                                         ;
        wshift                     :-90000.D0,$                                         ;
        zerolev                    :-90000.D0,$                                         ;
        bslope                     :-90000.D0,$                                         ;
        phase                      :-90000.D0,$                                         ;
        aprvmr                     :make_array(nlayers,/FLOAT,value=-90000.),$          ;
        aprlaycol                  :make_array(nlayers,/FLOAT,value=-90000.),$          ;
        aprtc                      :-90000.D0,$                                         ;
        retvmr                     :make_array(nlayers,/FLOAT,value=-90000.),$          ;
        retlaycol                  :make_array(nlayers,/FLOAT,value=-90000.),$          ; 
        rettc                      :-90000.D0,$                                         ; Primary retrieved gas total column amount (Old name = rettc)
        h2o_vmr                    :make_array(nlayers,/FLOAT,value=-90000.),$          ; Retrieved H2O vmr profile (old name = aprh2ovmr)
        h2o_tc                     :-90000.D0,$                                         ; H2O retrieved total column amount (Old name = aprh2otc)
        year                       :-90000,$                                            ; Year [YYYY]
        month                      :-90000,$                                            ; Month [MM]
        day                        :-90000,$                                            ; Day [DD]
        hrs                        :-90000.D0,$                                         ; Hours plus fractional hours  hh.fffffff
        doy                        :-90000.D0,$                                         ; Day of the year
        tyr                        :-90000.D0,$                                         ; Years plus fractional year    yyyy.ffffff
        datetime                   :-90000.D0,$                                         ; mjd2k date
        jday                       :-90000.D0,$                                         ; julian date
        latitude                   :-90000.00,$                                         ; Latitude of instrument []
        longitude                  :-90000.00,$                                         ; Longitude of insturment []
        alt_instrument             :-90000.00,$                                         ; Altitude of insturment [m]
        surface_pressure           :-90000.D0,$                                         ; Surface Pressure [hPa]
        surface_temperature        :-90000.D0,$                                         ; Surface temperature [K]
        alt_index                  :make_array(nlayers,/FLOAT,value=-90000.),$          ;
        alt_boundaries             :make_array(2,nlayers,/FLOAT,value=-90000.),$        ;
        altitude                   :make_array(nlayers,/FLOAT,value=-90000.),$          ;
        ak                         :make_array(nlayers,nlayers,/FLOAT,value=-90000.),$  ;check directions and normalize....
        aktc                       :make_array(nlayers,/FLOAT,value=-90000.),$          ; x plot and check we have the direction right!
        sens                       :make_array(nlayers,/FLOAT,value=-90000.),$          ;
        external_solar_sensor      :-90000.D0,$                                         ; External solar sensor (for MLO either E_radiance or W_radiance) [volts]
        detector_intern_temp_switch:-90000.D0,$                                         ; Detector Internal Temperature Switch []
        guider_quad_sensor_sum     :-90000.D0,$                                         ; Quad solar sensor from house log data [volts]
        outside_relative_humidity  :-90000.D0,$                                         ; Outside relative humidity. From either house log or external station data [%]
        int_time                   :-90000.D0,$                                         ; Integration time
        tot_rand_err               :-90000.D0,$                                         ; Total random uncertainity not including smoothing error     [molecules cm^-2]
        tot_sys_err                :-90000.D0,$                                         ; Total systematic uncertainity not including smoothing error [molecules cm^-2]
        rand_covar                 :make_array(nlayers,nlayers,/FLOAT,value=-90000.),$  ; Total random error covariance matrix without smoothing [vmr]
        sys_covar                  :make_array(nlayers,nlayers,/FLOAT,value=-90000.) $  ; Total systematic error covariance matrix without smoothing [vmr]
         } ,npnts)

end




