      program advect
c       box model version of regional model, hank

      implicit none
      include "chem_params.inc"       !number of species, indices of sp.
      include 'cchem.com'             !character list of species
      include 'ichem.com'             !mapping indices of reactions
      include 'rchem.com'             !MW and stoich. coef
      include 'irate.com'             !type of reaction (hv, Arrhen, etc)
      include 'rrate.com'             !variables for calc. Arrhen and TROE rxns
      include 'amap.com'              !maps species to/from gas and aqueous arrays
      include 'hvchem.com'
      include 'lnox.com'              !parameters for injecting a NO source mimicking lightning


c ****  constants ***

      common /dltprms/dtchem, dt_dbg
      real dtchem                ! the chemistry timestep  
      real dt_dbg                ! how often to do a debug print

      common /constant/pi,r,gr,cp
      real pi    ! pi                                             = PI
      real r     ! gas constant for dry air                       = RD
      real gr    ! acceleration of gravity                        = GZERO
      real cp    ! specific heat of dry air at constant pressure  = CPD

      real na    ! Avogadro's number (molec/mol)
      real mwair ! MW of air (g/mol)

      real rate(nrmx)                    ! reaction rate constants of gas rxns
      real ratea(nrmx)                   ! reaction rate constants of cw rxns

      real qca, qva                  ! cloud water kg/kg or g/g  and water vapor g/g
      real lww                       ! cloud water cm3/cm3
      real ph                        ! pH of drops
      integer ncpts                  ! = 1 if cloud exists
      integer npts                   ! = 1 for clear sky
      common/cldpar/qca, lww, ph, ncpts, npts

      real press, rho, ttemp, rhoair, zalt

c ktc is the rate constant of diffusion from gas to liquid
c aheff is henry's law coefficients for aqueous species
c phrat is phase ratio = Ca/Cg
      real ktc(naq), aheff(naq) 
      real phrat(naq)
      real rhow                       ! water density

c cloud water and rain data
      integer nctim, incw, np
      parameter(np=500)
      real cwtim(np), tair_in(np), rho_in(np), pres_in(np), z(np), w(np)
      real qv_in(np), qc_in(np), qi_in(np), qr_in(np), qs_in(np), 
     _     qg_in(np)
c--mcb


c ****  meteorological variables ***

      real q                   ! specific humidity (kg/kg)
      real t                   ! temperature (K)

      real spec(nspec)  ! gas species concentration
     _   ,aspec(naq)    ! cloud drop species


c *** time:

       real starttime  ! start time of integration -- read in Chemset
       real endtime    ! end time of integration
       real tim        ! the model tim starting from 0 to timf (in seconds)
       real timf       ! the final time of the model from beginning (in seconds)
       real timt       ! time value used for photolysis rate calculations
       real startcld   ! time to start cloud

c indexes
      integer i,ii,j,k,l,ir,isp,i2,n,ip

c unit numbers for input/output:
      integer 
     & inrt,              ! inrt:  react unit number
     & inph               ! inph:  photolysis unit number

      real cinit(nspec), cinit1(nspec) 
      real dens, tsteps, writtim
      integer icount, nt 
      logical iprnt

c photolysis rate variables (+ common block included above)
      integer mr1, jtim, jtimp, it, irphot(30)
      real tday, tj, pslope, tdif, dtj, otim

      real delttj
      character*4 ci(0:nspec)
      character*7 aname
      character*20 finame

      external 
     & chemset, xinit, getph, rates, glrates, chem

c
c initialization of parameters
c
c
c     initialize the chemistry
c 
      do i=1,nspec
       gasmap(i) = 0
      end do
c++mcb  --  map aqueous species to gas species
      aqmap(0) = 0
      aqmap(ko3a) = ko3
      aqmap(kh2o2a) = kh2o2
      aqmap(kho2a) = kho2
      aqmap(kch2oa) = kch2o
      aqmap(kmpera) = kch3ooh
      aqmap(kfaa) = kfa
      aqmap(kch3ooa) = kch3oo
      aqmap(khno3a) = khno3
      aqmap(koha) = koh
      aqmap(knoa) = kno
      aqmap(kno2a) = kno2
      aqmap(kn2o5a) = kn2o5
      aqmap(kno3a) = kno3
      aqmap(kco2a) = kco2
      aqmap(kso2a) = kso2
      aqmap(kso4a) = kso4
      aqmap(kcl2m) = kcl2mg
      aqmap(kclm) = kclmg
      aqmap(kcla) = kclag
      aqmap(kco3m) = kco3mg
      aqmap(khno4a) = khno4
      aqmap(ko011a) = ko011
      aqmap(kdo23a) = kdo23
      aqmap(kdk33a) = kdk33
      aqmap(kh021a) = kh021
      aqmap(kho34a) = kho34
      aqmap(khtA1a) = khtA1
      aqmap(khk44a) = khk44
      aqmap(khk33a) = khk33
      aqmap(khu51a) = khu51
      aqmap(kxooha) = kxooh
      aqmap(kd021a) = kd021
      aqmap(kud42a) = kud42
      aqmap(kdd21a) = kdd21
      aqmap(kdd55a) = kdd55
      aqmap(kk031a) = kk031
      aqmap(khyaca) = khyac
      aqmap(kuk41a) = kuk41
      aqmap(kg021a) = kg021
      aqmap(kn002a) = kn002
      aqmap(kp021a) = kp021
      aqmap(kpu44a) = kpu44
      aqmap(k3021a) = k3021
      
      gasmap(ko3) = ko3a
      gasmap(kh2o2) = kh2o2a
      gasmap(kho2) = kho2a
      gasmap(kch2o) = kch2oa
      gasmap(kch3ooh) = kmpera
      gasmap(kfa) = kfaa
      gasmap(kch3oo) = kch3ooa
      gasmap(khno3) = khno3a
      gasmap(koh) = koha
      gasmap(kno) = knoa
      gasmap(kno2) = kno2a
      gasmap(kn2o5) = kn2o5a
      gasmap(kno3) = kno3a
      gasmap(kco2) = kco2a
      gasmap(kso2) = kso2a
      gasmap(kso4) = kso4a
      gasmap(kcl2mg) = kcl2m
      gasmap(kclmg) = kclm
      gasmap(kclag) = kcla
      gasmap(kco3mg) = kco3m
      gasmap(khno4) = khno4a
      gasmap(ko011) = ko011a
      gasmap(kdo23) = kdo23a
      gasmap(kdk33) = kdk33a
      gasmap(kh021) = kh021a
      gasmap(kho34) = kho34a
      gasmap(khtA1) = khtA1a
      gasmap(khk44) = khk44a
      gasmap(khk33) = khk33a
      gasmap(khu51) = khu51a
      gasmap(kxooh) = kxooha
      gasmap(kd021) = kd021a
      gasmap(kud42) = kud42a
      gasmap(kdd21) = kdd21a
      gasmap(kdd55) = kdd55a
      gasmap(kk031) = kk031a
      gasmap(khyac) = khyaca
      gasmap(kuk41) = kuk41a
      gasmap(kg021) = kg021a
      gasmap(kn002) = kn002a
      gasmap(kp021) = kp021a
      gasmap(kpu44) = kpu44a
      gasmap(k3021) = k3021a

c initialize the species mixing ratios to zero
      call xinit(spec,nspec,0.)
      call xinit(aspec,naq,0.)

c  define density of water (kg/m3)
      rhow = 1000.

      inrt = 15
      inph = 11
      incw = 14
      icount = 0

      call chemset(inrt, cinit, nspec, starttime, endtime, t, rhoair)
c-----------------------------------------------------------------------

      write(*,*)
      write(*,*) '-----------------------------------------------------'
      write(*,*) 'after chemset: nreact, naqr  ', nreact, naqr
      write(*,*) '               aqmap   gasmap '
      do isp=1,naq
       write(*,'(a,2i6)') cx(aqmap(isp)), aqmap(isp), gasmap(aqmap(isp))
      end do

      na = 6.0221e23
      mwair = 28.97
      pi=2.*asin(1.)
      gr=9.81
      r=287.
      cp=1004.
c
c initialize time parameters   
c
      dtchem = 300.
      dtchem = 10.
      dt_dbg = 60.

      tim = starttime
      timf = endtime

      writtim = dtchem
      tsteps = (timf-tim)/writtim
      nt = tsteps + 1

c Calculate air density, water vapor mixing ratio
      dens = rhoair * na/mwair *1.e-3           !air density [molecules/cm3]
      q = cinit(kh2o) *18./28.97                !water vapor mixing ratio (cinit is mol/mol; q is g/g)

      print*,'dens= ',dens, rhoair, 'H2O = ',cinit(kh2o), q

      print*, 'AIR DENSITY AND TEMPERATURE WILL CHANGE TO FOLLOW A ',
     _        'PARCEL IN CLOUD'
      print*, 'VALUES ARE READ IN FILE CWRN -- see below ' 

      do i=1,nspec
        spec(i) = cinit(i)
      enddo

c Open output file and print initial concentration in file
        open(unit=17,file='thermo.out',form='formatted')
        open(unit=18,file='gas.out',form='formatted')
        open(unit=19,file='cw.out',form='formatted')
        open(unit=20,file='total.out',form='formatted')

1020    FORMAT(100(E13.5,'  '))

c-----------------------------------------------------------------------
      write(17,'(a,4x,10(3x,a11,2x))') '    Time', '   z_(km)  ', 
     _ '   T_(K)   ','   P_(hPa)   ', 'rho_(kg/m3)', 'qv_(kg/kg) ', 
     _ 'qc_(kg/kg) '
      write(18,'(a,4x,100(3x,a10,2x))') '    Time', (cx(j), j=1,nspec)
      write(19,'(a,4x,100(3x,a10,2x))') '    Time', 
     _   (cx(aqmap(j)), j=1,naq)
      write(20,'(a,4x,100(3x,a10,2x))') '    Time', 
     _   (cx(aqmap(j)), j=1,naq)

       write(19,1020) tim, (aspec(k) ,k=1,naq)
       write(18,1020) tim, (spec(k)  ,k=1,nspec)
       write(20,1020) tim, (spec(aqmap(k)) + aspec(k) ,k=1,naq)

       write(91,'(f8.1,6e12.5)') tim, spec(koh), aspec(koha), 
     _   spec(kho2), aspec(kho2a), spec(kch3oo), aspec(kch3ooa)

       icount=icount+1
       write(*,*) ' icount after initialization ', icount



c++ read in jval information 

       finame='new.jv'
       open(file=finame,unit=11,form='formatted')
       inph=11

       read(inph,*)
       READ(INPH,*,ERR=2000,END=2010) MR1, NPHOT
       read(inph,*)
       READ(INPH,*,ERR=2000,END=2010) (irphot(i),I=1,MR1)
c         print*,'photolysis reaction numbers '
c         write(6,*) (irphot(i),I=1,MR1)
       read(inph,*)
       READ(INPH,*,ERR=2000,END=2010) (TJVAL(J), J=1, NPHOT)
       delttj = tjval(2)-tjval(1)
c         write(*,*) tjval(1), tjval(2), delttj

       DO 58 I = 1, MR1
          read(inph,*)
          READ(INPH,*,ERR=2000,END=2010) (XJVAL(J,I), J=1, NPHOT)
  58   CONTINUE
c         write(*,*) xjval(42,1), xjval(43,1)


c++mcb -- read in output from WRF-Chem output
c 

      startcld = 10.*60.       ! start cloud after 10 minutes of running gas chemistry

      finame='cwrn'        ! clear 
      open(file=finame,unit=incw)
      read(incw,*) 
!!      read(incw,*) nctim
      i = 0
 59   continue
      i = i+1

c  mcb -- new version that reads time, state parameters, condensate, water vapor (could do trace gases too)
      read(incw,*,err=2002,end=23) 
     _  z(i), w(i), pres_in(i), tair_in(i), qv_in(i), qc_in(i),       ! km, m/s, hPa, K, g/kg, g/kg
     _  qi_in(i), qr_in(i), qs_in(i), qg_in(i)                        ! g/kg, g/kg, g/kg, g/kg

!!     _          cwtim(i), cldw(i), tair_in(i), pres_in(i)     !g/kg , K, hPa
      z(i) = z(i) * 1000.                                     ! m
      cwtim(i) = (z(i) - z(1)) / 3.                           ! assume 3 m/s updraft, cwtim(seconds)

      rho_in(i) = pres_in(i)*100./(287.*tair_in(i))
      qv_in(i) = qv_in(i) *0.001                              ! g/g
      cwtim(i) = cwtim(i) + starttime + startcld

      go to 59

  23  continue
      nctim = i-1
      write(*,*) starttime + startcld
      do i = 1,nctim
        write(*,*) " cld_in  ", cwtim(i), z(i), qv_in(i), qc_in(i), 
     _  tair_in(i), pres_in(i), rho_in(i)
      end do
!        write(*,*) " lastcld ", cwtim(nctim), qv_in(nctim), 
!     _                        qc_in(nctim), tair_in(nctim), nctim
c--mcb

c Initial guess at pH
      ph = 5.0

c Reset pressure, temperature and air density
      press = pres_in(1)
      t     = tair_in(1)
      rhoair = rho_in(1)

c Calculate air density, water vapor mixing ratio
      dens = rhoair * na/mwair *1.e-3           !air density
      q = qv_in(1)    ! *dens * mwair/18.

      spec(kh2o) = q * mwair/18.
      print*,'dens= ',dens, rhoair, 'H2O = ',spec(1),' mol/mol ',
     _   spec(1)*dens, ' molec/cm3 ', q,' kg/kg ', t


c**********************************************************************c
c    Time loop:
c**********************************************************************c
 1000 continue

      write(*,*) 'Time: ', tim, rhoair, q, t, dens
c++mcb
c   set cloud and rain water mixing ratios

      timt = tim
      do i=1,nctim-1
       if(timt .ge. cwtim(i) .and. timt .lt. cwtim(i+1)) then
        qca = qc_in(i) + (timt-cwtim(i)) * (qc_in(i+1)-qc_in(i)) /
     _                          (cwtim(i+1)-cwtim(i))

        t   = tair_in(i) + (timt-cwtim(i)) * (tair_in(i+1)-tair_in(i)) /
     _                          (cwtim(i+1)-cwtim(i))

        rhoair= rho_in(i) + (timt-cwtim(i)) * (rho_in(i+1)-rho_in(i)) /
     _                          (cwtim(i+1)-cwtim(i))

        dens = rhoair * na/mwair *1.e-3           !air density, molecules/cm3
        press= rhoair*287.*t/100.                 !use ideal gas law instead of interpolation to be sure rho, T, p are consistent

        zalt = z(i) + (timt-cwtim(i)) * (z(i+1)-z(i)) /
     _                          (cwtim(i+1)-cwtim(i))

        qva = qv_in(i) + (timt-cwtim(i)) * (qv_in(i+1)-qv_in(i)) /
     _                          (cwtim(i+1)-cwtim(i))
!!        q = cinit(kh2o)/dens *18./28.97           !water vapor mixing ratio

        q = qva            !! /dens *18./28.97           !water vapor mixing ratio  g/g

        go to 24
       else
        qca = 0.
       endif
      end do

  24  continue
      spec(kh2o) = q * mwair/18.        ! mol H2O/mol air
      write(*,*) tim, zalt, qca, t, rhoair, q, ' kg/kg ',
     _           spec(kh2o), ' mol/mol   set qca, q H2O '
      write(17,1030) tim, zalt, t, press, rhoair, qva, qca
1030  FORMAT(5(F13.5,'  '),5(E13.5,'  '))


c-----------------------------------------------------------------------
c ADD LIGHTNING NO AT tair_end < T < tair_start

      if(t .le. (273.15+tair_start) .and. t .ge. (273.15+tair_end)) then
        spec(kno) = spec(kno) + lno_src*1.e-12
      endif
c-----------------------------------------------------------------------


c adjust spec for new air density and put into molecules/cm3 units
      do isp=1,nspec
         spec(isp) = spec(isp)*dens
      end do
      do isp=1,naq
         aspec(isp) = aspec(isp)*dens
      end do


c++mcb  --  Find the cloudy grid points

      ncpts = 0
      npts = 1

      if(qca .gt. 1.e-12) then
        lww = rhoair * qca/(rhow*1000.)
        ncpts = 1
        npts = 0
        
        write(*,'(2e12.5)') qca, q
        write(*,'(1p,3e12.5,2i5)') lww, rhoair, rhow, ncpts, npts
c=======================================================================
c for species that are considered only in the aqueous phase
        aspec(kcl2m) = aspec(kcl2m) + spec(kcl2mg)
        spec(kcl2mg) = 0.

        aspec(kclm) = aspec(kclm) + spec(kclmg)
        spec(kclmg) = 0.

        aspec(kcla) = aspec(kcla) + spec(kclag)
        spec(kclag) = 0.

        aspec(kco3m) = aspec(kco3m) + spec(kco3mg)
        spec(kco3mg) = 0.

      else
        ncpts = 0
        npts = 1
        do isp=1,naq
         spec(aqmap(isp)) = spec(aqmap(isp)) + aspec(isp) 
         aspec(isp) = 0.
        end do
      endif

c get the reaction rates
c-----------------------------------------------------------------------
      call getph(t, rhoair, qca, aspec, ph)
      call rates(t, rhoair, q, qca, ph, rate, ratea, tim)


      call glrates(t, rhoair, ph, ktc, lww, aheff, phrat)
c-----------------------------------------------------------------------

      write(*,'(a,43e12.5)') 'done with glrates ',(phrat(isp),isp=1,naq)
      write(*,*) lww
      write(*,'(a,43e12.5)') 'heff ',(aheff(isp),isp=1,naq)

c do the chemistry
c-----------------------------------------------------------------------
        call chem(rate,t,q,spec, ratea, ktc,aheff,phrat,aspec, tim)
c-----------------------------------------------------------------------

c adjust spec back to vmr
      do isp=1,nspec
         spec(isp) = spec(isp)/dens
      end do
      do isp=1,naq
         aspec(isp) = aspec(isp)/dens
      end do

c increment the time
      tim=tim+dtchem
      print*,'elapsed time: ',tim

c
c write to the output file
      otim=float(icount)*writtim
      iprnt=.false.
      print*,'elapsed time: ',tim, otim, icount+1, nt
      if(tim.ge.otim.and.(icount+1).le.nt) then
        iprnt=.true.
        icount=icount+1
        write(18,1020) tim,((spec(k)),k=1,nspec)
        write(19,1020) tim, (aspec(k) ,k=1,naq)
        write(20,1020) tim, ((spec(aqmap(k)) + aspec(k)) ,k=1,naq)

        write(91,'(f8.1,6e12.5)') tim, spec(koh), aspec(koha), 
     _   spec(kho2), aspec(kho2a), spec(kch3oo), aspec(kch3ooa)
      endif


      if(tim.ge.timf) goto 1001

      goto 1000

 1001 continue
      write(17,1030) tim, zalt, t, press, rhoair, qva, qca
      print*,'times written ',nt,icount

      stop

2000  continue
      write(6,1140) finame
      stop

2010  continue
      write(6,1150) finame
      stop

2002  continue
      write(6,1140) finame
      stop

1140  FORMAT ('Error in Reading Input File:',A20)
1150  FORMAT ('Unexpected End-Of-File in Input File:',A20)

      end
c======================================================================c
      subroutine chemset(inrt, cinit, mspec, starttime, endtime, tair,
     _  rhoair)
c
      implicit none
      include 'chem_params.inc'
c   
c this routine sets the reactions and writes a map of the 
c reactions into icmap. icmap is labeled by the species 1....n. for each species
c the ordering is as follows:
c
c # of sources 
c # of sinks
c             then for each source reaction the following five numbers are read
c             until all the source reactions are accounted for 
c reaction number
c label of stoich coefficient in stc
c 1 reactant
c 2 reactant
c 3 reactant
c             then for each sink reaction the following five numbers are read
c             until all the sink reactions are accounted for 
c reaction number
c label of stoich coefficient in stc
c 1 reactant
c 2 reactant
c 3 reactant
c 4 reactant
c
c  variables:
c
c
c         common block storage
c
c  cw
c  cx                         character list of species
c  stc                        matrix of standard stoichometry coefficients
c  icmap                      reaction matrix for each species
c  nreact                     total number of reactions
c  a                          temperature independent coeff for arrehenius reactions
c  s                          temperature   dependent coeff for arrehenius reactions
c  stc                        stoichmetry coefficients
c  nspec                     number of chemically interactive species
c
c
c         temporary storage
c
c  isrc                       temporary storage of map of sinks
c  isnk                       temporary storage of map of sources
c  iblank                     blank character
c  c2                         index for special reaction
c  molec                      reaction molecules
c  sc                         stoichemetry coefficients
c  mr1                        number of photolysis reactions
c  mr2                        number of arrhenius reactions
c  mr3                        number of m reactions
c  mr4                        number of troe reactions
c  mr5                        number of special reactions
c  iwhich                     index indentifier for special reactions
c  inrt                       input file numbers
c  finame                     input file name
c    itype1 - photolysis - already assigned at read
c    itype2 - simple arrhenius
c    itype3 - a + m or a + b + m, possibly also arrhenius
c    itype4 - troe  -  already assigned at read
c    itype5 - special functions for rate constants 
c***  troe reaction data, molecule, cm3, sec units
c  ak0300 - zero pressure 300k rate constant, third order
c  an     - temperature exponent for zero pressure rate constant
c  aki300 - high pressure 300k rate constant, second order
c  am     - temperature exponent for high pressure rate constant
c  bas    - base of exponentiation, 0.6 for most
c  aequil - pre-exponential of equilibrium constant
c  tequil - activation temperature of equilibrium constant
c**

c input/output
      integer inrt, mspec
      real cinit(mspec)
      real starttime, endtime
      real tair, rhoair

c temporary storage

      integer mr1,mr2,mr3,mr4,mr5,mr6,mr7

      integer ntx,ncx,nx

c      integer 
c     & isrc(5*nrps),
c     & isnk(5*nrps)

      character*20    finame
      character*2 c2(nrmx)
      character*4 molec(nrmx,7)
      character*4 iblank

      real sc(nrmx,7)

c     indexes
      integer ix,i,ir,j,k,ik,n,jr,it,iy

      integer nsnks,nsrcs, nsnksa,nsrcsa
      integer ip,ir1,is1,ip1,ip2,ip3,ip4

      real csnk

c chemical storage block

      include 'cchem.com'
      include 'ichem.com'
      include 'rchem.com'
      include 'irate.com'
      include 'amap.com'
      include 'rrate.com'
      include 'hvchem.com'
      include 'lnox.com'


      external opndat, maprxns

      DATA IBLANK     /'   '/

      do i=0,nspec
       cx(i)=iblank
      enddo
      do i=0,naq
       cxa(i)=iblank
      enddo

c weight the molecular weights 
c molecules/mole
c cw=grams/mole
c 1.e3 is conversion to molecules/kg

      FINAME = 'react'
      CALL OPNDAT( INRT, FINAME )

      read(inrt,*) nreact,mr1,mr2,mr3,mr4,mr5,mr6,mr7
      print*,'Number of reactions ',nreact
      print*,'Number of photolysis reactions ',mr1
      print*,'Number of arrhenius reactions  ',mr2
      print*,'Number of m reactions          ',mr3
      print*,'Number of troe reactions       ',mr4
      print*,'Number of special reactions    ',mr5
      print*,'Number of aqueous reactions    ',mr6
      print*,'Number of aqueous photo reactions ',mr7

c read in initial time of simulation, final time, 
c  temperature (K), and air density (kg/m3)

      read(inrt,*) starttime, endtime, tair, rhoair
      print*,'Start/Stop times ', starttime, endtime
      print*,'Temperature, Air density ', tair, rhoair

c read in temperature range for sourcing lightning NO
c  tair_start, tair_end (deg C), lno_src (pptv)     see equation in main

      read(inrt,*) tair_start, tair_end, lno_src   
      print*,'Lightning NO ', tair_start, tair_end, lno_src

c read in species names, MWs and initial concentration
 
      READ(inrt,*) nx    ! total number of chemical species
      print*,'Number of total species ',nx

      if(nx.ne.nspec) then
       print*,'MISMATCH BETWEEN PARAMETER AND REACT FILES'
       print*,'REACT, TOTAL SPECIES ',nx
       print*,'PARAM, TOTAL SPECIES ',nspec
       stop
      endif

      print*,'         '
      print*,'SPECIES '

      do i=1,nspec
       read(inrt,*) cx(i),cw(i), cinit(i)            ! mol X / mol air for all species
       write(6,602) i,cx(i),cw(i), cinit(i)
      enddo

 602  format(i3,1x,a4,1x,f6.2,3x,1p,e13.5)

c stoich coefficients: set the standard ones

      call xinit(stc,(2*nspec),0.)
      call xinit(sc,(nrmx*7),0.)

      stc(0)=99.               
      stc(1)=1.               
      stc(2)=2.

c read in reaction file
        print*,'         '
        do 15 j = 1, nreact
           read(inrt,1090,err=2000,end=2010) c2(j),
     |     (molec(j,i),i=1,3),(sc(j,i),molec(j,i),i=4,7),a(j),s(j)
           write(6,1091) j,c2(j),
     |     (molec(j,i),i=1,3),(sc(j,i),molec(j,i),i=4,7),a(j),s(j)
  15    continue

      write(*,*) 
      write(*,*) '  rxns read in ' 
	do 17 i = 1, mr4
         read(inrt,1080,err=2000,end=2010) itype4(i),
     |	  ak0300(i),an(i),aki300(i),am(i),bas(i),
     |	  aequil(i),tequil(i)
17	continue

       close(inrt)        

      write(*,*) 
      write(*,*) '  cw and rain names ' 
      do i=1,naq 
       cxa(i) = cx(aqmap(i)) // 'a'
       do j=1,5
        if(cxa(i)(j:j) .eq. ' ') then
          cxa(i)(j:j) = '_'
        endif
       end do
       write(*,'(a,3x,a)') cxa(i)
      end do

C  check reactions and verify assignments of types

c    itype1(i) - photolysis - already assigned at read
c    itype2(i) - simple arrhenius
c    itype3(i) - a + m or a + b + m, possibly also arrhenius
c    itype4(i) - troe  -  already assigned at read
c    itype5(i) - special functions for rate constants 
c    itype6(i) - aqueous reactions need rate constants modified
c    itype7(i) - aqueous photolysis reactions 

	nr1 = 0
        nr2 = 0
        nr3 = 0
        nr4 = 0
        nr5 = 0
        nr6 = 0
        nr7 = 0

        do 30 j = 1, nreact

C---------------------------------------------------------------------------
C  	   Count photolysis reactions:

	   if (molec(j,2) .eq. 'HV  ' .and. c2(j)(1:1) .ne. 'a') then
             nr1 = nr1 + 1
             itype1(nr1)=j
           endif
C---------------------------------------------------------------------------
C          Identify and count simple Arrhenius type:

           if (s(j) .ne. 0.) then
              nr2 = nr2 + 1
              itype2(nr2) = j
           endif
        
           do 22 k = 1, 3

c---------------------------------------------------------------------------
c          identify and count reactions with m as a reagent (not troe):

              if (molec(j,k) .eq. 'M   ') then
                 nr3 = nr3 + 1
                 itype3(nr3) = j
              endif
 
 22        continue

c---------------------------------------------------------------------------
c          count troe reactions
              
           if ((molec(j,2).eq.'(M) ').or.(molec(j,3).eq.'(M) ')) then
              nr4 = nr4 + 1
           endif

c---------------------------------------------------------------------------
c  	   identify and count reactions requiring special treatment:
c	   these have a lower case 's' in column 2

	   if (c2(j)(2:2) .eq. 's') then
	      nr5 = nr5 + 1
	      itype5(nr5) = j
	   endif

c---------------------------------------------------------------------------
c  	   identify and count aqueous reactions 
c	   these have a lower case 'a' in column 1

           if (c2(j)(1:1) .eq. 'a') then
             nr6 = nr6 + 1
             itype6(nr6) = j
c              write(8,'(30x,a,2i4)') 'aqueous: ', j, nr6
           endif
           naqr = nr6
C---------------------------------------------------------------------------
C  	   Count aqueous photolysis reactions:

	   if (molec(j,2) .eq. 'HV  ' .and. c2(j)(1:1) .eq. 'a') then
             nr7 = nr7 + 1
             itype7(nr7)=j
           endif

C---------------------------------------------------------------------------
C     Convert blank stoichiometry coefficients to unity (except if species is
C     also blank.  Note: specific products cannot be "turned off" by
C     simply giving them zero stoichiometry coefficients.

           do 32 k = 1, 7
              if (molec(j,k) .eq. 'HV  ') molec(j,k) = iblank
              if (molec(j,k) .eq. 'M   ') molec(j,k) = iblank
c             if (molec(j,k) .eq. 'N2  ') molec(j,k) = iblank
c             if (molec(j,k) .eq. 'O2  ') molec(j,k) = iblank
              if (molec(j,k) .eq. '(M) ') molec(j,k) = iblank
              if (sc(j,k) .eq. 0.) sc(j,k)=1.
              if (molec(j,k) .eq. iblank) sc(j,k) = 0.
32         continue

30    continue

c   map aqueous photolysis reactions to gas analogies
      do j=1,nr7
       jr = itype7(j)
       do it=1,nr1
         ir = itype1(it)
         if(molec(ir,1) .eq. molec(jr,1)) then
           if(molec(ir,1) .eq. 'O3')  then
            if(molec(ir,4) .eq. 'O1D') then
               hvmap(j) = ir
            endif
           else
            hvmap(j) = ir
           endif
         endif
       end do
       write(*,*) 'hvmap ', j, hvmap(j)
      end do

       if (nr1 .ne. mr1) write(6,*) nr1,mr1,
     |	'warning:  incorrect number of photolysis reactions'

       if (nr2 .ne. mr2) write(6,*) nr2, mr2,
     |	'warning:  incorrect number of arrhenius reactions'

       if (nr3 .ne. mr3) write(6,*) nr3, mr3,
     |	'warning:  incorrect number of m reactions'

       if (nr4 .ne. mr4) write(6,*) nr4, mr4,
     |	'warning:  incorrect number of troe reactions'

       if (nr5 .ne. mr5) write(6,*) nr5, mr5,
     |	'warning:  incorrect number of special reactions'

       if (nr6 .ne. mr6) write(6,*) nr6, mr6,
     |	'warning:  incorrect number of aqueous reactions'
       if (nr7 .ne. mr7) write(6,*) nr7, mr7,
     |	'warning:  incorrect number of aqueous photo reactions'


      print*,'                        '
      print*,'REACTION MECHANISM CHECKED CORRECTLY '
      print*,'                        '

C---------------------------------------------------------------------------
C   find correct index for special reactions:
C   1  -  HO2 + HO2 -> H2O2 + O2
C   2  -  HNO3 + HO  ->  H2O + NO3
C   3  -  CO + HO -> CO2 + H

   	do 50 i = 1, nr5

	   iwhich(i) = 0

	   if((molec(itype5(i),1) .eq. 'HO2 ') .and.
     |        (molec(itype5(i),2) .eq. 'HO2 '))  iwhich(i) = 1

	   if((molec(itype5(i),1) .eq. 'HNO3') .and.
     |        (molec(itype5(i),2) .eq. 'HO  '))  iwhich(i) = 2

	   if((molec(itype5(i),1) .eq. 'CO  ') .and.
     |        (molec(itype5(i),2) .eq. 'HO  '))  iwhich(i) = 3

	   if((molec(itype5(i),1) .eq. 'O3  ') .and.
     |        (molec(itype5(i),2) .eq. '    '))  iwhich(i) = 4

	   if((molec(itype5(i),1) .eq. 'k031') .and.
     |        (molec(itype5(i),2) .eq. 'HO  '))  iwhich(i) = 18

	   if((molec(itype5(i),1) .eq. 'XOOH') .and.
     |        (molec(itype5(i),2) .eq. 'HO  '))  iwhich(i) = 19

c   account for special reactions that take place in the aqueous phase
          if(c2(itype5(i))(1:1) .eq. 'a') then
           if((molec(itype5(i),1) .eq. 'HO2 ') .and.
     |        (molec(itype5(i),2) .eq. 'HO2 '))  iwhich(i) = 6

           if((molec(itype5(i),1) .eq. 'HO  ') .and.
     |        (molec(itype5(i),2) .eq. 'HO2 '))  iwhich(i) = 7

           if((molec(itype5(i),1) .eq. 'O3  ') .and.
     |        (molec(itype5(i),2) .eq. 'HO2 '))  iwhich(i) = 8

           if((molec(itype5(i),1) .eq. '2011') .and.
     |        (molec(itype5(i),2) .eq. 'HO2 '))  iwhich(i) = 9

           if((molec(itype5(i),1) .eq. 'a011') .and.
     |        (molec(itype5(i),2) .eq. 'HO  '))  iwhich(i) = 10

           if((molec(itype5(i),1) .eq. 'CO2 ') .and.
     |        (molec(itype5(i),2) .eq. 'HO  '))  iwhich(i) = 11

           if((molec(itype5(i),1) .eq. 'CO2 ') .and.
     |        (molec(itype5(i),2) .eq. 'HO2 '))  iwhich(i) = 12

           if((molec(itype5(i),1) .eq. 'CO3M') .and.
     |        (molec(itype5(i),2) .eq. 'HO2 '))  iwhich(i) = 13

           if((molec(itype5(i),1) .eq. 'HO2 ') .and.
     |        (molec(itype5(i),2) .eq. 'CL2M'))  iwhich(i) = 14

           if((molec(itype5(i),1) .eq. 'NO3 ') .and.
     |        (molec(itype5(i),2) .eq. 'HO2 '))  iwhich(i) = 15

           if((molec(itype5(i),1) .eq. 'SO2 ') .and.
     |        (molec(itype5(i),2) .eq. 'H2O2'))  iwhich(i) = 16

           if((molec(itype5(i),1) .eq. 'SO2 ') .and.
     |        (molec(itype5(i),2) .eq. 'O3  '))  iwhich(i) = 17

          endif      !aq.phase special reactions

	   if(iwhich(i) .eq. 0) then
                 write(6,*)
     |	     'warning: could not identify special reaction'
           endif
           write(*,*) 'specrxn ', itype5(i), iwhich(i), i
50	continue

c determine the reaction matrix icmap
      call maprxns(molec, sc, iblank, itype6, nspec)

c find rates for all reactions 

      do 3000 ix=1,nspec
      print*,'                        '
      print*,' reactions for species: ',cx(ix)

      nsnks=icmap(ix,1)
      nsrcs=icmap(ix,2)

      print*, nsnks, ' SINKS: '
      do 3001 ir=1,nsnks
      ip=(ir-1)*5

      ir1=icmap(ix,2+ip+1)
      is1=icmap(ix,2+ip+2)
      ip1=icmap(ix,2+ip+3)
      ip2=icmap(ix,2+ip+4)
      ip3=icmap(ix,2+ip+5)

      write(6,601) ir,ir1,stc(is1),cx(ip1),cx(ip2),cx(ip3),a(ir1),s(ir1)

 3001 continue
      iy = gasmap(ix)
      nsnksa=icmapa(iy,1)

      if(iy .ge. 1 .and. iy .le. naq) then
       print*,'      '
       print*, nsnksa, ' AQUEOUS SINKS: '
       do ir=1,nsnksa
         ip=(ir-1)*5

         ir1=icmapa(iy,2+ip+1)
         is1=icmapa(iy,2+ip+2)
         ip1=icmapa(iy,2+ip+3)
         ip2=icmapa(iy,2+ip+4)
         ip3=icmapa(iy,2+ip+5)


       write(6,601) ir,ir1,stc(is1),cx(aqmap(ip1)),cx(aqmap(ip2)),
     _   cx(aqmap(ip3)),a(ir1),s(ir1)

       end do
      endif

      nsrcs=icmap(ix,2)
      print*,'      '
      print*,nsrcs, ' SOURCES: '
      do 3002 ir=1,nsrcs
      ip=(ir-1)*5

      ir1=icmap(ix,2+5*nsnks+ip+1)
      is1=icmap(ix,2+5*nsnks+ip+2)
      ip1=icmap(ix,2+5*nsnks+ip+3)
      ip2=icmap(ix,2+5*nsnks+ip+4)
      ip3=icmap(ix,2+5*nsnks+ip+5)

      write(6,601) ir,ir1,stc(is1),cx(ip1),cx(ip2),cx(ip3),
     & a(ir1),s(ir1)

 3002 continue

      iy = gasmap(ix)
      nsrcsa=icmapa(iy,2)
      if(iy .ge. 1 .and. iy .le. naq) then
       print*,'      '
       print*,nsrcsa, ' AQUEOUS SOURCES: '
       do ir=1,nsrcsa
         ip=(ir-1)*5

         ir1=icmapa(iy,2+5*nsnksa+ip+1)
         is1=icmapa(iy,2+5*nsnksa+ip+2)
         ip1=icmapa(iy,2+5*nsnksa+ip+3)
         ip2=icmapa(iy,2+5*nsnksa+ip+4)
         ip3=icmapa(iy,2+5*nsnksa+ip+5)

      write(6,601) ir,ir1,stc(is1), cx(aqmap(ip1)), cx(aqmap(ip2)),
     _  cx(aqmap(ip3)),a(ir1),s(ir1)

       end do
      endif
 3000 continue
      return

2000  continue
      write(6,1140) finame
      stop

2010  continue
      write(6,1150) finame
      stop

 601  format( i3,1x,i3,1x,f4.1,1x,a4,1x,a4,1x,a4,1x,e10.3,1x,e10.3)
1040  FORMAT (12I10)
1050  FORMAT (7E10.3)
1080  FORMAT (I5,1X,2(E10.3,1X,F6.2,1X),F5.2,2(1X,E10.3))
1090  FORMAT (A2,3(A4,1X),2X,
     _      3(F5.2,1X,A4,1X),F5.2,1X,A4,1X,
     _        E8.2,1X,F7.0)
1091  FORMAT(I3,1x,A2,3(A4,1X),3(F5.2,1X,A4,1X),F5.2,1X,A4,E8.2,1X,F7.0)
1140  FORMAT ('Error in Reading Input File:',A20)
1150  FORMAT ('Unexpected End-Of-File in Input File:',A20)
      end

c-----------------------------------------------------------------------
      subroutine maprxns(molec, sc, iblank, iaq, nsp)
c determine the reaction matrix icmap
      implicit none
      include 'chem_params.inc'

c Input arguments
      integer nsp
      integer iaq(nrmx)           ! index for aqueous reactions
      character*4 molec(nrmx,7)    ! reaction molecules
      character*4 iblank           ! blank character
      real sc(nrmx,7)              ! stoichiometric coefficients

c common blocks
      include 'cchem.com'
      include 'rchem.com'
      include 'ichem.com'
      include 'amap.com'

c indices
      integer ix, ir, j, is, jp, jspec, ip, irct, i, na, n, iy
      integer nsrcs, nsnks
      integer nsrcsa, nsnksa
      integer nrct
      integer isrc(5*nrps+2) ,        ! temporary storage of map of sinks
     &        isnk(5*nrps+2)          ! temporary storage of map of sources
      integer isrca(5*nrps+2),       ! temporary storage of map of sinks
     &        isnka(5*nrps+2)        ! temporary storage of map of sources
      real csnk

 
c----------------------------------------------------------------------c
c initialize
      do ir=1,5*nrps+2
       do ix=1,nsp
        icmap(ix,ir) = 0
       end do
       isnk(ir)  = 0
       isrc(ir)  = 0
      end do
      do ir=1,5*nrps+2
       do ix=1,naq
        icmapa(ix,ir) = 0
       end do
       isnka(ir) = 0
       isrca(ir) = 0
      end do


c loop through all the species
      do 1000 ix=1,nsp 

      call iinit(isnk, (5*nrps+2),0)
      call iinit(isrc, (5*nrps+2),0)
      call iinit(isnka,(5*nrps+2),0)
      call iinit(isrca,(5*nrps+2),0)

c initialize number of sources and sinks for each species to 0

      nsnks  = 0
      nsrcs  = 0
      nsnksa = 0
      nsrcsa = 0

c loop through all the rxns and look for sinks and sources for species ix
c  first sinks, then sources

      do 900 ir=1,nreact       

      csnk=0                     ! stoich coefficient for sinks

c if species appears as reactant sink is found for species ix for reaction ir

       do 110 j=1,3    
    
          if(molec(ir,j).eq.cx(ix)) then
             csnk=csnk+sc(ir,j)
          endif

 110         continue

       if(csnk.ne.0) then        ! sink reaction is found

c set index of sink reaction for species ix

          do na=1,naqr
           if(ir .eq. iaq(na)) then
            nsnksa=nsnksa+1          ! total number of sink reactions
            isnka((nsnksa-1)*5+1)=ir 
c set index of stoich coefficient for species ix for reaction ir: 
c  csnk is the stoich coeff.

            do is=1,2*nspec+1
             if(abs((stc(is)-csnk)).lt..01) then
                isnka((nsnksa-1)*5+2)=is
                goto 113
             elseif(stc(is).eq.0) then
                stc(is)=csnk
                isnka((nsnksa-1)*5+2)=is
                goto 113
             elseif(is.gt.2*nspec) then
                print*,'not enough space for stoich. coefficients'
                print*,'increase dimensions of rstoich'
                stop
             endif
            end do
 113        continue
c fill in the index of the reactants
            nrct=0
            do j=1,3
             if(molec(ir,j).eq.iblank) goto 114
             do irct=1,nspec
               if(molec(ir,j).eq.cx(irct)) then
                if((cx(irct) .eq. cx(ix)) .and.
     &             (isnka(((nsnksa-1)*5)+2+3) .eq. 0)) then
                 isnka(((nsnksa-1)*5)+2+3)=gasmap(irct)
                 goto 114
                else
                 nrct=nrct+1
                 isnka(((nsnksa-1)*5)+2+nrct)=gasmap(irct)
                 goto 114         
                endif
               endif
             end do
 114         continue
            end do

            go to 145       !skip gas reactions
           endif
          end do

c Sink of gas reaction
          nsnks=nsnks+1          ! total number of sink reactions
          isnk((nsnks-1)*5+1)=ir 

c set index of stoich coefficient for species ix for reaction ir: 
c  csnk is the stoich coeff.

          do 120 is=1,2*nspec+1

             if(abs((stc(is)-csnk)).lt..01) then
                isnk((nsnks-1)*5+2)=is
                goto 121
             elseif(stc(is).eq.0) then
                stc(is)=csnk
                isnk((nsnks-1)*5+2)=is
                goto 121
             elseif(is.gt.2*nspec) then
                print*,'not enough space for stoich. coefficients'
                print*,'increase dimensions of rstoich'
                stop
             endif

 120  continue
 121  continue


c fill in the index of the reactants

         nrct=0

         do 140 j=1,3

         if(molec(ir,j).eq.iblank) goto 140

         do 130 irct=1,2*nspec

            if(molec(ir,j).eq.cx(irct)) then
             if((cx(irct) .eq. cx(ix)) .and.
     &          (isnk(((nsnks-1)*5)+2+3) .eq. 0)) then
              isnk(((nsnks-1)*5)+2+3)=irct
              goto 140
             else
              nrct=nrct+1
              isnk(((nsnks-1)*5)+2+nrct)=irct
              goto 140
             endif
            endif
                  
 130  continue
 140  continue

      endif                              ! end of sink indexing for reaction ir

 145  continue
c
c if species appears as product source is found for species ix for reaction ir

      do 150 jp=4,7

      if(molec(ir,jp).eq.cx(ix)) then         ! source reaction is found

         jspec=jp

         do na=1,naqr
          if(ir .eq. iaq(na)) then
           nsrcsa=nsrcsa+1          ! total number of sink reactions
           isrca((nsrcsa-1)*5+1)=ir 
c set index of stoich coefficient for species ix for reaction ir: 
c  stoich(ir,jspec)

           do is=1,2*nspec+1
            if(abs((stc(is)-sc(ir,jspec))).lt..01) then
               isrca((nsrcsa-1)*5+2)=is
               goto 157
            elseif(stc(is).eq.0) then
               stc(is)=sc(ir,jspec)
               isrca((nsrcsa-1)*5+2)=is
               goto 157
            elseif(is.gt.2*nspec) then
               print*,'not enough space for stoich. coefficients'
               print*,'increase dimensions of rstoich'
               stop
            endif
           end do
 157       continue

c fill in the index of the reactants
           nrct=0
           do j=1,3
            if(molec(ir,j).eq.iblank) goto 158
            do irct=1,nspec
              if(molec(ir,j).eq.cx(irct)) then
                 nrct=nrct+1
                 isrca(((nsrcsa-1)*5)+2+nrct)=gasmap(irct)
                 goto 158
              endif
            end do               
 158        continue
           end do               
           go to 151          !source reaction found; get out of loop
          end if
         end do



c set index of source reaction for species ix

         nsrcs=nsrcs+1                    ! number of source reactions
         isrc(((nsrcs-1)*5)+1)=ir

c set index of stoich coefficient for species ix for reaction ir: 
c  stoich(ir,jspec)

         do 160 is=1,2*nspec+1

            if(abs((stc(is)-sc(ir,jspec))).lt..01) then
               isrc((nsrcs-1)*5+2)=is
               goto 161
            elseif(stc(is).eq.0) then
               stc(is)=sc(ir,jspec)
               isrc((nsrcs-1)*5+2)=is
               goto 161
            elseif(is.gt.2*nspec) then
               print*,'not enough space for stoich. coefficients'
               print*,'increase dimensions of rstoich'
               stop
            endif
 160     continue
 161     continue

c fill in the index of the reactants

         nrct=0

         do 180 j=1,3

         if(molec(ir,j).eq.iblank) goto 180

         do 170 irct=1,nspec

            if(molec(ir,j).eq.cx(irct)) then
               nrct=nrct+1
               isrc(((nsrcs-1)*5)+2+nrct)=irct
               goto 180
            endif

 170     continue               
 180     continue

      goto 151   ! source reaction found: skip the loop

      endif      ! end of source indexing for reaction ir

 150  continue
 151  continue

 900  continue   

c  looped through all reactions: 
c   total number of sources and sinks known for species ix: 
c    now set icmap for species ix.

c gas reactions:
      icmap(ix,1)=nsnks
      icmap(ix,2)=nsrcs

      ip=2
      do 400 i=1,nsnks*5
      ip=2+i
      icmap(ix,ip)=isnk(i)
 400  continue

      do 500 i=1,nsrcs*5
      icmap(ix,ip+i)=isrc(i)
 500  continue
       is = jp

c aqueous reactions:

      iy = gasmap(ix)
      if(iy .ge. 1 .and. iy .le. naq) then
       icmapa(iy,1)=nsnksa
       icmapa(iy,2)=nsrcsa

       ip=2
       do i=1,nsnksa*5
        ip=2+i
        icmapa(iy,ip)=isnka(i)
       end do

       do i=1,nsrcsa*5
        icmapa(iy,ip+i)=isrca(i)
       end do

      endif               ! 1 <= iy <= naq

 1000  continue                 ! icmap set for all species

       return
       end
c======================================================================c
      subroutine getph(t, rho, qc, aspec, ph)
c this subroutine computes the pH based on trial and error
      implicit none
      include 'chem_params.inc'
c
c Input parameters
      real t                ! temperature (K)
      real rho              ! air density (kg/m3)
      real qc               ! CW mixing ratio (g/kg)
      real aspec(naq)       ! (molec/cm3)
c Output parameter
      real ph

c Local variables:
      real lwc              ! liquid water content (cm3 H2O / cm3 air)
      real na               ! Avogadro's number
      real k1fa             ! dissociation constant of HCOOH
      real k1s              ! dissociation constant of H2SO3 <--> HSO3-
      real k2s              ! dissociation constant of HSO3  <--> SO3=
      real k1c              ! dissociation constant of H2CO3 <--> HCO3-
      real k2c              ! dissociation constant of HCO3  <--> CO3=
      real fact             ! temperature factor for dissoc. constants

      real hion             ! [H+] in mol/L or M
      real ahion            ! [H+] in mol/L or M  (temporary)
      real f_o2m            ! fraction of HO2(a) + O2- that is O2-
      real f_fo             ! fraction of HCOOH(a) + HCOO- that is HCOO-
      real f_hso3           ! fraction of S(IV) that is HSO3-
      real f_hco3           ! fraction of H2CO3 + HCO3- + CO3= that is HCO3-

      real so4a             ! aqueous conc of SO4 (M)
      real no3a             ! aqueous conc of HNO3 (M)
      real so2a             ! aqueous conc of S(IV) (M)
      real co2a             ! aqueous conc of H2CO3 + HCO3- + CO3= (M)
      real ho2a             ! aqueous conc of HO2 + O2- (M)
      real faa              ! aqueous conc of HCOOH + HCOO- (M)


c Electroneutrality equation:
c  Na+ + NH4+ + H+  =  2SO4= + NO3- + HSO3- + HCO3- + Cl- + O2- + HCOO- 

c assume NH4+ = SO4=  (i.e. sulfate aerosol is NH4HSO4)
c assume Na+ = Cl-    (i.e. NaCl aerosol)
c assume HSO3- dominates S(IV) (which it does for pH of most cloudwater)
c to get:

c H+ = SO4= + NO3- + HSO3- + HCO3- + O2- + HCOO-

c assume aspec(kso4)  = SO4= entirely (all SO4 is in drops)
c        aspec(khno3a) = NO3- (all HNO3 in drops is NO3- and not HNO3(a))
c other species gotten by dissociation equilibrium:
c aspec(kso2) = concentration of S(IV) family, so HSO3- = f_hso3 * aspec(kso2)
c  where f_hso3 = HSO3/S(IV)  =  K1*H+ / (H+*H+ + K1*H+ + K1*K2)
c aspec(kfa) = concen of HCOOH(a) + HCOO-, so HCOO- = f_fo * aspec(kfa)
c  where f_fo = K1/(H+ + K1)

c because these fractions depend on H+, the calculation will be iterated
c  to determine H+ and pH

      if(qc .lt. 1.e-12) return

      na = 6.0221e23

c Convert concentrations to mol/L
      lwc = rho*qc * 1.e-6               !1.e-6 converts g H2O to m3 H2O
      so4a = aspec(kso4a) *1000./ (na * lwc)
      no3a = aspec(khno3a)*1000./ (na * lwc)
      so2a = aspec(kso2a) *1000./ (na * lwc)
      co2a = aspec(kco2a) *1000./ (na * lwc)
      ho2a = aspec(kho2a) *1000./ (na * lwc)
      faa  = aspec(kfaa)  *1000./ (na * lwc)
c-----------------------------------------------------------------------
      write(*,'(1p,(8e12.4))') aspec(kso4a), aspec(khno3a), 
     _                        aspec(kso2a), aspec(kco2a), 
     _                        aspec(kho2a), aspec(kfaa)
      write(*,'(1p,(8e12.4))') so4a, no3a, so2a, co2a, ho2a, faa

c use initial guess of pH for this case
      if(so4a.eq.0. .and. no3a.eq.0. .and. so2a.eq.0. .and.
     _   co2a.eq.0. .and. ho2a.eq.0. .and. faa .eq.0.) return


      fact = (1./t) - 1./298.
      k1fa = 1.8e-4*exp(-1510.*fact)
      k1s  = 1.3e-2*exp(-1960.*fact)
      k2s  = 6.6e-8*exp(-1500.*fact)
      k1c = 4.5e-7*exp(-100.*fact)
      k2c = 3.61e-11*exp(-1760.*fact)

   5  continue
      hion = 10.**(-ph)

      f_o2m  = 3.5e-5/(3.5e-5 + hion)
      f_fo   = k1fa/(k1fa + hion)
      f_hso3 = k1s*hion/(hion*hion + k1s*hion + k1s*k2s)            
      f_hco3 = k1c*hion/(hion*hion + k1c*hion + k1c*k2c)            

      ahion = so4a + no3a + f_hso3*so2a + f_hco3*co2a + 
     _        f_o2m*ho2a + f_fo*faa 

      ph = -alog10(ahion)
      if(ph .lt. 0. .or. ph .gt. 10.) then
       write(*,*) 'pH = ', ph, ' stopping program'
       write(*,'(1p,(8e12.4))') ahion, hion, so4a, no3a, 
     _           f_hso3*so2a, f_hco3*co2a, f_o2m*ho2a, f_fo*faa
       stop
      endif
      if(abs(hion-ahion) .gt. 0.001*hion) go to 5
      write(*,*) 'pH = ', ph
      write(*,'(1p,(8e12.4))') ahion, hion, so4a, no3a, 
     _           f_hso3*so2a, f_hco3*co2a, f_o2m*ho2a, f_fo*faa
      return
      end
c**********************************************************************c
      subroutine rates(t, rho, qv, qc, ph, rate, ratea, time)
c this subroutine computes the rates for the various reactions
      implicit none
      include 'chem_params.inc'
c
c     input parameters
c  t                         temp                     
c  rho                       air density (kg/m3)                     
c  qv                        water vapor mixing ratio (g/g)
c  qc                        cloud water mixing ratio
c  ph                        ph of liquid water
c
c         output parameters
c  rate                      gas-phase reaction rates
c  ratea                     cloud reaction rates
c
c         common block storage
c
c  dx                        grid length x direction (m)
c  dy                        grid length y direction (m)
c  p0                        reference pressure
c  dtchem                        time step
c
c  cw                         molecular weights
c  cx                         character list of species
c  stc                        matrix of standard stoichometry coefficients
c  icmap                      reaction matrix for each species
c  nreact                     total number of reactions
c  a                          temperature independent coeff for arrehenius reactions
c  s                          temperature   dependent coeff for arrehenius reactions
c  stc                        stoichmetry coefficients
c
c*** chemical reaction types
c    nr1,nr2,nr3,nr4,nr5      number of each reaction type
c    itype1,...itype5         reaction identifier for each reaction type
c    itype1 - photolysis 
c    itype2 - simple arrhenius
c    itype3 - a + m or a + b + m, possibly also arrhenius
c    itype4 - troe  -  already assigned at read
c    itype5 - special functions for rate constants 
c
c***  troe reaction data, molecule, cm3, sec units
c  ak0300 - zero pressure 300k rate constant, third order
c  an     - temperature exponent for zero pressure rate constant
c  aki300 - high pressure 300k rate constant, second order
c  am     - temperature exponent for high pressure rate constant
c  bas    - base of exponentiation, 0.6 for most
c  aequil - pre-exponential of equilibrium constant
c  tequil - activation temperature of equilibrium constant
c
c         temporary storage
c
c  rate                       reaction rate
c  iwhich                     index indentifier for special reactions
c  press                      press
c  conv                       conversion factor
c  m                          concentration of air
c  h2o                        concentration of water
c  **variables used in calculating rates
c  fact 
c  trel 
c  ak0t 
c  akit 
c  f1 
c  f2 
c  f3 
c  f4 
c  forw 
c  fh2o
c  rk0
c  rk2
c  rk3
      
c     input variables
 
      real qv, qc, t, rho
      real ph
      real time

c     output variable

      real rate(nrmx)
      real ratea(nrmx)

c
      common /dltprms/dtchem, dt_dbg
      real dtchem                ! the chemistry timestep  
      real dt_dbg                ! how often to do a debug print

c chemical storage block

      include 'cchem.com'
      include 'ichem.com'
      include 'rchem.com'
      include 'irate.com'
      include 'rrate.com'
      include 'hvchem.com'


c temporary storage

      integer mr1, jtim, jtimp, irphot(30), ip
      real tday, tj, pslope, tdif, dtj, otim, delttj

      real  m, h2o, press

      real fact,trel,ak0t,akit,f1,f2,f3,f4,forw,fh2o,
     & rk0,rk2,rk3,conv

      integer icld
      integer i,j,k,it,ir, kern

      real rstr(nrmx)
      real lww              !liquid water content of cloud water
      real hion             !H+ concentration in cloud/rain drops
      real na               !Avogadro's number
      real f_ho2            !fraction of HO2 + O2- that is HO2
      real f_o2m            !fraction of HO2 + O2- that is O2-
      real f_fa             !fraction of HCOOH + HCOO- that is HCOOH
      real f_fo             !fraction of HCOOH + HCOO- that is HCOO-
      real f_hco3           !fraction of CO2 + HCO3- that is HCO3-
      real k1fa             !dissociation const for formic acid
      real k1c              !dissociation const for CO2
      real k1s              !dissociation const for H2SO3-->HSO3-
      real k2s              !dissociation const for HSO3-->SO3=
      real f_hso3           !fraction of SO2 + HSO3- + SO3= that is HSO3-
      real f_so3            !fraction of SO2 + HSO3- + SO3= that is SO3=
      real rk               !temporary rate constant for a01A+HOA
      real rkh2o            !rate constant for O1D + H2O
      real rkm              !rate constant for O1D + M  

      do 10 ir=1,nreact
       rate(ir) =a(ir)
       ratea(ir)=0.
 10   continue

       write(*,*) 'beg rates ', rate(4)

c     find concentrations of air and water
c     conv converts to from kg/m3 to grams/cm3

      press = rho*287.*t                        ! N/m2
      conv  = 1.e-3*rho                           ! g/cm3
      m     = conv*         (6.0221e23/(28.97  )) ! molec/cm3
      h2o   = conv*qv*(6.0221e23/(18.0152))       ! g air/cm3 * g H2O/g air * mol H2O/g H2O * molecules/mol --> molecules/cm3
       write(*,*) 'M, H2O ', m, h2o, press, rho, qv, t, time


C
C  PHOTOLYSIS REACTIONS
C
      if (nr1 .gt. 0)   then

       tday = 24.*3600.

       tj = time - tday*float(int(time/tday))
       delttj = tjval(2)-tjval(1)
       jtim=int(tj/delttj) +1
       jtimp=jtim+1

       if(jtim.eq.nphot) then
         jtimp=1
       elseif(jtim.gt.nphot) then
         print*,'something is wrong '
         stop
       endif

       tdif = tjval(jtimp) - tjval(jtim)
       dtj = tj - tjval(jtim)
         print*,tjval(jtim),tj,tjval(jtimp),jtim
         print*,xjval(jtim,1),xjval(jtimp,1),tdif, dtj

        if(mod(time,3600.).eq.0.) print*,'photolysis rates'
       do 30 it=1,nr1
         ir=itype1(it)
         pslope=(xjval(jtimp,it) - xjval(jtim,it)) / tdif
         rate(ir) = abs(xjval(jtim,it) + pslope*dtj)
         if(mod(time,3600.).eq.0.) print*,it,ir,rate(ir)
  30   continue
 341   format(f10.1,1x,i2,1x,i2,e12.5)


      endif ! end of photolysis block
c aqueous photolysis rates:
      if(nr7 .gt. 0) then
       do it=1,nr7
         ir=itype7(it)
         ip = hvmap(it)
         ratea(ir) = 1.5 * rate(ip)
       end do
      endif

C
C      THIS CALCULATES RATES CONSTANTS FOR SIMPLE ARRHENIUS 
C        EXPRESSIONS FOR SELECTED REACTIONS

        if (nr2 .gt. 0)	  then
         do 40 it = 1, nr2
           ir=itype2(it)        

           rate(ir) = a(ir)*exp(s(ir)/t)

           rstr(ir) = rate(ir)
 40      continue
        endif
       write(*,*) 'after arr ', rate(4)

C       THIS CALCULATES RATES CONSTANTS FOR SIMPLE ARRHENIUS 
C        EXPRESSIONS WHICH INCLUDE THE COLLISION PARTNER M
C        NOTE THAT TROE AND REVERSE TROE ARE EXCLUDED.

        IF (NR3 .GT. 0)	  then
! mcb added 14 Oct 2015
!**** Since this is for O + O2 + M --> O3, just write as JPL2011 has it (and Emmons et al 2010) ****!
         do 50 it = 1, nr3
           ir=itype3(it)        
           rate(ir) = m*a(ir)*(300./t)**2.4            ! mcb end 14 Oct

!           fact = 3.3556e-03  -  1./t                        ! old method
!           rate(ir) = m*a(ir)*exp(fact*s(ir))                ! old method
 50      continue
        endif
        write(*,*) 'after  M  ', rate(4)

	IF (NR4 .GT. 0) then
C      THIS CALCULATES RATE CONSTANTS FOR REACTIONS OF THE TYPE
C          A + B + (M)  ->  C + (M)      and
C          C + (M)  ->  A + B + (M)
C       USING TROE EXPRESSIONS AND EQUILIBRIUM CONSTANTS FOR THE REVERSE
C       TREL   - temperature / 300.
C       AK0T   - low press lim k at current temp
C       AKIT   - high press. lim k at current temp
C       F1,F2,F3,F4 - temporary values
C       FORW   - forward rate const.
C______________________________________________________________________

         do 60 it = 1, nr4
           ir=itype4(it)        

           fact = (1./t)- (1./298.)  
           trel = t/300.
           ak0t = ak0300(it) * trel**(-an(it))
           akit = aki300(it) * trel**(-am(it))
           f1 = ak0t*m/akit
	   f2 = ak0t*m/(1. + f1)
	   f3 = alog(max(f1,1.e-30))/2.303
	   f4 = 1./(1. + f3*f3)
	   forw = f2 * bas(it)**f4

c          print*,'troe reaction for reaction ',ir
c          print*,forw,aequil(it),tequil(it),fact

           rate(ir) = forw / ( aequil(it) * 
     |          exp(tequil(it)*(fact)))
 60      continue
        endif
       write(*,*) 'after (M) ', rate(4)

	if (nr5 .gt. 0) then
c SPECIAL REACTIONS

        do 70 it = 1, nr5
          ir=itype5(it)        
           write(*,*) 'start special ', ir, rate(ir)

C HO2+HO2        
	   if (iwhich(it) .eq. 1) then
                fact = (1./t)- 3.3556e-03
		rk2 =  1.7e-12*exp( 600.*(fact))        ! = 2.27e-12*exp(600/T)
	        rk3 =  4.9e-32*exp(1000.*(fact))        ! = 1.71e-33*exp(1000/T)
	        fh2o = 1.4e-21*exp(2200./t)             
	        rate(ir) = (rk2 + rk3*m)*(1. + fh2o*h2o)
	   endif

C HNO3+HO
	   if (iwhich(it) .eq. 2) then
	        rk0 = 2.4e-14*exp(460./t)
                rk2 = 2.7e-17*exp(2199./t)
	        rk3 = 6.5e-34*exp(1335./t)
	        rate(ir) = rk0 + rk3*m/(1. + rk3*m/rk2)
	   endif

C CO+HO

	   if (iwhich(it) .eq. 3) then
c              print*,'pressure: ',press,press/101325.
              rate(ir)=1.5e-13*(1. + 0.6*press/(101325.)) 
	   endif

C O3+hv

	   if (iwhich(it) .eq. 4) then
            write(*,*) 'O3+hv ', rate(ir), h2o, m
              rkh2o = 2.2e-10
              rkm   = 2.9e-11
              rate(ir)= rate(ir) * rkh2o*h2o / (rkh2o*h2o + rkm*m)
	   endif

C CH3COCH3+HO  or  k031+HO
           if (iwhich(it) .eq. 18) then
              rate(ir) = 3.82e-11*exp(-2000./t) + 1.33e-13
           endif
C XOOH+HO
           if (iwhich(it) .eq. 19) then
              rate(ir) = (t*t)*7.69e-17*exp(253./t)
           endif

c Special Reactions in Aqueous Reaction Mechanism
c  First determine [H+] concentration

        hion = 10.**(-ph)

C HO2aq + O2-

           if (iwhich(it) .eq. 6) then
             f_o2m = 3.5e-5/(3.5e-5 + hion)
             f_ho2 = hion/(3.5e-5 + hion)
             rate(ir)=rstr(ir) * f_ho2 * f_o2m
           endif

C HOaq + O2-

           if (iwhich(it) .eq. 7) then
             fact = (1./t) - 3.3556e-03
             f_ho2 = hion/(3.5e-5 + hion)
             f_o2m = 3.5e-5/(3.5e-5 + hion)
c             rate(ir)=(f_ho2*7.e9 + f_o2m*1.e10) * 
             rate(ir)=(f_o2m*1.e10) * exp(-1500.*fact)
           endif

C O3aq + O2-

           if (iwhich(it) .eq. 8) then
             f_o2m = 3.5e-5/(3.5e-5 + hion)
             rate(ir)=rstr(ir) * f_o2m
           endif

C CH3OOaq + O2-

           if (iwhich(it) .eq. 9) then
             f_o2m = 3.5e-5/(3.5e-5 + hion)
             rate(ir)=rstr(ir) * f_o2m
           endif

C HCOOHaq + OHaq and HCOO- + OHaq

           if (iwhich(it) .eq. 10) then
             fact = (1./t)- 3.3556e-03
             k1fa = 1.8e-4*exp(-1510.*fact)
             f_fa = k1fa/(k1fa + hion)
             f_fo = hion/(k1fa + hion)
             rk = (1.6e8*f_fo + 2.5e9*f_fa) * exp(-1510.*fact)
             rate(ir)= rk
             write(*,*) '          ',
     _         1.6e8*f_fo,  2.5e9*f_fa, rate(ir), ir
           endif

c SO2(aq) + H2O2(aq)
          if(iwhich(it) .eq. 16) then          !SO2 aqchem
             fact = (1./t)- 3.3556e-03
             k1s  = 1.3e-2*exp(-1960.*fact)
             k2s  = 6.6e-8*exp(-1500.*fact)
             f_hso3 = k1s*hion/(hion*hion + k1s*hion + k1s*k2s)            !fraction of S(IV) that's HSO3
             rk = 7.45e7*exp(-4430.*fact)
             rate(ir) = f_hso3 * rk*hion/(1. + 13.*hion)
             write(*,*) '      ', rk, hion, hion/(1. + 13.*hion),
     +        f_hso3, rate(ir)
          endif

c SO2(aq) + O3(aq)
          if(iwhich(it) .eq. 17) then          !SO2 aqchem
             fact = (1./t)- 3.3556e-03
             k1s  = 1.3e-2*exp(-1960.*fact)
             k2s  = 6.6e-8*exp(-1500.*fact)
             f_hso3 = k1s*hion/(hion*hion + k1s*hion + k1s*k2s)            !fraction of S(IV) that's HSO3
             f_so3  = k1s*k2s /(hion*hion + k1s*hion + k1s*k2s)            !fraction of S(IV) that's HSO3
             rk = 3.7e5*exp(-5530.*fact)*f_hso3 +
     +            1.5e9*exp(-5280.*fact)*f_so3
             rate(ir) = rk
             write(*,*) '      ', 3.7e5*exp(-5530.*fact)*f_hso3,
     +        1.5e9*exp(-5280.*fact)*f_so3, f_hso3, f_so3,
     +        rate(ir)
          endif

C HCO3- + OHaq 

           if (iwhich(it) .eq. 11) then
             fact = (1./t)- 3.3556e-03
             k1c = 4.5e-7*exp(-100.*fact)
             f_hco3 = k1c/(k1c + hion)
             rk = 1.0e7*f_hco3 * exp(-1510.*fact)
             rate(ir)= rk
           endif

C HCO3- + O2-

           if (iwhich(it) .eq. 12) then
             fact = (1./t)- 3.3556e-03
             k1c = 4.5e-7*exp(-100.*fact)
             f_hco3 = k1c/(k1c + hion)
             f_o2m = 3.5e-5/(3.5e-5 + hion)
             rk = 1.5e6*f_hco3 * f_o2m * exp(-1510.*fact)
             rate(ir)= rk
           endif

C CO3- + O2-

           if (iwhich(it) .eq. 13) then
             fact = (1./t)- 3.3556e-03
             f_o2m = 3.5e-5/(3.5e-5 + hion)
             rk = 4.0e8 * f_o2m * exp(-1510.*fact)
             rate(ir)= rk
           endif

C CL2- + HO2 and CL2- + O2-

           if (iwhich(it) .eq. 14) then
             fact = (1./t)- 3.3556e-03
             f_o2m = 3.5e-5/(3.5e-5 + hion)
             f_ho2 = hion/(3.5e-5 + hion)
             rk = (4.5e9*f_ho2 + 1.0e9*f_o2m) * exp(-1510.*fact)
             rate(ir)= rk
           endif

C NO3 + O2-

           if (iwhich(it) .eq. 15) then
             f_o2m = 3.5e-5/(3.5e-5 + hion)
             rate(ir)=rstr(ir) * f_o2m
           endif

C CL2- reactions
c           if (iwhich(it) .eq. 10) then
c             fact = (1./t)- 3.3556e-03
c             k1cl = 5.3e-6
c             f_cl2m= k1fa/(k1fa + hion)
c             f_cla = hion/(k1fa + hion)
c             f_clm = hion/(k1fa + hion)
c             rk = (1.6e8*f_fo + 2.5e9*f_fa) * exp(-1510.*fact)
c             rate(ir)= rk
c           endif



           write(*,*) 'end special ', ir, rate(ir)
 70        continue

        endif

       write(*,*) 'after special ', rate(4)



        na = 6.0221e20
        if(nr6 .gt. 0) then
c modify aqueous reaction rate constants (molar units to molec/cm3 units)
         lww = rho * qc/1.e6
         icld = nint(lww / max(lww,1.e-20))

         do it=1,2
          ir = itype6(it)
c          ratea(ir) = rate(ir)    !photolysis rates -- already assigned above
          ratea(ir) = ratea(ir) *icld
          rate(ir)  = 0.
         end do
         
         write(*,*) 'in rates lww: ', lww, icld
         do it=3,nr6
          ir = itype6(it)
           if(ir.eq.4) write(*,*) ir, rate(ir), na, lww
           ratea(ir) = rate(ir)/(na*max(lww, 1.e-20)) * icld
           rate(ir)  = 0.
         end do

        endif

       write(*,*) 'after aq  ', rate(4)

 1000 continue

      if(mod(time,1800.) .eq. 0.) then
       print*,'reaction rates at time ', time
       do i=1,nreact
        print*,i,rate(i), ratea(i)
       enddo
      endif

      return
      end

c----------------------------------------------------------------------c
c            heff= Kh*Ra*T and is dimensionless                        c
c----------------------------------------------------------------------c
      subroutine glrates(tair, rho, ph, ktc, lww, heff, phrat)

      implicit none
      include 'chem_params.inc'
c Input variables
      real tair, rho, ph, lww

c Output variables
      real ktc(naq)               ! first-order rate for transfer of species
c                                   into/out of drop
      real heff(naq)              ! dimensionless Henry's law coefficient
      real phrat(naq)             ! phase ratio = Ca/Cg

c Local variables
      integer ik, it
      real na, ahion
      data na /6.0221e23/

c Externals
      external gasflux,           !gets ktc
     _         solub              !gets heff, phrat


      ahion = 10.**(-ph)

      call gasflux(lww, tair, ktc)
      call solub(tair, ahion, lww, rho, heff, phrat) 

      return
      end
c----------------------------------------------------------------------c
      subroutine gasflux(lww, tair, ktc)
c  Parameterization of gas to drop transfer; follows Schwartz (1986)
c
      implicit none
      include 'chem_params.inc'

c Input variables
      real lww, tair

c Output variables
      real ktc(naq)


c Local variables
      integer ik, n
      real third
      parameter(third=1./3.)
      real taudg, taui
      real nd, amean, na, acc(naq)
      real ru, dg, thvel 
      parameter(ru=8.314e7, dg=0.10, na=6.0221e23)
      real nor, rhol, pi            !Nor=0.08 (cm-4), rhol=1g/cm3, pi
      data nor, rhol/0.08, 1./

      real mw(naq)

c-----------------------------------------------------------------------
c
c Initialize
c acc = 0.05 is default value from Lelieveld & Crutzen 1991
c acc for VOCs are from McNeill et al (2012) supplementary info and 
c her references:  (some refs may be for KH)
c  5 = Betterton and Hoffmann (1988)
c  6 = Lim et al (2005)
c  7 = Kroll et al (2005)
c  8 = Herrmann et al (2005) CAPRAM 3.0
c 11 = Khan et al (1995)
c 12 = Davidovits et al (1995)
c 17 = Sander and Crutzen (1996)
c 16 = Sander (2012) NIST chemistry webbook
c 18 = McNeill et al Atmos Chem (2012)
c 19 = Lelieveld & Crutzen 1991
c 20 = Kolb et al (2010)


      pi = 4.*atan(1.)
      acc(ko3a)   = 5.3e-4
      acc(kh2o2a) = 0.02
      acc(koha)   = 0.05
      acc(kho2a)  = 0.2
      acc(kch3ooa)= 0.05
      acc(kmpera) = 0.05
      acc(kch2oa) = 0.05
      acc(kfaa)   = 0.012          ! following McNeill et al 2012, (8, 11, 12)
      acc(knoa)   = 0.005
      acc(kno2a)  = 6.3e-4
      acc(khno3a) = 0.2            ! following McNeill et al 2012, (19,20)
      acc(kn2o5a) = 0.005          ! following McNeill et al 2012, (6,17,18)
      acc(kno3a)  = 0.001
      acc(kco2a)  = 0.05
      acc(kso2a)  = 0.05
      acc(kso4a)  = 0.2            ! assumed same as HNO3
      acc(kclm)   = 0.1
      acc(kcl2m)  = 0.1
      acc(kcla)   = 0.1
      acc(kco3m)  = 0.1
! Following accommodation coefficients are from McNeil et al (2012) supplementary material
      acc(khno4a) = 0.2             ! assumed same as HNO3
      acc(ko011a) = 0.015           ! 8, 12, 16
      acc(kdo23a) = 0.023           ! estimated by McNeil et al
      acc(kdk33a) = 0.023           ! assumed same as glyoxal
      acc(kh021a) = 0.05            ! assumed same as h011
      acc(kho34a) = 0.05            ! assumed same as h011
      acc(khtA1a) = 0.05            ! assumed same as h011
      acc(khk44a) = 0.05            ! assumed same as h011
      acc(khk33a) = 0.05            ! assumed same as h011
      acc(khu51a) = 0.05            ! assumed same as h011
      acc(kxooha) = 0.05            ! assumed same as h011
      acc(kd021a) = 0.03            ! 5, 8
      acc(kud42a) = 0.02            ! estimated by McNeil et al
      acc(kdd21a) = 0.023           ! 7, 8
      acc(kdd55a) = 0.03            ! assumed same as d021
      acc(kk031a) = 0.02            ! assumed same as MVK
      acc(khyaca) = 0.02            ! assumed same as MVK
      acc(kuk41a) = 0.02            ! estimated by McNeil et al
      acc(kg021a) = 0.02            ! estimated
      acc(kn002a) = 0.02            ! estimated
      acc(kp021a) = 0.02            ! estimated
      acc(kpu44a) = 0.02            ! estimated
      acc(k3021a) = 0.02            ! estimated

      mw(ko3a) = 48.
      mw(kh2o2a) = 34.
      mw(koha) = 17.
      mw(kho2a) = 33.
      mw(kch3ooa) = 47.
      mw(kmpera) = 48.
      mw(kch2oa) = 30.
      mw(kfaa) = 46.
      mw(knoa) = 30.
      mw(kno2a) = 46.
      mw(khno3a) = 63.
      mw(kn2o5a) = 108.
      mw(kno3a) = 62.
      mw(kso2a) = 64.
      mw(kso4a) = 96.
      mw(kco2a) = 44.
      mw(kclm) = 35.453
      mw(kcl2m) = 70.906
      mw(kcla) = 35.453
      mw(kco3m) = 60.
      mw(khno4a) = 79.
      mw(ko011a) = 32.
      mw(kdo23a) = 60.
      mw(kdk33a) = 72.
      mw(kh021a) = 62.
      mw(kho34a) = 92.
      mw(khtA1a) = 186.
      mw(khk44a) = 120.
      mw(khk33a) = 90.
      mw(khu51a) = 118.
      mw(kxooha) = 134. 
      mw(kd021a) = 44.
      mw(kud42a) = 70.
      mw(kdd21a) = 58.
      mw(kdd55a) = 100.
      mw(kk031a) = 58.
      mw(khyaca) = 74.
      mw(kuk41a) = 70.
      mw(kg021a) = 76.
      mw(kn002a) = 147.
      mw(kp021a) = 121.
      mw(kpu44a) = 147.
      mw(k3021a) = 75.

      do n=1,naq

        ktc(n) = 0.
c cloud water
        amean = 0.001                      !*********CHANGE for PARCEL MODEL**
        taudg = amean**2/(3.*dg)

        thvel = sqrt(8.*ru*tair/(pi*mw(n)))
        taui = 4.*amean/(3.*thvel*acc(n))
        if(taudg .gt. 0. .or. taui .gt. 0.) 
     _               ktc(n) = 1./(taudg + taui)
        if(lww .lt. 1.e-15) ktc(n) = 0.

      end do
c
      return
      end
c----------------------------------------------------------------------c
      subroutine solub(temp, ahp, lww, rho, heff, phrat) 
 
c  calculate the effective henry's law constant and the phase ratio
c   for each species

      implicit none
      include 'chem_params.inc'
      include 'cchem.com'
      include 'amap.com'

c Input variables
      real temp, lww, rho
      real ahp
    
c Output variables
      real heff(naq), phrat(naq)

c Common blocks
      real hft(142,9,2)
      common/effhenry/hft

c Local variables
      integer ik, n, it
      real ra

      real ek, e298, dhr, tt, t
c-----------------------------------------------------------------------
      real ekw, ho3, hh2o2, eh2o2, hoh, hho2, eho2, hch3oo, hch3ooh
      real hch2o, hhcooh, ehcooh, hno, hno2, hhno3, ehno3, hco2, eh2co3
      real ehco3, hnh3, enh4oh, hso2, eh2so3, ehso3, hn2o5, hno3
      real hclm, hcl2m, hcla, hco3m
      real hhno4, ho011, hdo23, hdk33, hh021, hd021, hud42, hdd21 
      real hdd55, hk031, huk41, hg021, hp021 
      real ehno4

c  functions
      ek(e298,dhr,tt) = e298*exp(dhr*(1./tt-1./298.))

      ekw(t)    = ek(1.e-14,-6716.,t)        !ln H = A + B/T + C ln(T) from JPL Sander et al. (2011)
      ho3(t)    = ek(1.03e-2,2830.,t)         ! H298=1.03e-2  A=-14.08  B=2830 C=0
      hh2o2(t)  = ek(8.44e4,7600.,t)          ! 8.44e4         -14.16    7600
      eh2o2(t)  = ek(2.2e-12,-3700.,t)     
      hoh(t)    = ek(39.,   0.,t)            ! 39
      hho2(t)   = ek(690.,    0.,t)          ! 690            JPL Sander et al !      = ek(3.8e3,5900.,t)     ! Hanson and Burkholder (1992)
      eho2(t)   = ek(3.5e-5,0.,t)           
      hch3oo(t) = ek( 2.7,2030.,t)           ! 2.7*exp(2030(1/T-1/298))   Leriche et al 2013 GMD
!                                             !H298          A         B      C  
      hch3ooh(t)= ek(300., 5280.,t)          ! 300         -11.99   5280
      hch2o(t)  = ek(3.23e3,7100.,t)         ! 3.23e3      -15.73   7100
      hhcooh(t) = ek(8.9e3, 6100.,t)         ! 8.9e3        -11.40   6100
      ehcooh(t) = ek(1.8e-4,-1500.,t)     
      hno(t)    = ek(1.91e-3,1790.,t)        ! 1.92e-3      -157.1   7950    21.298   !Using 2 parameter Battino (1981) param
      hno2(t)   = ek(1.2e-2,2360.,t)        ! 1.2e-2        -12.32  2360
      hhno3(t)  = ek(2.6e6,8700.,t)         ! 
      ehno3(t)  = ek(15.,0.,t)
      hno3(t)   = ek(3.8e-2,   0.,t)        ! 3.8e-2
      hso2(t)   = ek(1.23,3120.,t)          ! 1.36          -39.72  4250    4.525
      eh2so3(t) = ek(1.3e-2,2000.,t)        !   
      ehso3(t)  = ek(6.3e-8,1500.,t)        !   
      hco2(t)   = ek(3.6e-2,2200.,t)        ! 3.38e-2       -145.1  8350    19.960
      eh2co3(t) = ek(4.5e-7,-1000.,t)       !   
      ehco3(t)  = ek(3.61e-11,-1760.,t)     !   
      hnh3(t)   = ek(56.,4100.,t)           ! 60.2          -9.84   4160
      enh4oh(t) = ek(1.7e-5,-450.,t)
      hhno4(t)  = ek(1.2e4,6900.,t)        
      ehno4(t)  = ek(0., 0., t)        !*************
      ho011(t)  = ek(203., 9240., t)        ! 203        -97.53    9240    12.16
      hdo23(t)  = ek(4.1e4,4600., t)        !Betterton and Hoffmann (1988)
      hdk33(t)  = ek(3.4e3,7500., t)        ! ditto
      hh021(t)  = ek(336., 5995., t)        ! 336        -14.28    5995           O'Sullivan et al (1996)
      hd021(t)  = ek(12.9, 5890., t)        ! 12.9       -17.19    5890           Betterton and Hoffmann (1988)
      hud42(t)  = ek(4.8,  4300., t)        ! Ji and Evans (2007)
      hdd21(t)  = ek(4.19e5,7480., t)        ! 4.19e5     -12.15    7480  Ip et al (2009) & Volkamer et al (2009) & Sander et al (2011)
      hdd55(t)  = ek(3.0e4,9200., t)        ! Olson (1998)
      hk031(t)  = ek(27.8, 5530., t)        ! 27.8       -15.23    5530  Betterton (1991)
      huk41(t)  = ek(26.,  4800., t)        ! Ji and Evans (2007)
      hg021(t)  = ek(837., 5310., t)        ! 837        -11.07    5310        O'Sullivan et al (1996)
      hp021(t)  = ek(2.8,  5730., t)        ! 2.8        -18.15    5730        Warneck and Williams (2012, Sander et al (2006, 2012)


      
!!!   hMEK(t) = ek(18., 5700., t)          ! Sander et al (2011) and close to Snider and Dawson (1985)      
!!!   ha021(t) = ek(5400., 8300., t)          ! Khan and Brimblecombe (1992)  (Sander uses ek(4000., 6200., t))

c----------------------------------------------------------------------c
 
c----------------------------------------------------------------------c
c calculate effective henry's law constants
!  organic compounds KH from Sander (2015) ACP, but chosen value has specific reference
 
       heff(ko3a)   = ho3(temp)
       heff(kh2o2a) = hh2o2(temp) *(1.+eh2o2(temp)/ahp)
       heff(koha)   = hoh(temp)
       heff(kho2a)  = hho2(temp) *(1.+eho2(temp)/ahp)
       heff(kch3ooa)= hch3oo(temp)
       heff(kmpera) = hch3ooh(temp)
       heff(kch2oa) = hch2o(temp)
       heff(kfaa)   = hhcooh(temp) *(1.+ehcooh(temp)/ahp)
       heff(knoa)   = hno(temp)
       heff(kno2a)  = hno2(temp)
       heff(khno3a) = hhno3(temp) *(1.+ehno3(temp)/ahp)
       heff(kn2o5a) = 1.e12
       heff(kno3a)  = hno3(temp)
       heff(kso2a)  = hso2(temp) *(1.+eh2so3(temp)/ahp +
     _                eh2so3(temp)*ehso3(temp)/ahp**2 )
       heff(kso4a)  = 1.e12            ! have high heff to keep in aq phase
       heff(kco2a)  = hco2(temp) *(1.+eh2co3(temp)/ahp +
     _                eh2co3(temp)*ehco3(temp)/ahp**2 )
       heff(kclm)   = 1.e12            ! PRESCRIBE to
       heff(kcl2m)  = 1.e12            ! have high heff to keep in aq phase
       heff(kcla)   = 1.e12
       heff(kco3m)  = 1.e12
       heff(khno4a) = hhno4(temp)
       heff(ko011a) = ho011(temp)
       heff(kdo23a) = hdo23(temp)
       heff(kdk33a) = hdk33(temp)
       heff(kh021a) = hh021(temp)
       heff(kho34a) = heff(kh021a)      !????
       heff(khtA1a) = heff(kmpera)      !mozart mech method
       heff(khk44a) = heff(kmpera)
       heff(khk33a) = heff(kmpera)
       heff(khu51a) = heff(kmpera)
       heff(kxooha) = heff(kmpera)
       heff(kd021a) = hd021(temp)
       heff(kud42a) = hud42(temp)
       heff(kdd21a) = hdd21(temp)
       heff(kdd55a) = hdd55(temp)
       heff(kk031a) = hk031(temp)
       heff(khyaca) = 7700.             ! Lee and Zhou (1993)
       heff(kuk41a) = huk41(temp)
       heff(kg021a) = hg021(temp)
       heff(kn002a) = 1000.             ! Kames and Schurath (1992) and Sander uses it
       heff(kp021a) = hp021(temp)
       heff(kpu44a) = 1.7               ! Kames and Schurath (1992)
       heff(k3021a) = 0.1               ! Sander et al (2006, 2011) and Villalta et al (1996)
! 23 more aq species

      write(*,*) 'in solub', (heff(n), n=1,naq)
      write(*,*) '        ', temp, lww
      ra = 8314./101325.
      do n=1,naq
        heff(n)  = heff(n)*ra*temp           !dimensionless form
        phrat(n) = heff(n)*lww               !Phase ratio
        write(*,'(i4,1p,2e12.5,3x,a4)') n, heff(n), phrat(n), 
     _   cx(aqmap(n))
      end do

      return
      end
c======================================================================c
      subroutine chem(rate,t,q,spec,ratea,ktc,heff,phrat,aspec,time)

c  this subroutine computes the rates for the various reactions
c  then solves the rate equations. they are solved with the Euler-
c  Backward iterative method
c
c  Aug. 14/95
c  For steady-state species, the fluxes and losses in the boundary
c  Layer are included in the chemistry scheme  
c
c
c  Mar '96: adjusted for concentration mixing ratios
c           took away deposition and flux terms from lower layer
c           allowed chemistry in the first layer
c
c   variables: see chemset for chemical storage block and parameters
c
c         temporary storage
c   nsrcs                    number of sources for each species
c   nsnks                    number of sinks for each species
c   srca(b)                  sum of total sources for each species a=time n, b=time n+1
c   xlsa(b)                  sum of total sinks for each species   a=time n, b=time n+1
c   xa(b)                    species concentration                 a=time n, b=time n+1
c   xspec                    species concentration

      implicit none

      include 'chem_params.inc'      !c++mcb

c inputs

      real t, q, spec(nspec)
      real qca, lww, ph 
      integer ncpts, npts
      common/cldpar/qca, lww, ph, ncpts, npts

      real rate(nrmx)  
      real ratea(nrmx) 
      real aspec(naq)                !cloud water chemical species
      real ktc(naq)                  !diffusion transfer rate coefs.
      real heff(naq)                 !effective henry's law coef
      real phrat(naq)                !phase ratio = Ca/Cg

c commons
c chemical storage block

c grid storage block

      common /dltprms/dtchem, dt_dbg
      real dtchem                ! the chemistry timestep  
      real dt_dbg                ! how often to do a debug print

c chemical storage block

      include 'cchem.com'
      include 'ichem.com'
      include 'rchem.com'
      include 'amap.com'

      common /constant/pi,r,gr,cp
      real pi    ! pi                                             = PI
      real r     ! gas constant for dry air                       = RD
      real gr    ! acceleration of gravity                        = GZERO
      real cp    ! specific heat of dry air at constant pressure  = CPD

c temporary storage
      real srca, xlsa, xa(nspec), xb(0:nspec)
      real press, convu

      integer ik, ir1, iy, iu
      real time
      integer ixa, nsnka, nsrca
      real xac(naq)
      real xbc(0:naq)

      real ctot1t(naq), ctot1(naq), xold
      real srcaa, snkaa
      real xbn, xbcn
      real xtmp

      integer icyc, ncyc

      real xls, src, p1,p2, mair, mh2o

      real ccrit, xnew

      logical more,pconv
      integer conv

c indexes
      integer ii,i3,i,j,k,ki,iter,ix,ir,kern,iii
      integer ip,nr1,is1,ip1,ip2,ip3
      integer iintt
      integer icount
      integer imxc, istr

      external soln          !c++mcb
c----------------------------------------------------------------------c

      ccrit=1.e-4

      ncyc = 1

      do icyc=1,ncyc

        xb(0) =1.
        xbc(0)=1.

c conv: density

c** The solution is somewhat sensitive to the
c** Order of the steady-state species in the input file

        do ix=1,nspec
          xb(ix)=spec(ix)                 ! molecules/cm3
          if(xb(ix).lt.1.e-10) xb(ix)=0.
        enddo      

        do ix=1,naq
          xbc(ix) = aspec(ix)    
        end do


      write(*,*) 'in chem, species concentration at time =', time
      write(*,'(10e11.4)') (spec(i), i=1,18 )

c iterate the solution
c++mcb  Do Gauss-Seidel method over two groups in this order,
c   (1) nspec and clear air points, and (2) all species on cloudy points 

      if(ncpts .eq. 0) then

       do ix=1,nspec
        xa(ix)=xb(ix)
       enddo

       iter=1 
       more=.true.

 1001  continue

       if(iter.gt.2) then
        more=.false.

        if(conv.eq.0) then
         icount=iter
         more=.true.
        endif

       endif

       if(more) then                 ! continue iterating over some gridpoints

        conv=1

c   solve for all other species

       do 500 ix=1,nspec              ! loop over species

        srca=0.
        xlsa=0.

c  find the total sinks for each species
        if(mod(time,dt_dbg) .eq. 0. .and. ix.eq.kch2o) 
     _     write(56,*) 'Gas sinks ', time, iter

        do 401 ir=1,icmap(ix,1)        ! loop over number of sink reactions 

         ip=(ir-1)*5
         nr1=icmap(ix,2+ip+1)        ! index of reaction rate
         is1=icmap(ix,2+ip+2)        ! index of stoich. coeff.
         ip1=icmap(ix,2+ip+3)        ! index of species #1
         ip2=icmap(ix,2+ip+4)        ! index of species #2
         ip3=icmap(ix,2+ip+5)        ! index of species #3
      
c  computation of the loss, where the loss is divided by the species concentration
c  xb(i,ip3)

         xlsa = xlsa + stc(is1)*rate(nr1)*xb(ip1)*xb(ip2)

         if(mod(time,dt_dbg) .eq. 0. .and. ix.eq.kch2o) then
          write(56,'(2i4,4e12.5,4x,e12.5)') ir, nr1, rate(nr1), 
     _     xb(ip1), xb(ip2), xb(ip3), stc(is1)*rate(nr1)*xb(ip1)*xb(ip2) 
         endif

  401   continue                     ! end of sink loop

c  find the total sources for each species
        if(mod(time,dt_dbg) .eq. 0. .and. ix.eq.kch2o) 
     _     write(56,*) 'Gas sources ', time, iter

        do 402 ir=1,icmap(ix,2)         ! loop over species (number of sources)

         ip=((ir-1)+icmap(ix,1))*5      
       
         nr1=icmap(ix,2+ip+1)        ! index of reaction rate
         is1=icmap(ix,2+ip+2)        ! index of stoich. coeff.
         ip1=icmap(ix,2+ip+3)        ! index of species #1
         ip2=icmap(ix,2+ip+4)        ! index of species #2
         ip3=icmap(ix,2+ip+5)        ! index of species #3

c  computation of the source

         srca = srca + stc(is1)*rate(nr1)*xb(ip1)*xb(ip2)*xb(ip3)

           if(mod(time,dt_dbg) .eq. 0. .and. ix.eq.kch2o) then
            write(56,'(2i4,4e12.5,4x,e12.5)') ir, nr1, rate(nr1), 
     _       xb(ip1), xb(ip2), xb(ip3),
     +       stc(is1)*rate(nr1)*xb(ip1)*xb(ip2)*xb(ip3) 
           endif

  402   continue                     ! end of source loop

c  computation of the new concentrations
c    solve the equations:

        src=srca     ! average source
        xls=xlsa     ! average loss
        xnew = (xa(ix) + dtchem*src)/(1. + xls*dtchem)

        if(abs(xnew-xb(ix)).gt.xb(ix)*ccrit) then
           conv=0
        endif

        if(iter .lt. 30) then
         xb(ix)=xnew
        else
         xtmp = xb(ix)
         xb(ix) = (xnew + xtmp) * 0.5
        endif

  500  continue                      ! end of species loop

       iter=iter+1
       if(iter.lt.50) then
         goto 1001
       else
         print*,'MORE THAN 50 ITERATIONS '
       endif

       endif                       ! no more points

      elseif(ncpts .gt. 0) then

c++mcb  (2) all species and cloudy points 

       iter = 0
       icount = 2

       do ix=1,nspec
        iy = gasmap(ix)
        xa(ix) = xb(ix)
        if(iy .gt. 0) then
         xac(iy) = xbc(iy)
c   define total concentration of species in aqueous phase
         ctot1(iy) = xa(ix) + xac(iy) 
         ctot1t(iy) = ctot1(iy)
        endif
       end do

       iter = 1
       more = .true.

 1002  continue

       if(iter.gt.2) then
        more=.false.
        if(conv.eq.0) then
          icount=iter
          more=.true.
        endif
       endif

       if(more) then                 ! continue iterating over some gridpoints

        conv=1

c  solve for aqueous species
        do ix=1,nspec              ! loop over species
         iy = gasmap(ix)
         srca=0.
         xlsa=0.
         srcaa=0.
         snkaa=0.

c find the gas sinks for each species
c        if(mod(time,dt_dbg) .eq. 0.) write(56,*) 'Gas sinks '
         do ir=1,icmap(ix,1)        ! loop over number of sink reactions 
          ip=(ir-1)*5
          nr1=icmap(ix,2+ip+1)        ! index of reaction rate
          is1=icmap(ix,2+ip+2)        ! index of stoich. coeff.
          ip1=icmap(ix,2+ip+3)        ! index of species #1
          ip2=icmap(ix,2+ip+4)        ! index of species #2
          ip3=icmap(ix,2+ip+5)        ! index of species #3
     
c computation of the loss; the loss is divided by the species concentration
c xb(i,ip3)

          xlsa = xlsa + stc(is1)*rate(nr1)*xb(ip1)*xb(ip2) 

c          if(mod(time,dt_dbg) .eq. 0.) then
c           write(56,'(2i4,4e12.5,4x,e12.5)') ir, nr1, rate(nr1), 
c     _      xb(ip1), xb(ip2), xb(ip3), stc(is1)*rate(nr1)*xb(ip1)*xb(ip2) 
c          endif

         end do                       ! end of sink loop

c find the gas sources for each species
c         if(mod(time,dt_dbg) .eq. 0.) write(56,*) 'Gas sources '

         do ir=1,icmap(ix,2)         ! loop over species (number of sources)
          ip=((ir-1)+icmap(ix,1))*5      
          nr1=icmap(ix,2+ip+1)        ! index of reaction rate
          is1=icmap(ix,2+ip+2)        ! index of stoich. coeff.
          ip1=icmap(ix,2+ip+3)        ! index of species #1
          ip2=icmap(ix,2+ip+4)        ! index of species #2
          ip3=icmap(ix,2+ip+5)        ! index of species #3

c computation of the source

          srca = srca + stc(is1)*rate(nr1)*xb(ip1)*xb(ip2)*xb(ip3)

c          if(mod(time,dt_dbg) .eq. 0.) then
c           write(56,'(2i4,4e12.5,4x,e12.5)') ir, nr1, rate(nr1), 
c     _      xb(ip1), xb(ip2), xb(ip3),
c     +      stc(is1)*rate(nr1)*xb(ip1)*xb(ip2)*xb(ip3) 
c          endif
         end do                     ! end of source loop

         if(iy .gt. 0) then

c find the cloud water sinks for each species
c          if(mod(time,dt_dbg) .eq. 0.) write(56,*) 'Aqueous sinks '
          nsnka = icmapa(iy,1)
          do i=1,nsnka
           ip = (i-1)*5
           ir1=icmapa(iy,2+ip+1)        ! index of reaction rate
           is1=icmapa(iy,2+ip+2)        ! index of stoich. coeff.
           ip1=icmapa(iy,2+ip+3)        ! index of species #1
           ip2=icmapa(iy,2+ip+4)        ! index of species #2
           ip3=icmapa(iy,2+ip+5)        ! index of species #3
 
           snkaa = snkaa + stc(is1) * ratea(ir1) * xbc(ip1) * xbc(ip2)

c           if(mod(time,dt_dbg) .eq. 0.) then
c            write(56,'(2i3,f5.2,5e11.4)') i, ir1, stc(is1), ratea(ir1), 
c     _       xbc(ip1), xbc(ip2), xbc(ip3), 
c     _       stc(is1)*ratea(ir1)*xbc(ip1)*xbc(ip2)
c           endif

          end do

c          if(mod(time,dt_dbg) .eq. 0.) write(56,*) 'Aqueous sources '
c find the cloud water sources for each species
          nsrca = icmapa(iy,2)
          do i=1,nsrca
           ip = 5.*nsnka + (i-1)*5
           ir1=icmapa(iy,2+ip+1)        ! index of reaction rate
           is1=icmapa(iy,2+ip+2)        ! index of stoich. coeff.
           ip1=icmapa(iy,2+ip+3)        ! index of species #1
           ip2=icmapa(iy,2+ip+4)        ! index of species #2
           ip3=icmapa(iy,2+ip+5)        ! index of species #3

           srcaa = srcaa + stc(is1) * ratea(ir1) * 
     _                  xbc(ip1) * xbc(ip2) * xbc(ip3)

c           if(mod(time,dt_dbg) .eq. 0.) then
c            write(56,'(2i3,f5.2,5e11.4)') i, ir1, stc(is1), ratea(ir1), 
c     _       xbc(ip1), xbc(ip2), xbc(ip3), 
c     _       stc(is1)*ratea(ir1)*xbc(ip1)*xbc(ip2)
c           endif
          end do


         endif      ! iy > 0

c computation of the new concentrations
c solve the equations

         if(iy .gt. 0) then
          xold = xb(ix) + xbc(iy) 
         else
          xold = xb(ix) 
         endif

     
         call soln(lww, srca, xlsa, srcaa, snkaa, ktc, heff, phrat, 
     _    xb, xbc, ctot1t, xa, xac, ctot1, xbn, xbcn, ix, dtchem)


         if(iy .gt. 0) then
          xnew = ctot1t(iy)
         else
          xnew = xbn
         endif
         if(abs(xnew-xold) .gt. xold*ccrit) then
          conv = 0
         endif

         iu = 70+ix
c         write(iu,'(a4,i4,4e15.8)') cx(ix), iter, xnew(1), xold(1),
c     _        abs(xnew(1)-xold(1)), xold(1)*ccrit

         if(iter .lt. 30) then
          xb(ix) = xbn
          if(iy .gt. 0) then
           xbc(iy) = xbcn
          endif
         else
          xtmp = xb(ix)
          xb(ix) = (xbn + xtmp) * 0.5
          if(iy .gt. 0) then
           xtmp = xbc(iy)
           xbc(iy) = (xbcn + xtmp) * 0.5
          endif
         endif

        end do                      ! end of species loop

        iter=iter+1
c        if(iter.lt.200) then
        if(iter.lt. 50) then
         goto 1002
        else
         print*,'MORE THAN  50 cldy ITERATIONS '
        endif

       endif                       ! no .more. points

      endif       ! end of whether there are cloudy points


 2000 continue

      do ix=1,nspec
       spec(ix)=xb(ix)         
      enddo
      do ix=1,naq
        aspec(ix) =  xbc(ix)         
      end do

      end do      !ncyc loop

      return
      end
c======================================================================c
      subroutine soln(lww, srcg, snkg, srca, snka, ktc, heff, phrat, 
     _  csp, aqc, ctot1t, csp1, aqc1, ctot1, xbn, xbcn, n, dt)

c This routine finds C using the Euler Backward Iterative solution technique

      implicit none
      include 'chem_params.inc'
      include 'amap.com'

c Input variables
      real lww                         ! LWC (cm3/cm3)
      real srcg, snkg
      real srca, snka
      real ktc(naq), heff(naq), phrat(naq)
      real csp(0:nspec), aqc(0:naq)
      real csp1(nspec),  aqc1(naq)
      real ctot1t(naq), ctot1(naq) 
      integer n                       ! species index
      real dt


c Output variables
      real xbn, xbcn

c Local variables
      integer ik, ic, it, iy
      real ctotal, cliq, xing 
      real snk, srcavg 
      real srcgavg, srcaavg, snkgavg, snkaavg
      real todrop, otdrop
      real snkgtmp, srcgtmp, snkatmp, srcatmp
      real cg                                !temp storage of gas conc
c-----------------------------------------------------------------------
c begin:            
      iy = gasmap(n)

c Store gas concentrations at cloudy points
      cg = csp(n)

c Determine new concentration for species that don't interact with drops
      srcgavg = srcg 
      snkgavg = snkg 
      xbn = (csp1(n) + dt*srcgavg)/(1. + snkgavg*dt)


c Determine new concentration for species that interact with drops
      if(iy .gt. 0) then

       if(iy .eq. kho2a .or. iy .eq. kch3ooa .or. iy .eq. khno3a .or.
     _    iy .eq. koha  .or. iy .eq. kn2o5a  .or. iy .eq. kno3a) then
c solve HO2, CH3OO and HNO3 and OH separately     
c  and N2O5, NO3

        if(iy.eq.1) then
         write(31,'(i4,1p,7e10.3)') iy, ktc(iy), 
     _       heff(iy), lww, aqc(iy), cg, srca, snka
        endif

        srcgavg = srcg + ktc(iy)*aqc(iy)/heff(iy) 
        srcaavg = srca + ktc(iy)*lww*cg 

        snkgavg = snkg + ktc(iy)*lww 
        snkaavg = snka + ktc(iy)/heff(iy) 

c Euler-backward method:
        xbn  = (csp1(n)  + dt*srcgavg)/(1. + snkgavg*dt)
        xbcn = (aqc1(iy) + dt*srcaavg)/(1. + snkaavg*dt)
        ctot1t(iy) = xbcn + xbn


       else
c solve other species together

       srcavg = srcg + srca 
       snkgavg = snkg 
       snkaavg = snka 
       ctotal = csp(n) + aqc(iy) 
       snk = 0.
       if(ctotal .gt. 0.)
     _  snk =  (snkgavg*csp(n) + snkaavg*aqc(iy) )/ctotal 


c Euler-backward method:
       ctot1t(iy) = (ctot1(iy) + dt*srcavg)/(1. + snk*dt)

c-----------------------------------------------------------------------
c partition ctot1t
       xing = phrat(iy)/(1.+phrat(iy))

c-----------------------------------------------------------------------
       if(cg .lt. 1.e-30 .and. aqc(iy) .lt. 1.e-30 .and. 
     _    ctot1t(iy) .gt. 1.e-10) then
        if(iy .eq. kfaa) then
           xbcn = ctot1t(iy)
           xbn = 0.
        elseif(iy .eq. kco3m) then
           xbcn = ctot1t(iy)
           xbn = 0.
        else
           write(*,*) ' produced ctot, but from where?  ', n
           write(*,'(i6,5e12.5)') iy, ctot1t(iy), cg, 
     _          aqc(iy), ctot1(iy)
           write(*,'(6x,6e12.5)') srcg, srca, snkg, snka
           if(iy .eq. ko3a) then
             xbn = ctot1t(iy)
             xbcn = 0.
           endif
        endif
        write(85,'(i10,2x,4e12.5)') 
     _     iy, ctot1t(iy), cg, aqc(iy)
        go to 15
       endif


c-----------------------------------------------------------------------
       if(xing*ctot1t(iy) .lt.  (ktc(iy)*lww) *cg*dt) then
           xbcn = xing*ctot1t(iy)
           xbn = ctot1t(iy) - xbcn
       elseif(ctot1t(iy)/(1.+phrat(iy)) .lt. ktc(iy)*aqc(iy)*dt/heff(iy) 
     _      .and. cg .eq. 0. .and. aqc(iy) .gt. 0.) then
           xbcn = xing*ctot1t(iy)
           xbn = ctot1t(iy) - xbcn
       else
           todrop = ktc(iy)*lww
           otdrop = ktc(iy)/heff(iy)
           snkgtmp = snkg + todrop 
           srcgtmp = srcg + otdrop*aqc(iy) 
           snkatmp = snka + otdrop
           srcatmp = srca + todrop*cg
 
c Euler backward:
           xbn = (csp1(n) + dt*srcgtmp)/(1. + dt*snkgtmp)
           xbcn = (aqc1(iy) + dt*srcatmp)/(1. + dt*snkatmp)
 
c scale if needed
c======================================================================c
           if(xbn + xbcn - ctot1t(iy) .gt. 0.001*ctot1t(iy)) then
             xbn  = xbn  * ctot1t(iy)/ (xbn + xbcn)
             xbcn = xbcn * ctot1t(iy)/ (xbn + xbcn)
           endif
          endif        !Kh equil. or diffusion-limited?
  15     continue
       endif           !OH,HO2,CH3O2,HNO3 or other?
      endif            !aq species?

      return
      end
c======================================================================c
      subroutine xinit(buf,iend,x)
      implicit none
      integer i,iend
      real buf(*)
      real x

      do i=1,iend
       buf(i)=x
      enddo

      return
      end
c======================================================================c
      subroutine iinit(buf,iend,ival)
      integer buf(*)
      do i=1,iend
       buf(i)=ival
      enddo
      return
      end
c======================================================================c
      SUBROUTINE OPNDAT( iucl, clfile )

      CHARACTER*20 clfile

C-----------------------------------------------------------
C     Open clfile, on error write error number and terminate.
C-----------------------------------------------------------
      nend=index(clfile,' ')
      open(unit=iucl, file=clfile(1:nend-1),status='OLD', 
     1     access='sequential',form='formatted',iostat=ios)
      if (ios .ne. 0) then
         print 1000, ios
         stop
      end if

      Return
1000  Format(1x,'error opening data file   ',i5)
      END 

